"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundary = void 0;
const constructs_1 = require("constructs");
const iam_generated_1 = require("./iam.generated");
/**
 * Modify the Permissions Boundaries of Users and Roles in a construct tree.
 *
 * @stability stable
 * @example
 *
 * const policy = ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess');
 * PermissionsBoundary.of(stack).apply(policy);
 */
class PermissionsBoundary {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Access the Permissions Boundaries of a construct tree.
     *
     * @stability stable
     */
    static of(scope) {
        return new PermissionsBoundary(scope);
    }
    /**
     * Apply the given policy as Permissions Boundary to all Roles in the scope.
     *
     * Will override any Permissions Boundaries configured previously; in case
     * a Permission Boundary is applied in multiple scopes, the Boundary applied
     * closest to the Role wins.
     *
     * @stability stable
     */
    apply(boundaryPolicy) {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (node instanceof iam_generated_1.CfnRole || node instanceof iam_generated_1.CfnUser) {
                    node.permissionsBoundary = boundaryPolicy.managedPolicyArn;
                }
            },
        });
    }
    /**
     * Remove previously applied Permissions Boundaries.
     *
     * @stability stable
     */
    clear() {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (node instanceof iam_generated_1.CfnRole || node instanceof iam_generated_1.CfnUser) {
                    node.permissionsBoundary = undefined;
                }
            },
        });
    }
}
exports.PermissionsBoundary = PermissionsBoundary;
//# sourceMappingURL=data:application/json;base64,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