import base64
import os

from typing import List
from datetime import datetime

from dotenv import load_dotenv
from langchain_community.document_loaders import PyPDFLoader
from langchain_core.documents import Document
from fastapi.responses import  JSONResponse
from langchain_text_splitters import RecursiveCharacterTextSplitter
from qdrant_client import models
from starlette.status import HTTP_500_INTERNAL_SERVER_ERROR

from src.ragapi.config import llm, rag_vector_store, feedback_vector_store
from src.ragapi.dtos import (
    AskRequestDto,
    FeedbackDto,
    FileUploadRequestDto,
    LoadFileRequestDto,
    SuggestionRequestDto
)
from langchain_core.prompts import ChatPromptTemplate

from uuid import uuid4
from pathlib import Path
from importlib import resources

load_dotenv()


class RagService():


    def __init__(self):
        self._chunk_size = os.getenv("CHUNK_SIZE")


    async def saveFeedback(self, data: FeedbackDto):
        try:
            text = data.text
            metadata = {
                "file": data.file,
                "url": data.url,
                "date": datetime.now(),
                "feedbackValue": data.feedbackValue,
                **data.metadata
            }

            document = Document(page_content=text, metadata=metadata)
            documents = [document]
            uuids = [str(uuid4())]
            feedback_vector_store.add_documents(documents=documents, ids=uuids)

            return {"content": data}
        except Exception as e:
            return JSONResponse(
                content={"content": str(e)},
                status_code=500
            )

    async def suggestAnswers(self,data: SuggestionRequestDto):
        question = data.question
        responses = feedback_vector_store.similarity_search(
            query=question,
            score_threshold=0.5,
            filter=models.Filter(
                should=[
                    models.FieldCondition(
                        key="metadata.feedbackValue",
                        match=models.MatchValue(
                            value=True
                        ),
                    ),

                ]
            )

        )

        return responses


    async def ask(self, data: AskRequestDto):
        current_prompt = ""

        if not data.use_rag:
            current_prompt = resources.read_text("src.python-ragapi.prompt_templates","without_rag_prompt.txt")
        else:
            current_prompt = resources.read_text("src.python-ragapi.prompt_templates", "use_rag_prompt.txt")

        prompt = ChatPromptTemplate.from_messages([
            ("system", current_prompt),
            "user", data.question
        ])

        chain = prompt | llm

        if not data.use_rag:
            return chain.invoke({}).content

        # Otherwise provide the context for the rag support

        similar_contexts = rag_vector_store.similarity_search(
            query = data.question,
            k=5
            #score_threshold = float(os.getenv("SCORE_TRESHOLD"))
        )

        #Put all found similar contexts into one big string
        similar_contexts_str = ""

        if len(similar_contexts) > 0:
            for context in similar_contexts:
                similar_contexts_str += f"{context.page_content} \n "


        return chain.invoke({
            'context': similar_contexts_str
        }).content












    async def load_files(self, data: List[LoadFileRequestDto]):
        loaded_files = []
        for file_to_load in data:
            file_path =  resources.files(os.getenv("LOAD_DIR")) / file_to_load.metadata["filename"]  # os.path.join(os.getenv("LOAD_DIR"), file_to_load.metadata["filename"])
            if os.path.exists(file_path):
                metadata = file_to_load.metadata
                chunk_size = file_to_load.chunk_size
                documents = await self.create_documents_from_file(file_path, metadata, chunk_size)
                uuids = [str(uuid4()) for document in documents]
                rag_vector_store.add_documents(documents=documents, ids=uuids)
                loaded_files.append(file_to_load.metadata["filename"])

        return {"loaded_files": loaded_files}


    async def upload_file(self,data: FileUploadRequestDto):
        try:
            upload_dir = Path(os.getenv("UPLOAD_DIR"))
            upload_dir.mkdir(exist_ok=True)
            content_bytes = base64.b64decode(data.file)

            file_path = resources.files(os.getenv("UPLOAD_DIR")) / data.metadata['filename']
            with open(file_path, "wb") as file:
                file.write(content_bytes)

            if os.path.exists(file_path):
                metadata = data.metadata
                chunk_size = data.chunk_size
                documents = await self.create_documents_from_file(file_path, metadata, chunk_size)
                uuids = [str(uuid4()) for document in documents]
                rag_vector_store.add_documents(documents=documents, ids=uuids)

                return {"message": data.metadata["filename"] + " has been uploaded" }

            return {"message": "Something went wrong while upload " }

        except Exception as ex:
            return JSONResponse(
                content = {"error": str(ex)},
                status_code = HTTP_500_INTERNAL_SERVER_ERROR
            )




    async def split_pages_by_chuuk(self, page, chunk_size) -> List[Document]:
         #self._chunk_size = chunk_size
         text_splitter = RecursiveCharacterTextSplitter(
             chunk_size = chunk_size,
             chunk_overlap=20,
             length_function=len,
             is_separator_regex=True
         )

         return text_splitter.create_documents([page.page_content], [page.metadata])

    async def create_documents_from_file(self, file_path, metadata, chunk_size):
        loader = PyPDFLoader(file_path)
        documents = []
        async for page in loader.alazy_load():
            splitted_pages = await self.split_pages_by_chuuk(page, chunk_size)
            for document in splitted_pages:
                for key, value in metadata.items() :
                    document.metadata[key] = value
                documents.append(document)

        return documents




