import Oasys.gRPC


# Metaclass for static properties and constants
class SectionType(type):
    _consts = {'ALE1D', 'ALE2D', 'BEAM', 'DISCRETE', 'POINT_SOURCE', 'SEATBELT', 'SHELL', 'SOLID', 'SPH', 'TSHELL'}

    def __getattr__(cls, name):
        if name in SectionType._consts:
            return Oasys.PRIMER._connection.classGetter(cls.__name__, name)

        raise AttributeError("Section class attribute '{}' does not exist".format(name))


    def __setattr__(cls, name, value):
# If one of the constants we define then error
        if name in SectionType._consts:
            raise AttributeError("Cannot set Section class constant '{}'".format(name))

# Set the property locally
        cls.__dict__[name] = value


class Section(Oasys.gRPC.OasysItem, metaclass=SectionType):
    _props = {'a', 'aafac', 'aet', 'afac', 'aisc', 'aisc_label', 'ale', 'aleform', 'area', 'baselm', 'bfac', 'ca', 'cdl', 'cfac', 'cid', 'cl', 'cmid', 'cohoff', 'cohthk', 'colour', 'cslh', 'cst', 'd1', 'd2', 'd3', 'd4', 'd5', 'd6', 'death', 'dfac', 'dofn1', 'dofn2', 'domint', 'dr', 'dro', 'ds', 'dx', 'dy', 'dz', 'ecut', 'edgset', 'efac', 'efg', 'elform', 'ellipse', 'end', 'failcr', 'fd', 'fs', 'gaskett', 'hmax', 'hmin', 'hxcslh', 'hxini', 'hycslh', 'hyini', 'hzcslh', 'hzini', 'ibr', 'icomp', 'idam', 'idila', 'idim', 'idof', 'iebt', 'iform', 'ihgf', 'iken', 'iloc', 'include', 'iner', 'interaction', 'iovpr', 'iprstr', 'ips', 'irr', 'ispline', 'iss', 'ist', 'itaj', 'itb', 'ithelfm', 'itoff', 'itorm', 'itt', 'iunf', 'iw', 'iwr', 'iyr', 'izr', 'j', 'kd', 'kernel', 'label', 'lcidt', 'lcidvel', 'lcidvolr', 'lcmdot1', 'lcmdot2', 'lcmdot3', 'lcmdot4', 'lcmdot5', 'lcmdot6', 'lcmdot7', 'lcmdot8', 'lmc', 'lprint', 'lscale', 'marea', 'misc', 'mixture', 'nhsv', 'nidlc001', 'nidlc002', 'nidlc003', 'nip', 'nipp', 'nloc', 'nsloc', 'nsm', 'ntloc', 'nxdof', 'offset', 'peri', 'points', 'pr', 'print', 'propcr', 'propt', 'ptype', 'qr', 'rampt', 'rrcon', 'sa', 'scoor', 'secid', 'setyp', 'sf', 'shrf', 'smstep', 'spg', 'sphini', 'srcon', 'start', 'stime', 'stress', 'stretch', 'stype', 'swtime', 't1', 't2', 't3', 't4', 'tdl', 'tensor', 'thermal', 'thick', 'title', 'toldef', 'transparency', 'trcon', 'ts1', 'ts2', 'tshear', 'tt1', 'tt2', 'user', 'v0', 'vol', 'xfem', 'ys', 'zs'}
    _rprops = {'exists', 'model', 'type'}


    def __del__(self):
        if not Oasys.PRIMER._connection:
            return

        if self._handle is None:
            return

        Oasys.PRIMER._connection.destructor(self.__class__.__name__, self._handle)


    def __getattr__(self, name):
# If constructor for an item fails in program, then _handle will not be set and when
# __del__ is called to return the object we will call this to get the (undefined) value
        if name == "_handle":
            return None

# If one of the properties we define then get it
        if name in Section._props:
            return Oasys.PRIMER._connection.instanceGetter(self.__class__.__name__, self._handle, name)

# If one of the read only properties we define then get it
        if name in Section._rprops:
            return Oasys.PRIMER._connection.instanceGetter(self.__class__.__name__, self._handle, name)

        raise AttributeError("Section instance attribute '{}' does not exist".format(name))


    def __setattr__(self, name, value):
# If one of the properties we define then set it
        if name in Section._props:
            Oasys.PRIMER._connection.instanceSetter(self.__class__.__name__, self._handle, name, value)
            return

# If one of the read only properties we define then error
        if name in Section._rprops:
            raise AttributeError("Cannot set read-only Section instance attribute '{}'".format(name))

# Set the property locally
        self.__dict__[name] = value


# Constructor
    def __init__(self, model, secid, type, title=Oasys.gRPC.defaultArg):
        handle = Oasys.PRIMER._connection.constructor(self.__class__.__name__, model, secid, type, title)
        Oasys.gRPC.OasysItem.__init__(self, self.__class__.__name__, handle)
        """
        Create a new Section object

        Parameters
        ----------
        model : Model
            Model that section will be created in
        secid : integer or string
            Section number or character label
        type : constant
            Section type. Can be Section.BEAM,
            Section.DISCRETE,
            Section.POINT_SOURCE,
            Section.SEATBELT,
            Section.SHELL,
            Section.SOLID,
            Section.SPH or
            Section.TSHELL
        title : string
            Optional. Title for the section

        Returns
        -------
        dict
            Section object
        """


# String representation
    def __repr__(self):
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "toString")


# Static methods
    def BlankAll(model, redraw=Oasys.gRPC.defaultArg):
        """
        Blanks all of the sections in the model

        Parameters
        ----------
        model : Model
            Model that all sections will be blanked in
        redraw : boolean
            Optional. If model should be redrawn or not.
            If omitted redraw is false. If you want to do several (un)blanks and only
            redraw after the last one then use false for all redraws apart from the last one.
            Alternatively you can redraw using View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "BlankAll", model, redraw)

    def BlankFlagged(model, flag, redraw=Oasys.gRPC.defaultArg):
        """
        Blanks all of the flagged sections in the model

        Parameters
        ----------
        model : Model
            Model that all the flagged sections will be blanked in
        flag : Flag
            Flag set on the sections that you want to blank
        redraw : boolean
            Optional. If model should be redrawn or not.
            If omitted redraw is false. If you want to do several (un)blanks and only
            redraw after the last one then use false for all redraws apart from the last one.
            Alternatively you can redraw using View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "BlankFlagged", model, flag, redraw)

    def Create(model, modal=Oasys.gRPC.defaultArg):
        """
        Starts an interactive editing panel to create a section

        Parameters
        ----------
        model : Model
            Model that the sect will be created in
        modal : boolean
            Optional. If this window is modal (blocks the user from doing anything else in PRIMER
            until this window is dismissed). If omitted the window will be modal

        Returns
        -------
        dict
            Section object (or None if not made)
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "Create", model, modal)

    def First(model):
        """
        Returns the first section in the model

        Parameters
        ----------
        model : Model
            Model to get first section in

        Returns
        -------
        Section
            Section object (or None if there are no sections in the model)
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "First", model)

    def FirstFreeLabel(model, layer=Oasys.gRPC.defaultArg):
        """
        Returns the first free section label in the model.
        Also see Section.LastFreeLabel(),
        Section.NextFreeLabel() and
        Model.FirstFreeItemLabel()

        Parameters
        ----------
        model : Model
            Model to get first free section label in
        layer : Include number
            Optional. Include file (0 for the main file) to search for labels in (Equivalent to
            First free in layer in editing panels). If omitted the whole model will be used (Equivalent to
            First free in editing panels)

        Returns
        -------
        int
            Section label
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "FirstFreeLabel", model, layer)

    def FlagAll(model, flag):
        """
        Flags all of the sections in the model with a defined flag

        Parameters
        ----------
        model : Model
            Model that all sections will be flagged in
        flag : Flag
            Flag to set on the sections

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "FlagAll", model, flag)

    def GetAll(model):
        """
        Returns a list of Section objects for all of the sections in a model in PRIMER

        Parameters
        ----------
        model : Model
            Model to get sections from

        Returns
        -------
        list
            List of Section objects
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "GetAll", model)

    def GetFlagged(model, flag):
        """
        Returns a list of Section objects for all of the flagged sections in a model in PRIMER

        Parameters
        ----------
        model : Model
            Model to get sections from
        flag : Flag
            Flag set on the sections that you want to retrieve

        Returns
        -------
        list
            List of Section objects
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "GetFlagged", model, flag)

    def GetFromID(model, number):
        """
        Returns the Section object for a section ID

        Parameters
        ----------
        model : Model
            Model to find the section in
        number : integer
            number of the section you want the Section object for

        Returns
        -------
        Section
            Section object (or None if section does not exist)
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "GetFromID", model, number)

    def Last(model):
        """
        Returns the last section in the model

        Parameters
        ----------
        model : Model
            Model to get last section in

        Returns
        -------
        Section
            Section object (or None if there are no sections in the model)
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "Last", model)

    def LastFreeLabel(model, layer=Oasys.gRPC.defaultArg):
        """
        Returns the last free section label in the model.
        Also see Section.FirstFreeLabel(),
        Section.NextFreeLabel() and
        see Model.LastFreeItemLabel()

        Parameters
        ----------
        model : Model
            Model to get last free section label in
        layer : Include number
            Optional. Include file (0 for the main file) to search for labels in (Equivalent to
            Highest free in layer in editing panels). If omitted the whole model will be used

        Returns
        -------
        int
            Section label
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "LastFreeLabel", model, layer)

    def NextFreeLabel(model, layer=Oasys.gRPC.defaultArg):
        """
        Returns the next free (highest+1) section label in the model.
        Also see Section.FirstFreeLabel(),
        Section.LastFreeLabel() and
        Model.NextFreeItemLabel()

        Parameters
        ----------
        model : Model
            Model to get next free section label in
        layer : Include number
            Optional. Include file (0 for the main file) to search for labels in (Equivalent to
            Highest+1 in layer in editing panels). If omitted the whole model will be used (Equivalent to
            Highest+1 in editing panels)

        Returns
        -------
        int
            Section label
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "NextFreeLabel", model, layer)

    def Pick(prompt, limit=Oasys.gRPC.defaultArg, modal=Oasys.gRPC.defaultArg, button_text=Oasys.gRPC.defaultArg):
        """
        Allows the user to pick a section

        Parameters
        ----------
        prompt : string
            Text to display as a prompt to the user
        limit : Model or Flag
            Optional. If the argument is a Model then only sections from that model can be picked.
            If the argument is a Flag then only sections that
            are flagged with limit can be selected.
            If omitted, or None, any sections from any model can be selected.
            from any model
        modal : boolean
            Optional. If picking is modal (blocks the user from doing anything else in PRIMER
            until this window is dismissed). If omitted the pick will be modal
        button_text : string
            Optional. By default the window with the prompt will have a button labelled 'Cancel'
            which if pressed will cancel the pick and return None. If you want to change the
            text on the button use this argument. If omitted 'Cancel' will be used

        Returns
        -------
        dict
            Section object (or None if not picked)
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "Pick", prompt, limit, modal, button_text)

    def RenumberAll(model, start):
        """
        Renumbers all of the sections in the model

        Parameters
        ----------
        model : Model
            Model that all sections will be renumbered in
        start : integer
            Start point for renumbering

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "RenumberAll", model, start)

    def RenumberFlagged(model, flag, start):
        """
        Renumbers all of the flagged sections in the model

        Parameters
        ----------
        model : Model
            Model that all the flagged sections will be renumbered in
        flag : Flag
            Flag set on the sections that you want to renumber
        start : integer
            Start point for renumbering

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "RenumberFlagged", model, flag, start)

    def Select(flag, prompt, limit=Oasys.gRPC.defaultArg, modal=Oasys.gRPC.defaultArg):
        """
        Allows the user to select sections using standard PRIMER object menus

        Parameters
        ----------
        flag : Flag
            Flag to use when selecting sections
        prompt : string
            Text to display as a prompt to the user
        limit : Model or Flag
            Optional. If the argument is a Model then only sections from that model can be selected.
            If the argument is a Flag then only sections that
            are flagged with limit can be selected (limit should be different to flag).
            If omitted, or None, any sections can be selected.
            from any model
        modal : boolean
            Optional. If selection is modal (blocks the user from doing anything else in PRIMER
            until this window is dismissed). If omitted the selection will be modal

        Returns
        -------
        int
            Number of sections selected or None if menu cancelled
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "Select", flag, prompt, limit, modal)

    def SketchFlagged(model, flag, redraw=Oasys.gRPC.defaultArg):
        """
        Sketches all of the flagged sections in the model. The sections will be sketched until you either call
        Section.Unsketch(),
        Section.UnsketchFlagged(),
        Model.UnsketchAll(),
        or delete the model

        Parameters
        ----------
        model : Model
            Model that all the flagged sections will be sketched in
        flag : Flag
            Flag set on the sections that you want to sketch
        redraw : boolean
            Optional. If model should be redrawn or not after the sections are sketched.
            If omitted redraw is true. If you want to sketch flagged sections several times and only
            redraw after the last one then use false for redraw and call
            View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "SketchFlagged", model, flag, redraw)

    def Total(model, exists=Oasys.gRPC.defaultArg):
        """
        Returns the total number of sections in the model

        Parameters
        ----------
        model : Model
            Model to get total for
        exists : boolean
            Optional. true if only existing sections should be counted. If false or omitted
            referenced but undefined sections will also be included in the total

        Returns
        -------
        int
            number of sections
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "Total", model, exists)

    def UnblankAll(model, redraw=Oasys.gRPC.defaultArg):
        """
        Unblanks all of the sections in the model

        Parameters
        ----------
        model : Model
            Model that all sections will be unblanked in
        redraw : boolean
            Optional. If model should be redrawn or not.
            If omitted redraw is false. If you want to do several (un)blanks and only
            redraw after the last one then use false for all redraws apart from the last one.
            Alternatively you can redraw using View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "UnblankAll", model, redraw)

    def UnblankFlagged(model, flag, redraw=Oasys.gRPC.defaultArg):
        """
        Unblanks all of the flagged sections in the model

        Parameters
        ----------
        model : Model
            Model that the flagged sections will be unblanked in
        flag : Flag
            Flag set on the sections that you want to unblank
        redraw : boolean
            Optional. If model should be redrawn or not.
            If omitted redraw is false. If you want to do several (un)blanks and only
            redraw after the last one then use false for all redraws apart from the last one.
            Alternatively you can redraw using View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "UnblankFlagged", model, flag, redraw)

    def UnflagAll(model, flag):
        """
        Unsets a defined flag on all of the sections in the model

        Parameters
        ----------
        model : Model
            Model that the defined flag for all sections will be unset in
        flag : Flag
            Flag to unset on the sections

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "UnflagAll", model, flag)

    def UnsketchAll(model, redraw=Oasys.gRPC.defaultArg):
        """
        Unsketches all sections

        Parameters
        ----------
        model : Model
            Model that all sections will be unblanked in
        redraw : boolean
            Optional. If model should be redrawn or not after the sections are unsketched.
            If omitted redraw is true. If you want to unsketch several things and only
            redraw after the last one then use false for redraw and call
            View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "UnsketchAll", model, redraw)

    def UnsketchFlagged(model, flag, redraw=Oasys.gRPC.defaultArg):
        """
        Unsketches all flagged sections in the model

        Parameters
        ----------
        model : Model
            Model that all sections will be unsketched in
        flag : Flag
            Flag set on the sections that you want to unsketch
        redraw : boolean
            Optional. If model should be redrawn or not after the sections are unsketched.
            If omitted redraw is true. If you want to unsketch several things and only
            redraw after the last one then use false for redraw and call
            View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.classMethod(__class__.__name__, "UnsketchFlagged", model, flag, redraw)



# Instance methods
    def AssociateComment(self, comment):
        """
        Associates a comment with a section

        Parameters
        ----------
        comment : Comment
            Comment that will be attached to the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "AssociateComment", comment)

    def Blank(self):
        """
        Blanks the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Blank")

    def Blanked(self):
        """
        Checks if the section is blanked or not

        Returns
        -------
        bool
            True if blanked, False if not
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Blanked")

    def Browse(self, modal=Oasys.gRPC.defaultArg):
        """
        Starts an edit panel in Browse mode

        Parameters
        ----------
        modal : boolean
            Optional. If this window is modal (blocks the user from doing anything else in PRIMER
            until this window is dismissed). If omitted the window will be modal

        Returns
        -------
        None
            no return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Browse", modal)

    def ClearFlag(self, flag):
        """
        Clears a flag on the section

        Parameters
        ----------
        flag : Flag
            Flag to clear on the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "ClearFlag", flag)

    def Copy(self, range=Oasys.gRPC.defaultArg):
        """
        Copies the section. The target include of the copied section can be set using Options.copy_target_include

        Parameters
        ----------
        range : boolean
            Optional. If you want to keep the copied item in the range specified for the current include. Default value is false.
            To set current include, use Include.MakeCurrentLayer()

        Returns
        -------
        Section
            Section object
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Copy", range)

    def DetachComment(self, comment):
        """
        Detaches a comment from a section

        Parameters
        ----------
        comment : Comment
            Comment that will be detached from the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "DetachComment", comment)

    def Edit(self, modal=Oasys.gRPC.defaultArg):
        """
        Starts an interactive editing panel

        Parameters
        ----------
        modal : boolean
            Optional. If this window is modal (blocks the user from doing anything else in PRIMER
            until this window is dismissed). If omitted the window will be modal

        Returns
        -------
        None
            no return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Edit", modal)

    def ExtractColour(self):
        """
        Extracts the actual colour used for section.
        By default in PRIMER many entities such as elements get their colour automatically from the part that they are in.
        PRIMER cycles through 13 default colours based on the label of the entity. In this case the section colour
        property will return the value Colour.PART instead of the actual colour.
        This method will return the actual colour which is used for drawing the section

        Returns
        -------
        int
            colour value (integer)
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "ExtractColour")

    def Flagged(self, flag):
        """
        Checks if the section is flagged or not

        Parameters
        ----------
        flag : Flag
            Flag to test on the section

        Returns
        -------
        bool
            True if flagged, False if not
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Flagged", flag)

    def GetBetaData(self, ipt):
        """
        Returns the beta angle data for an integration point in \*SECTION_SHELL or \*SECTION_TSHELL

        Parameters
        ----------
        ipt : integer
            The integration point you want the data for. Note that integration points start at 0, not 1

        Returns
        -------
        float
            float
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetBetaData", ipt)

    def GetComments(self):
        """
        Extracts the comments associated to a section

        Returns
        -------
        list
            List of Comment objects (or None if there are no comments associated to the node)
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetComments")

    def GetLmcData(self, i):
        """
        Returns the LMC property parameter for \*SECTION_SHELL or \*SECTION_SOLID

        Parameters
        ----------
        i : integer
            The point you want the parameter for. Note that points start at 0, not 1

        Returns
        -------
        float
            float
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetLmcData", i)

    def GetParameter(self, prop):
        """
        Checks if a Section property is a parameter or not.
        Note that object properties that are parameters are normally returned as the integer or
        float parameter values as that is virtually always what the user would want. For this function to
        work the JavaScript interpreter must use the parameter name instead of the value. This can be done by setting
        the Options.property_parameter_names option to true
        before calling the function and then resetting it to false afterwards..
        This behaviour can also temporarily be switched by using the Section.ViewParameters()
        method and 'method chaining' (see the examples below)

        Parameters
        ----------
        prop : string
            section property to get parameter for

        Returns
        -------
        dict
            Parameter object if property is a parameter, None if not
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetParameter", prop)

    def GetPointData(self, ipt):
        """
        Returns the point data for a single point in \*SECTION_POINT_SOURCE

        Parameters
        ----------
        ipt : integer
            The point you want the data for. Note that integration points start at 0, not 1

        Returns
        -------
        list
            A list of numbers containing the node id, vector id and orifice area
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetPointData", ipt)

    def GetUserData(self, ipt):
        """
        Returns the user defined data for an integration point in \*SECTION_SHELL and \*SECTION_SOLID

        Parameters
        ----------
        ipt : integer
            The integration point you want the data for. Note that integration points start at 0, not 1

        Returns
        -------
        list
            A list containing the data (XI, ETA, WGT for \*SECTION_SHELL, XI, ETA, ZETA, WGT for \*SECTION_SOLID)
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "GetUserData", ipt)

    def Keyword(self):
        """
        Returns the keyword for this section (\*SECT, \*SECT_SCALAR or \*SECT_SCALAR_VALUE).
        Note that a carriage return is not added.
        See also Section.KeywordCards()

        Returns
        -------
        str
            string containing the keyword
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Keyword")

    def KeywordCards(self):
        """
        Returns the keyword cards for the section.
        Note that a carriage return is not added.
        See also Section.Keyword()

        Returns
        -------
        str
            string containing the cards
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "KeywordCards")

    def Next(self):
        """
        Returns the next section in the model

        Returns
        -------
        Section
            Section object (or None if there are no more sections in the model)
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Next")

    def Previous(self):
        """
        Returns the previous section in the model

        Returns
        -------
        Section
            Section object (or None if there are no more sections in the model)
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Previous")

    def SetBetaData(self, ipt, beta):
        """
        Sets the beta angle for an integration point in \*SECTION_SHELL or \*SECTION_TSHELL

        Parameters
        ----------
        ipt : integer
            The integration point you want to set the data for.
            Note that integration points start at 0, not 1
        beta : float
            Beta angle for the integration point

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "SetBetaData", ipt, beta)

    def SetFlag(self, flag):
        """
        Sets a flag on the section

        Parameters
        ----------
        flag : Flag
            Flag to set on the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "SetFlag", flag)

    def SetLmcData(self, ipt, lmc):
        """
        Sets the lmc parameter for a point in \*SECTION_SHELL or \*SECTION_SOLID

        Parameters
        ----------
        ipt : integer
            The point you want to set the data for.
            Note that points start at 0, not 1
        lmc : float
            Lmc parameter for the point

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "SetLmcData", ipt, lmc)

    def SetPointData(self, ipt, nodeid, vecid, area):
        """
        Sets the data for a single point in \*SECTION_POINT_SOURCE

        Parameters
        ----------
        ipt : integer
            The point you want to set the data for.
            Note that integration points start at 0, not 1
        nodeid : integer
            Node ID for the point
        vecid : integer
            Vector ID for the point
        area : real
            Orifice area for the point

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "SetPointData", ipt, nodeid, vecid, area)

    def SetUserData(self, ipt, xi, eta, zeta, wgt):
        """
        Sets the user defined data for an integration point in \*SECTION_SHELL and \*SECTION_SOLID

        Parameters
        ----------
        ipt : integer
            The integration point you want to set the data for.
            Note that integration points start at 0, not 1
        xi : real
            First isoparametric coordinate
        eta : real
            Second isoparametric coordinate
        zeta : real
            Second isoparametric coordinate (SOLID) or
            Isoparametric weight (SHELL)
        wgt : real
            Isoparametric weight (SOLID)

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "SetUserData", ipt, xi, eta, zeta, wgt)

    def Sketch(self, redraw=Oasys.gRPC.defaultArg):
        """
        Sketches the section. The section will be sketched until you either call
        Section.Unsketch(),
        Section.UnsketchAll(),
        Model.UnsketchAll(),
        or delete the model

        Parameters
        ----------
        redraw : boolean
            Optional. If model should be redrawn or not after the section is sketched.
            If omitted redraw is true. If you want to sketch several sections and only
            redraw after the last one then use false for redraw and call
            View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Sketch", redraw)

    def Unblank(self):
        """
        Unblanks the section

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Unblank")

    def Unsketch(self, redraw=Oasys.gRPC.defaultArg):
        """
        Unsketches the section

        Parameters
        ----------
        redraw : boolean
            Optional. If model should be redrawn or not after the section is unsketched.
            If omitted redraw is true. If you want to unsketch several sections and only
            redraw after the last one then use false for redraw and call
            View.Redraw()

        Returns
        -------
        None
            No return value
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Unsketch", redraw)

    def ViewParameters(self):
        """
        Object properties that are parameters are normally returned as the integer or
        float parameter values as that is virtually always what the user would want. This function temporarily
        changes the behaviour so that if a property is a parameter the parameter name is returned instead.
        This can be used with 'method chaining' (see the example below) to make sure a property argument is correct

        Returns
        -------
        dict
            Section object
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "ViewParameters")

    def Xrefs(self):
        """
        Returns the cross references for this section

        Returns
        -------
        dict
            Xrefs object
        """
        return Oasys.PRIMER._connection.instanceMethod(self.__class__.__name__, self._handle, "Xrefs")

