"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const token_1 = require("./token");
/**
 * This class contains functions for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class FromCloudFormation {
    static parseValue(cfnValue) {
        return parseCfnValueToCdkValue(cfnValue);
    }
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(value, this.getString);
    }
    static getArray(value, mapper) {
        if (!Array.isArray(value)) {
            // break the type system, and just return the given value,
            // which hopefully will be reported as invalid by the validator
            // of the property we're transforming
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        return value.map(mapper);
    }
    static getMap(value, mapper) {
        if (typeof value !== 'object') {
            // if the input is not a map (= object in JS land),
            // just return it, and let the validator of this property handle it
            // (unless it's a deploy-time value,
            // which we can't map over at build time anyway)
            return value;
        }
        const ret = {};
        for (const [key, val] of Object.entries(value)) {
            ret[key] = mapper(val);
        }
        return ret;
    }
    static parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
}
exports.FromCloudFormation = FromCloudFormation;
function parseCfnValueToCdkValue(cfnValue) {
    // == null captures undefined as well
    if (cfnValue == null) {
        return undefined;
    }
    // if we have any late-bound values,
    // just return them
    if (token_1.isResolvableObject(cfnValue)) {
        return cfnValue;
    }
    if (Array.isArray(cfnValue)) {
        return cfnValue.map(el => parseCfnValueToCdkValue(el));
    }
    if (typeof cfnValue === 'object') {
        // an object can be either a CFN intrinsic, or an actual object
        const cfnIntrinsic = parseIfCfnIntrinsic(cfnValue);
        if (cfnIntrinsic) {
            return cfnIntrinsic;
        }
        const ret = {};
        for (const [key, val] of Object.entries(cfnValue)) {
            ret[key] = parseCfnValueToCdkValue(val);
        }
        return ret;
    }
    // in all other cases, just return the input
    return cfnValue;
}
function parseIfCfnIntrinsic(object) {
    var _a;
    const key = looksLikeCfnIntrinsic(object);
    switch (key) {
        case undefined:
            return undefined;
        case 'Ref': {
            // ToDo handle translating logical IDs
            return (_a = specialCaseRefs(object[key])) !== null && _a !== void 0 ? _a : cfn_fn_1.Fn._ref(object[key]);
        }
        case 'Fn::GetAtt': {
            // Fn::GetAtt takes a 2-element list as its argument
            const value = object[key];
            // ToDo same comment here as in Ref above
            return cfn_fn_1.Fn.getAtt((value[0]), value[1]);
        }
        case 'Fn::Join': {
            // Fn::Join takes a 2-element list as its argument,
            // where the first element is the delimiter,
            // and the second is the list of elements to join
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.join(value[0], value[1]);
        }
        case 'Fn::If': {
            // Fn::If takes a 3-element list as its argument
            // ToDo the first argument is the name of the condition,
            // so we will need to retrieve the actual object from the template
            // when we handle preserveLogicalIds=false
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionIf(value[0], value[1], value[2]);
        }
        case 'Fn::Equals': {
            const value = parseCfnValueToCdkValue(object[key]);
            return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
        }
        default:
            throw new Error(`Unsupported CloudFormation function '${key}'`);
    }
}
function looksLikeCfnIntrinsic(object) {
    const objectKeys = Object.keys(object);
    // a CFN intrinsic is always an object with a single key
    if (objectKeys.length !== 1) {
        return undefined;
    }
    const key = objectKeys[0];
    return key === 'Ref' || key.startsWith('Fn::') ? key : undefined;
}
function specialCaseRefs(value) {
    switch (value) {
        case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
        case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
        case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
        case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
        case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
        case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
        case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
        case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
        default: return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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