"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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