"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = nodeunit.testCase({
    'negative amount'(test) {
        test.throws(() => lib_1.Duration.seconds(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazyDuration = lib_1.Duration.seconds(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazyDuration.toSeconds()), 1337);
        test.throws(() => stack.resolve(lazyDuration.toMinutes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Duration in seconds'(test) {
        const duration = lib_1.Duration.seconds(300);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'300 seconds' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.seconds(60 * 60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in minutes'(test) {
        const duration = lib_1.Duration.minutes(5);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'5 minutes' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.minutes(60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in hours'(test) {
        const duration = lib_1.Duration.hours(5);
        test.equal(duration.toSeconds(), 18000);
        test.equal(duration.toMinutes(), 300);
        test.throws(() => duration.toDays(), /'5 hours' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 5 / 24);
        test.equal(lib_1.Duration.hours(24).toDays(), 1);
        test.done();
    },
    'seconds to milliseconds'(test) {
        const duration = lib_1.Duration.seconds(5);
        test.equal(duration.toMilliseconds(), 5000);
        test.done();
    },
    'Duration in days'(test) {
        const duration = lib_1.Duration.days(1);
        test.equal(duration.toSeconds(), 86400);
        test.equal(duration.toMinutes(), 1440);
        test.equal(duration.toDays(), 1);
        test.done();
    },
    'toISOString'(test) {
        test.equal(lib_1.Duration.seconds(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(5).toISOString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toISOString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toISOString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toISOString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toISOString(), 'PT1D1H1M1S');
        test.done();
    },
    'toIsoString'(test) {
        test.equal(lib_1.Duration.seconds(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toIsoString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(5).toIsoString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toIsoString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toIsoString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toIsoString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toIsoString(), 'PT1D1H1M1S');
        test.done();
    },
    'parse'(test) {
        test.equal(lib_1.Duration.parse('PT0S').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0M').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0H').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0D').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT5S').toSeconds(), 5);
        test.equal(lib_1.Duration.parse('PT5M').toSeconds(), 300);
        test.equal(lib_1.Duration.parse('PT5H').toSeconds(), 18000);
        test.equal(lib_1.Duration.parse('PT5D').toSeconds(), 432000);
        test.equal(lib_1.Duration.parse('PT1D1H1M1S').toSeconds(), 1 + 60 * (1 + 60 * (1 + 24)));
        test.done();
    },
    'to human string'(test) {
        test.equal(lib_1.Duration.minutes(0).toHumanString(), '0 minutes');
        test.equal(lib_1.Duration.minutes(lib_1.Lazy.numberValue({ produce: () => 5 })).toHumanString(), '<token> minutes');
        test.equal(lib_1.Duration.minutes(10).toHumanString(), '10 minutes');
        test.equal(lib_1.Duration.minutes(1).toHumanString(), '1 minute');
        test.equal(lib_1.Duration.minutes(62).toHumanString(), '1 hour 2 minutes');
        test.equal(lib_1.Duration.seconds(3666).toHumanString(), '1 hour 1 minute');
        test.equal(lib_1.Duration.millis(3000).toHumanString(), '3 seconds');
        test.equal(lib_1.Duration.millis(3666).toHumanString(), '3 seconds 666 millis');
        test.equal(lib_1.Duration.millis(3.6).toHumanString(), '3.6 millis');
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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