import { Reference } from "../reference";
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
export declare class CfnReference extends Reference {
    private readonly displayName;
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x: IResolvable): x is CfnReference;
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new CfnReference(...) })
     */
    static for(target: CfnElement, attribute: string): CfnReference;
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName: string, scope: Construct): CfnReference;
    /**
     * Static table where we keep singleton CfnReference instances
     */
    private static referenceTable;
    /**
     * Get or create the table
     */
    private static singletonReference;
    /**
     * What stack this Token is pointing to
     */
    private readonly producingStack?;
    /**
     * The Tokens that should be returned for each consuming stack (as decided by the producing Stack)
     */
    private readonly replacementTokens;
    private readonly originalDisplayName;
    private readonly humanReadableDesc;
    protected constructor(value: any, displayName: string, target: IConstruct);
    resolve(context: IResolveContext): any;
    /**
     * Register a stack this references is being consumed from.
     */
    consumeFromStack(consumingStack: Stack, consumingConstruct: IConstruct): void;
    /**
     * Implementation of toString() that will use the display name
     */
    toString(): string;
    /**
     * Export a Token value for use in another stack
     *
     * Works by mutating the producing stack in-place.
     */
    private exportValue;
    private generateExportName;
}
import { CfnElement } from "../cfn-element";
import { Construct, IConstruct } from "../construct";
import { IResolvable, IResolveContext } from "../resolvable";
import { Stack } from "../stack";
