"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
//# sourceMappingURL=data:application/json;base64,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