from abc import ABC, abstractmethod
from typing import List, Optional, Dict, Any

class VectorStore(ABC):
    """Abstract base class for vector store implementations."""
    
    @abstractmethod
    def add(self, text: str, embedding: List[float]) -> None:
        """
        Add a single text and its embedding to the vector store.
        
        Args:
            text (str): The text to store
            embedding (List[float]): The embedding vector for the text
        """
        pass
    
    @abstractmethod
    def query(self, query_embedding: List[float], k: int = 10) -> List[str]:
        """
        Query the vector store for similar texts.
        
        Args:
            query_embedding (List[float]): The query embedding vector
            k (int): Number of results to return
            
        Returns:
            List[str]: List of similar texts
        """
        pass

def QdrantVectorStore(*args, **kwargs):
    """
    Factory function for QdrantVectorStore.
    
    Returns:
        QdrantVectorStore: An instance of QdrantVectorStore
    """
    try:
        from evoagent.vector_stores.qdrant_vectorstore import QdrantVectorStore as _QdrantVectorStore
        return _QdrantVectorStore(*args, **kwargs)
    except ImportError:
        raise ImportError(
            "Could not import qdrant-client. Please install it with: "
            "pip install qdrant-client"
        )

def FaissVectorStore(embedding_dim: int, **kwargs):
    """
    Factory function for FaissVectorStore.
    
    Args:
        embedding_dim (int): Dimension of the embedding vectors
        **kwargs: Additional configuration options
        
    Returns:
        FaissVectorStore: An instance of FaissVectorStore
    """
    try:
        from evoagent.vector_stores.faiss_vectorstore import FaissVectorStore as _FaissVectorStore
        return _FaissVectorStore(embedding_dim=embedding_dim, **kwargs)
    except ImportError:
        raise ImportError(
            "Could not import faiss-cpu. Please install it with: "
            "pip install faiss-cpu"
        )

__all__ = ['VectorStore', 'QdrantVectorStore', 'FaissVectorStore']