"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const node_version_1 = require("./node-version");
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the @aws-cdk npm scope
    for (const name of Object.keys(libraries)) {
        if (!name.startsWith('@aws-cdk/')) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    const paths = mod.paths.map(stripNodeModules);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
    /**
     * @param s a path.
     * @returns ``s`` with any terminating ``/node_modules``
     *      (or ``\\node_modules``) stripped off.)
     */
    function stripNodeModules(s) {
        if (s.endsWith('/node_modules') || s.endsWith('\\node_modules')) {
            // /node_modules is 13 characters
            return s.substr(0, s.length - 13);
        }
        return s;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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