"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App({
        runtimeInfo: false,
        stackTraces: false,
        ...props,
    });
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStackByName(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.stackName, 'stack1');
        test.deepEqual(stack1.id, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.stackName, 'stack2');
        test.deepEqual(stack2.id, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4'
            }
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: 'DummyResource',
                    Properties: {
                        Prog2: 'Prog2'
                    }
                },
                s1c2r1D1791C01: {
                    Type: 'ResourceType1'
                },
                s1c2r25F685FFF: {
                    Type: 'ResourceType2',
                    Properties: {
                        FromContext: 'HELLO'
                    }
                }
            }
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar'
            }
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: 'fake',
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: 'fake2',
                    props: {
                        foo: 'bar',
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: 'missing-context-key',
                provider: 'fake',
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: 'missing-context-key-2',
                provider: 'fake2',
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                    foo: 'bar',
                },
            }
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], 'Java/1.2.3.4');
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            '@aws-cdk/cloud-assembly-schema': version,
            'jsii-runtime': `node.js/${process.version}`
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.stackName, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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