"use strict";
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk = require("../lib");
function createModernApp() {
    return new cdk.App({
        context: {
            [cxapi.DISABLE_VERSION_REPORTING]: 'true',
        }
    });
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.synth();
        // THEN
        test.same(app.synth(), session); // same session if we synth() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json', 'tree.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {
            Tree: {
                type: 'cdk:tree',
                properties: { file: 'tree.json' }
            }
        });
        test.deepEqual(readJson(session.directory, 'tree.json'), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: { id: 'Tree', path: 'Tree' }
                }
            }
        });
        test.done();
    },
    'synthesis respects disabling tree metadata'(test) {
        const app = new cdk.App({
            treeMetadata: false,
        });
        const assembly = app.synth();
        test.deepEqual(list(assembly.directory), ['cdk.out', 'manifest.json']);
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.assembly.outdir, 'foo.json', { bar: 123 });
                s.assembly.addArtifact('my-random-construct', {
                    type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json'
                    }
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.ok(list(session.directory).includes('foo.json'));
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: cxschema.Manifest.version(),
            artifacts: {
                'Tree': {
                    type: 'cdk:tree',
                    properties: { file: 'tree.json' }
                },
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' }
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                }
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends cdk.Construct {
            constructor() {
                super(undefined, 'id');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.assembly.addArtifact('art', {
                    type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    properties: {
                        templateFile: 'hey.json',
                        parameters: {
                            paramId: 'paramValue',
                            paramId2: 'paramValue2'
                        }
                    },
                    environment: 'aws://unknown-account/us-east-1'
                });
                writeJson(session.assembly.outdir, 'hey.json', { hello: 123 });
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
        }
        const root = new SynthesizeMe();
        const assembly = cdk.ConstructNode.synth(root.node, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStackByName('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.templateFile, 'hey.json');
        test.deepEqual(stack.parameters, { paramId: 'paramValue', paramId2: 'paramValue2' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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