# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/01_settings.ipynb (unless otherwise specified).

__all__ = ['change_log_config', 'run_shell_installs', 'action_over_list_f', 'replacement_f', 'updates_faq_config_file',
           'select_faq_responses', 'select_squad_responses', 'load_qa_models', 'get_responses', 'format_responses',
           'get_input', 'question_response', 'new_question_answer', 'new_context', 'set_minimal_faq_questions',
           'set_minimal_contexts', 'set_data_dict', 'load_and_prepare_data']

# Cell
from deeppavlov.core.common.paths import get_settings_path
from deeppavlov import configs, build_model, train_model
import json
from os import path, popen, mkdir
from shutil import copyfile
import pandas as pd
import numpy as np

from pathlib import Path
import logging
from collections import defaultdict



# Cell
def change_log_config():
    '''Change Deeppavlov configuration files to ERROR mode
    '''
    settings_file = path.join(get_settings_path(), 'log_config.json')
    #logs_key = 'disable_existing_loggers'

    settings_json = json.load(open(settings_file))
    settings_json['handlers']['file']['level'] = 'ERROR'
    settings_json['handlers']['stderr']['level'] = 'ERROR'
    settings_json['handlers']['stdout']['level'] = 'ERROR'
    settings_json['handlers']['uvicorn_handler']['level'] = 'ERROR'

    settings_json['loggers']['deeppavlov']['level'] = 'ERROR'
    settings_json['loggers']['deeppavlov']['propagate'] = True

    settings_json['loggers']['uvicorn.access']['level'] = 'ERROR'
    settings_json['loggers']['uvicorn.access']['propagate'] = True

    settings_json['loggers']['uvicorn.error']['level'] = 'ERROR'
    settings_json['loggers']['uvicorn.error']['propagate'] = True


    json.dump(settings_json, open(settings_file, 'w'))


def run_shell_installs():
    ''' Run install commands
    '''
    logging.info(f'..Installing NLP libraries')
    change_log_config()

    command_strings = (
        ' pip install deeppavlov', ' python -m deeppavlov install squad',
        ' python -m deeppavlov install squad_bert',
        ' python -m deeppavlov install fasttext_avg_autofaq',
        ' python -m deeppavlov install fasttext_tfidf_autofaq',
        ' python -m deeppavlov install tfidf_autofaq',
        ' python -m deeppavlov install tfidf_logreg_autofaq ',
        ' python -m deeppavlov install tfidf_logreg_en_faq'
    )
    for command in command_strings:

        logging.debug(command)
        logging.debug(popen(command).read())

# Cell
def action_over_list_f(arr, v):
    ''' v[0] and v[1] are dictionaries
        arr is array of dictionaries
    '''

    k_id, v_id = next(iter(v[0].items()))

    for p, a in enumerate(arr):
        if k_id in a.keys() and a[k_id] == v_id:
            for k_rep, v_rep in v[1].items():
                arr[p][k_rep] = v_rep


def replacement_f(model_config, **args):
    '''Replaces the model config dictionary with new values
    provided in **args
    '''
    for k, v in args.items():
        if isinstance(v, dict):
            replacement_f(model_config[k], **v)
        if isinstance(v, str):
            model_config[k] = v
        if isinstance(model_config[k], list):
            action_over_list_f(model_config[k], v)

# Cell
def updates_faq_config_file(
    configs_path,
    **args
):
    '''Updates deepplavov json config file
    '''
    #set FAQ data in config file
    model_config = json.load(open(configs_path))

    if 'data_url' in model_config['dataset_reader']:
        del model_config['dataset_reader']['data_url']

    replacement_f(model_config=model_config,**args)

    json.dump(model_config, open(configs_path, 'w'))

# Cell
def select_faq_responses(faq_model, question):
    '''Calls Deeppavlov FAQ model
    '''
    return faq_model([question])[0]

# Cell
def select_squad_responses(
    contexts, squad_model, question, best_results=1
):
    '''Calls Deeppavlov BERT and RNET Context Question Answering
    '''
    responses = contexts.context.apply(
        lambda context: squad_model([context], [question])
    ).values

    logging.debug(f'Responses: {responses}')
    top_responses = [
        r[0][0] for r in sorted(responses, key=lambda x: -1 * x[2][0])
        [:best_results]
    ]

    logging.debug(f'Top Responses: {top_responses}')
    return responses, top_responses

# Cell
def load_qa_models(
    config_rnet=configs.squad.squad,
    config_bert=configs.squad.squad_bert,
    config_tfidf=configs.faq.tfidf_logreg_en_faq,
    download=True
):
    ''' Load the squad and faq models
    '''
    qa_models = {
        'squad':
            {
                'rnet': build_model(config_rnet, download=download),
                'bert': build_model(config_bert, download=download)
            },
        'faq': {
            'tfidf': train_model(config_tfidf, download=download)
        }
    }
    return qa_models

def get_responses(contexts, question, qa_models, nb_squad_results=1):
    ''' Get response from a question using qa_models and contexts
    '''
    responses = {'squad': defaultdict(list), 'faq': defaultdict(list)}
    for squad_name, squad_model in qa_models['squad'].items():
        responses['squad'][squad_name] = select_squad_responses(
            contexts, squad_model, question, best_results=nb_squad_results
        )[1]
    for faq_name, faq_model in qa_models['faq'].items():
        responses['faq'][faq_name] = select_faq_responses(faq_model, question)
    return question, responses

    #format_responses(
    #    question, set([r for r in responses if r.strip()])
    #)

# Cell


def format_responses(dict_responses):
    '''Format question-response pair
    '''
    flatten_responses = [
        res for model_responses in dict_responses.values()
        for res in model_responses.values()
    ]
    flatten_responses = [r for res in flatten_responses for r in res]
    logging.debug(flatten_responses)

    formatted_response = f'\n Answers:\n\n'
    for k, res in enumerate(set(flatten_responses)):
        formatted_response += f'{k+1}: {res}\n'

    logging.debug(formatted_response)
    return flatten_responses, formatted_response

# Cell
def get_input(text):
    '''This redundancy is needed for testing'''
    return input(text)


def question_response(data, qa_models, num_returned_values_per_squad_model=1):
    ''' Receive response and call get_response()
    '''
    question = get_input('Introduce question:\n')

    _, responses = get_responses(
        data['context']['df'], question, qa_models, nb_squad_results=1
    )
    _,formatted_responses = format_responses(responses)

    return question, formatted_responses

# Cell
def new_question_answer(data, qa_models):
    ''' Asks for a new question-answer pair; store the result in the
        faq dataframe and retrain the faq-model
    '''

    question = get_input('Introduce question:\n')

    new_faq = pd.DataFrame(
        {
            'Question': [question],
            'Answer': [get_input('Introduce the answer:\n')]
        }
    )
    data['faq']['df'] = data['faq']['df'].append(new_faq)
    data['faq']['df'].to_csv(data['faq']['path'], index=False)
    qa_models['faq']['tfidf'] = train_model(
        data['faq']['config'], download=False
    )
    logging.info('FAQ dataset and model updated..')

# Cell
def new_context(data):
    ''' Stores the new context in the context dataframe
    '''

    new_context = pd.DataFrame(
        {
            'topic': [get_input('Give context a title:\n')],
            'context': [get_input('Introduce the context:\n')]
        }
    )
    data['context']['df'] = data['context']['df'].append(new_context)
    data['context']['df'].to_csv(data['context']['path'], index=False)

    logging.info('contexts dataset updated..')



# Cell
def set_minimal_faq_questions(data):
    ''' Sets the faq configurations that assure a proper operation
    '''
    if data['df'].shape[0] > 1:
        return
    minimal_questions = [
        'Is this the Intekglobal Dialog System?',
        'What is the purpose of these two automated questions?'
    ]
    minimal_answers = [
        'This is the default reponse of the Dialog System, ' +
        'please populate your dataset with better reponses',
        'The purpose of this automated answer is to initalize the FAQ system, if you are ' +
        'seeing this, you probably need to feed your FAQ dataset'
    ]
    minimal_faqs_df = pd.DataFrame(
        {
            'Question': minimal_questions,
            'Answer': minimal_answers
        }
    )
    data['df'] = pd.concat([data['df'], minimal_faqs_df])
    data['df'].to_csv(data['path'], index=False)
    logging.info(f' File created at {data["path"]}')


def set_minimal_contexts(data):
    ''' Sets the context configurations that assure a proper operation
    '''
    if data['df'].shape[0] > 0:
        return

    minimal_context_df = pd.DataFrame({'topic': [], 'context': []})
    data['df'] = minimal_context_df
    data['df'].to_csv(data['path'], index=False)

    logging.info(f' File created at {data["path"]}')


def set_data_dict(file, data, question_type, data_dir):
    '''Creates unexistent files
    '''

    data['path'] = file if file is not None else path.join(
        data_dir, question_type + '_data.csv'
    )

    data['df'] = pd.read_csv(data['path']) if path.isfile(data['path']
                                                         ) else pd.DataFrame()

    if question_type == 'faq':
        set_minimal_faq_questions(data)
    if question_type == 'context':
        set_minimal_contexts(data)


def load_and_prepare_data(context_data_file, faq_data_file, data, configs_faq):
    '''Calls the context and faq configuration routines
    '''

    PARENT_DIR = popen('$PWD').read().strip()

    if faq_data_file or context_data_file is None:
        DATA_DIR = path.join(PARENT_DIR, 'data')

        if not path.isdir(DATA_DIR):
            mkdir(DATA_DIR)
            logging.info(f'Data directory created at {DATA_DIR}')

    if configs_faq is None:
        configs_faq = configs.faq.tfidf_logreg_en_faq

    data['faq']['config'] = configs_faq

    set_data_dict(
        file=faq_data_file,
        data=data['faq'],
        question_type='faq',
        data_dir=DATA_DIR
    )
    set_data_dict(
        file=context_data_file,
        data=data['context'],
        question_type='context',
        data_dir=DATA_DIR
    )

    updates_faq_config_file(
        configs_path=configs_faq,
        dataset_reader={'data_path': data['faq']['path']}
    )