"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateFlinkApplicationProps = void 0;
const core = require("aws-cdk-lib/core");
/**
 * Early validation for the props used to create FlinkApplications.
 */
function validateFlinkApplicationProps(props) {
    validateApplicationName(props.applicationName);
    validateParallelism(props.parallelism);
    validateParallelismPerKpu(props.parallelismPerKpu);
    validateVpcProps(props);
}
exports.validateFlinkApplicationProps = validateFlinkApplicationProps;
function validateApplicationName(applicationName) {
    if (applicationName === undefined || core.Token.isUnresolved(applicationName)) {
        return;
    }
    if (applicationName.length === 0) {
        throw new Error('applicationName cannot be empty. It must contain at least 1 character.');
    }
    if (!/^[a-zA-Z0-9_.-]+$/.test(applicationName)) {
        throw new Error(`applicationName may only contain letters, numbers, underscores, hyphens, and periods. Name: ${applicationName}`);
    }
    if (applicationName.length > 128) {
        throw new Error(`applicationName max length is 128. Name: ${applicationName} is ${applicationName.length} characters.`);
    }
}
function validateParallelism(parallelism) {
    if (parallelism === undefined || core.Token.isUnresolved(parallelism)) {
        return;
    }
    if (parallelism < 1) {
        throw new Error('parallelism must be at least 1');
    }
}
function validateParallelismPerKpu(parallelismPerKpu) {
    if (parallelismPerKpu === undefined || core.Token.isUnresolved(parallelismPerKpu)) {
        return;
    }
    if (parallelismPerKpu < 1) {
        throw new Error('parallelismPerKpu must be at least 1');
    }
}
function validateVpcProps({ vpc, securityGroups = [], vpcSubnets }) {
    if (!vpc) {
        if (vpcSubnets) {
            throw new Error('vpc prop required when passing vpcSubnets');
        }
        if (securityGroups.length > 0) {
            throw new Error('vpc prop required when passing securityGroups');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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