from __future__ import absolute_import

import numpy as np
from ..regression import OLSModel
import nipy.testing as niptest
import scipy.stats

from .exampledata import x, y

Rscript = '''
d = read.table('data.csv', header=T, sep=' ')

y.lm = lm(Y ~ X1 + X2 + X3 + X4 + X5 + X6 + X7 + X8 + X9 + X10 + X11 + X12 + X13 + X14, data=d)
print(summary(y.lm))

y.lm2 = lm(Y ~ X1 + X2 + X3 + X4 + X5 + X6 + X7 + X8 + X9 + X10 + X11 + X12 + X13 + X14 - 1, data=d)
print(summary(y.lm2))

SSE = sum(resid(y.lm)^2)
SST = sum((d$Y - mean(d$Y))^2)
SSR = SST - SSE

print(data.frame(SSE, SST, SSR))

MSE = SSE / y.lm$df.resid
MST = SST / (length(d$Y) - 1)
MSR = SSR / (length(d$Y) - y.lm$df.resid - 1)

print(data.frame(MSE, MST, MSR))
print(AIC(y.lm))
print(AIC(y.lm2))
'''

# lines about "Signif. codes" were deleted due to a character encoding issue

Rresults = \
"""
These are the results from fitting the model in R, i.e. running the commands Rscript in R
A few things to note, X8 is a column of 1s,
so by not including a '-1' in the formula, X8 gets
thrown out of the model, with its coefficients
being the "(Intercept)" term. An alternative is to use "-1"
in the formula, but then R gives nonsensical F, R2 and adjusted R2
values. This means that R2, R2a and F cannot fully be trusted in R.

In OLSModel, we have checked whether a column of 1s is in the column
space, in which case the F, R2, and R2a are seneible.

> source('test.R')
[1] "Without using '-1'"
[1] "------------------"

Call:
lm(formula = Y ~ X1 + X2 + X3 + X4 + X5 + X6 + X7 + X8 + X9 + 
    X10 + X11 + X12 + X13 + X14, data = d)

Residuals:
      Min        1Q    Median        3Q       Max 
-2.125783 -0.567850  0.004305  0.532145  2.372263 

Coefficients: (1 not defined because of singularities)
              Estimate Std. Error t value Pr(>|t|)    
(Intercept)  2.603e+02  8.226e-01 316.463  < 2e-16 ***
X1           1.439e-02  2.649e-02   0.543   0.5881    
X2          -6.975e+00  1.022e+01  -0.683   0.4963    
X3           4.410e+01  5.740e+00   7.682 6.42e-12 ***
X4           3.864e+00  5.770e+00   0.670   0.5044    
X5           2.458e+02  4.594e+02   0.535   0.5937    
X6           9.789e+02  3.851e+02   2.542   0.0124 *  
X7           1.339e+03  8.418e+02   1.591   0.1145    
X8                  NA         NA      NA       NA    
X9          -1.955e-02  1.539e-02  -1.270   0.2066    
X10          7.042e-05  2.173e-04   0.324   0.7465    
X11         -3.743e-08  6.770e-07  -0.055   0.9560    
X12          3.060e-06  2.094e-06   1.461   0.1469    
X13          1.440e-06  1.992e-06   0.723   0.4711    
X14         -1.044e-05  7.215e-06  -1.448   0.1505    
---

Residual standard error: 0.8019 on 112 degrees of freedom
Multiple R-squared: 0.5737,Adjusted R-squared: 0.5242 
F-statistic: 11.59 on 13 and 112 DF,  p-value: 1.818e-15 

[1] "Using '-1'"
[1] "------------------"

Call:
lm(formula = Y ~ X1 + X2 + X3 + X4 + X5 + X6 + X7 + X8 + X9 + 
    X10 + X11 + X12 + X13 + X14 - 1, data = d)

Residuals:
      Min        1Q    Median        3Q       Max 
-2.125783 -0.567850  0.004305  0.532145  2.372263 

Coefficients:
      Estimate Std. Error t value Pr(>|t|)    
X1   1.439e-02  2.649e-02   0.543   0.5881    
X2  -6.975e+00  1.022e+01  -0.683   0.4963    
X3   4.410e+01  5.740e+00   7.682 6.42e-12 ***
X4   3.864e+00  5.770e+00   0.670   0.5044    
X5   2.458e+02  4.594e+02   0.535   0.5937    
X6   9.789e+02  3.851e+02   2.542   0.0124 *  
X7   1.339e+03  8.418e+02   1.591   0.1145    
X8   2.603e+02  8.226e-01 316.463  < 2e-16 ***
X9  -1.955e-02  1.539e-02  -1.270   0.2066    
X10  7.042e-05  2.173e-04   0.324   0.7465    
X11 -3.743e-08  6.770e-07  -0.055   0.9560    
X12  3.060e-06  2.094e-06   1.461   0.1469    
X13  1.440e-06  1.992e-06   0.723   0.4711    
X14 -1.044e-05  7.215e-06  -1.448   0.1505    
---

Residual standard error: 0.8019 on 112 degrees of freedom
Multiple R-squared:     1,Adjusted R-squared:     1 
F-statistic: 9.399e+05 on 14 and 112 DF,  p-value: < 2.2e-16 

       SSE      SST      SSR
1 72.02328 168.9401 96.91685
       MSE      MST      MSR
1 0.643065 1.351521 7.455142
[1] "AIC"
[1] 317.1017
[1] "BIC"
[1] 359.6459


"""

def test_results():
    m = OLSModel(x)
    r = m.fit(y)
    # results hand compared with R's printout

    yield niptest.assert_equal, '%0.4f' % r.R2, '0.5737'
    yield niptest.assert_equal, '%0.4f' % r.R2_adj, '0.5242'

    f = r.F_overall
    yield niptest.assert_equal, '%0.2f' % f['F'], '11.59'
    yield niptest.assert_equal, f['df_num'], 13
    yield niptest.assert_equal, f['df_den'], 112
    yield niptest.assert_equal, '%0.3e' % f['p_value'], '1.818e-15'

    # test Fcontrast, the 8th column of m.design is all 1s
    # let's construct a contrast matrix that tests everything
    # but column 8 is zero


    M = np.identity(14)
    M = np.array([M[i] for i in [0,1,2,3,4,5,6,8,9,10,11,12,13]])
    Fc = r.Fcontrast(M)
    yield niptest.assert_array_almost_equal, [Fc.F], [f['F']], 6
    yield niptest.assert_array_almost_equal, [Fc.df_num], [f['df_num']], 6
    yield niptest.assert_array_almost_equal, [Fc.df_den], [f['df_den']], 6

    thetas = []
    sds = []
    ts = []
    ps = []

    # the model has an intercept

    yield niptest.assert_true, r.model.has_intercept

    # design matrix has full rank

    yield niptest.assert_equal, r.model.rank, 14

    # design matrix has full rank

    yield niptest.assert_equal, r.df_model, 14
    yield niptest.assert_equal, r.df_total, 126
    yield niptest.assert_equal, r.df_resid, 112

    # entries with '*****' are not tested as they were a different format

    resultstr = \
'''
X1   1.439e-02  2.649e-02   0.543   0.5881    
X2  -6.975e+00  1.022e+01  -0.683   0.4963    
X3   4.410e+01  5.740e+00   7.682   ******
X4   3.864e+00  5.770e+00   0.670   0.5044    
X5   2.458e+02  4.594e+02   0.535   0.5937    
X6   9.789e+02  3.851e+02   2.542   0.0124 
X7   1.339e+03  8.418e+02   1.591   0.1145    
X8   2.603e+02  8.226e-01 316.463   ******
X9  -1.955e-02  1.539e-02  -1.270   0.2066    
X10  7.042e-05  2.173e-04   0.324   0.7465    
X11 -3.743e-08  6.770e-07  -0.055   0.9560    
X12  3.060e-06  2.094e-06   1.461   0.1469    
X13  1.440e-06  1.992e-06   0.723   0.4711    
X14 -1.044e-05  7.215e-06  -1.448   0.1505    
X1   1.439e-02  2.649e-02   0.543   0.5881    
X2  -6.975e+00  1.022e+01  -0.683   0.4963    
X3   4.410e+01  5.740e+00   7.682   ****** 
X4   3.864e+00  5.770e+00   0.670   0.5044    
X5   2.458e+02  4.594e+02   0.535   0.5937    
X6   9.789e+02  3.851e+02   2.542   0.0124   
X7   1.339e+03  8.418e+02   1.591   0.1145    
X8   2.603e+02  8.226e-01 316.463   ******
X9  -1.955e-02  1.539e-02  -1.270   0.2066    
X10  7.042e-05  2.173e-04   0.324   0.7465    
X11 -3.743e-08  6.770e-07  -0.055   0.9560    
X12  3.060e-06  2.094e-06   1.461   0.1469    
X13  1.440e-06  1.992e-06   0.723   0.4711    
X14 -1.044e-05  7.215e-06  -1.448   0.1505    
'''

    for row in resultstr.strip().split('\n'):
        row = row.strip()
        _, th, sd, t, p = row.split()
        thetas.append(th)
        sds.append(sd)
        ts.append(t)
        ps.append(p)

    for th, thstr in zip(r.theta, thetas):
        yield niptest.assert_equal, '%0.3e' % th, thstr

    for sd, sdstr in zip([np.sqrt(r.vcov(column=i)) for i in range(14)], sds):
        yield niptest.assert_equal, '%0.3e' % sd, sdstr

    for t, tstr in zip([r.t(column=i) for i in range(14)], ts):
        yield niptest.assert_equal, '%0.3f' % t, tstr

    for i, t in enumerate([r.t(column=i) for i in range(14)]):
        m = np.zeros((14,))
        m[i] = 1.
        tv = r.Tcontrast(m)
        e = r.theta[i]
        sd = np.sqrt(r.vcov(column=i))
        yield niptest.assert_almost_equal, tv.t, t, 6
        yield niptest.assert_almost_equal, tv.sd, sd, 6
        yield niptest.assert_almost_equal, tv.effect, e, 6


    for p, pstr in zip([2*scipy.stats.t.sf(np.fabs(r.t(column=i)), r.df_resid) for i in range(14)], ps):
        if pstr.find('*') < 0:
            yield niptest.assert_equal, '%0.4f' % p, pstr

    yield niptest.assert_equal, "%0.5f" % r.SSE, "72.02328"
    yield niptest.assert_equal, "%0.4f" % r.SST, "168.9401"
    yield niptest.assert_equal, "%0.5f" % r.SSR, "96.91685"

    yield niptest.assert_equal, "%0.6f" % r.MSE, "0.643065"
    yield niptest.assert_equal, "%0.6f" % r.MST, "1.351521"
    yield niptest.assert_equal, "%0.6f" % r.MSR, "7.455142"

    yield niptest.assert_equal, "%0.4f" % np.sqrt(r.MSE), "0.8019"
    
    # the difference here comes from the fact that
    # we've treated sigma as a nuisance parameter,
    # so our AIC is the AIC of the profiled log-likelihood...

    yield niptest.assert_equal, '%0.4f'% (r.AIC + 2,), '317.1017'
    yield niptest.assert_equal, '%0.4f'% (r.BIC + np.log(126),), '359.6459'


# this is the file "data.csv" referred to in Rscript above

Rdata = '''
Y X1 X2 X3 X4 X5 X6 X7 X8 X9 X10 X11 X12 X13 X14
2.558020266818153345e+02 -4.423009200784273898e-02 -6.615177603161188392e-03 -2.429792163411158279e-02 4.236447886547620167e-02 1.618533936246031348e-03 -8.683269025079367589e-04 -8.181821468255191711e-04 1.000000000000000000e+00 0.000000000000000000e+00 0.000000000000000000e+00 0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.573856564029982792e+02 -1.247753847628743987e-02 8.132393396825286086e-03 -4.413603363412710312e-02 3.174380286547619917e-02 1.507591026246031356e-03 -8.321096135079367661e-04 -5.268108768253958792e-04 1.000000000000000000e+00 2.027260000000000062e+00 4.109783107600000207e+00 8.331598902713176713e+00 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.590080857852332201e+02 -3.265906165554512651e-03 1.963457496825285822e-03 -1.398771363412710383e-02 3.088127086547619998e-02 1.672285950246031301e-03 -8.927174265079367271e-04 -4.244701868253958994e-04 1.000000000000000000e+00 4.054520000000000124e+00 1.643913243040000083e+01 6.665279122170541370e+01 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.607408786477914759e+02 -8.017150588157394330e-04 2.213062996825285525e-03 1.398740365872893493e-03 1.085352386547620146e-02 1.533498042246031435e-03 -7.043727325079367782e-04 -4.042463468253959091e-04 1.000000000000000000e+00 6.081780000000000186e+00 3.698804796840000364e+01 2.249531703732557730e+02 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.611418084786566283e+02 -1.861685769802005528e-04 1.047713639682528591e-02 1.167152736587289547e-02 1.489745686547620102e-02 1.548124779246031315e-03 -5.563730125079367241e-04 -1.481969968253959513e-04 1.000000000000000000e+00 8.109040000000000248e+00 6.575652972160000331e+01 5.332223297736433096e+02 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.625281634787599501e+02 -4.117603177916723598e-05 9.983357396825286167e-03 2.268076636587289252e-02 3.341529466547620009e-02 1.378939226246031274e-03 -5.824833125079368051e-04 -1.637155968253958946e-04 1.000000000000000000e+00 1.013630000000000031e+01 1.027445776899999998e+02 1.041449862839147045e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.600881821274363688e+02 -8.724125662125817594e-06 2.118458339682528432e-02 -3.638986341271063796e-04 7.819901865476201752e-03 1.343526296246031447e-03 -4.266495825079367706e-04 -3.036430682539588335e-05 1.000000000000000000e+00 1.216356000000000037e+01 1.479521918736000146e+02 1.799625362986046184e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.604916986023719687e+02 -1.779095604735100062e-06 2.110365339682528443e-02 -1.333419963412710470e-02 3.556263356547620380e-02 1.176156066246031480e-03 -2.915726925079367704e-04 -1.372058068253959344e-04 1.000000000000000000e+00 1.419082000000000043e+01 2.013793722724000190e+02 2.857738423630619764e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.631421465595319091e+02 -3.505829544571274576e-07 3.057060839682528355e-02 2.450720636587289808e-02 2.371273386547620085e-02 1.109560806246031196e-03 -4.451344925079367475e-04 -4.868320682539588849e-05 1.000000000000000000e+00 1.621808000000000050e+01 2.630261188864000133e+02 4.265778638189146477e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.607404475404462687e+02 -6.698859808659203534e-08 4.212096239682527887e-02 4.201216436587289910e-02 1.535293186547620134e-02 1.200805636246031222e-03 -4.756955025079367830e-04 4.163935317460414412e-05 1.000000000000000000e+00 1.824533999999999878e+01 3.328924317155999688e+02 6.073735600077903655e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.602563139919928403e+02 -1.244731173797263160e-08 3.868433239682528280e-02 3.198940136587289512e-02 1.951312986547620171e-02 1.210561816246031458e-03 -5.037184525079367245e-04 1.853174317460412092e-05 1.000000000000000000e+00 2.027260000000000062e+01 4.109783107599999994e+02 8.331598902713176358e+03 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.597932184819765098e+02 -2.254732652415686417e-09 3.464322639682528016e-02 2.498494136587289804e-02 6.040923865476201249e-03 1.251570966246031346e-03 -3.408492325079367884e-04 -2.053166825395852726e-06 1.000000000000000000e+00 2.229986000000000246e+01 4.972837560196001050e+02 1.108935813951124146e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.600692996257253071e+02 -3.990740854251533582e-10 3.209237439682528781e-02 1.811942636587289546e-02 2.605920586547620307e-02 1.177732906246031254e-03 -5.077881225079367488e-04 5.365363174604119087e-06 1.000000000000000000e+00 2.432712000000000074e+01 5.918087674944000582e+02 1.439700290388836947e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.605557611538409901e+02 -6.912161668563663771e-11 4.299601339682528056e-02 2.895994436587289583e-02 1.417107986547620074e-02 1.265060666246031361e-03 -7.339628625079367124e-04 1.238756831746040893e-04 1.000000000000000000e+00 2.635437999999999903e+01 6.945533451843999728e+02 1.830452278926084546e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.595077391981066626e+02 -1.172812338719269821e-11 3.317149439682529066e-02 1.328090936587289494e-02 1.022893186547620126e-02 1.374031606246031408e-03 -5.220871725079368267e-04 1.413575031746041374e-04 1.000000000000000000e+00 2.838164000000000087e+01 8.055174890896000761e+02 2.286190738904495811e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.587979640652715148e+02 -1.964186707357858839e-12 2.405623739682528558e-02 -1.810522634127103431e-03 1.576445486547620178e-02 1.135956976246031312e-03 -5.014120825079368057e-04 1.611867531746041847e-04 1.000000000000000000e+00 3.040890000000000271e+01 9.247011992100001407e+02 2.811914629665697794e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.596659555937277446e+02 -3.223083090335421760e-13 3.234481339682528100e-02 2.004408536587289763e-02 2.356408786547620204e-02 1.221481986246031413e-03 -6.670757425079366920e-04 1.487958231746040706e-04 1.000000000000000000e+00 3.243616000000000099e+01 1.052104475545600053e+03 3.412622910551317182e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.584320401508220471e+02 -9.003308688382024074e-14 3.619885939682528087e-02 2.789771365872894399e-03 9.189109865476198513e-03 1.135373276246031326e-03 -4.355060825079367357e-04 1.002332231746041503e-04 1.000000000000000000e+00 3.446341999999999928e+01 1.187727318096400040e+03 4.093314540902982844e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.580819809866689525e+02 -3.906123070653587881e-14 3.660551639682528557e-02 -1.860463412710344766e-05 2.714363586547620388e-02 1.120834376246031315e-03 -4.501944025079367639e-04 1.202024331746040682e-04 1.000000000000000000e+00 3.649067999999999756e+01 1.331569726862399875e+03 4.858988480062322924e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.594975650647641601e+02 1.601430181974213516e-14 3.905011839682528962e-02 9.654908365872898190e-03 1.281982286547620267e-02 1.076811816246031270e-03 -6.519448025079367355e-04 1.400206731746040907e-04 1.000000000000000000e+00 3.851794000000000295e+01 1.483631701843600240e+03 5.714643687370968837e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.605247214249999956e+02 9.795389708948228080e-02 3.677422139682529068e-02 2.608958736587289190e-02 2.185457486547620273e-02 1.235064666246031345e-03 -6.071577725079368385e-04 1.763112331746040417e-04 1.000000000000000000e+00 4.054520000000000124e+01 1.643913243039999998e+03 6.665279122170541086e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.615678287570015073e+02 7.636684347997682032e+00 2.837993739682528535e-02 3.336949636587289297e-02 2.712176086547619935e-02 1.121492386246031227e-03 -3.887845825079367800e-04 9.757465317460415049e-05 1.000000000000000000e+00 4.257245999999999952e+01 1.812414350451600058e+03 7.715893743802672543e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.606581922590170848e+02 1.688917484910420086e+01 3.424000439682528540e-02 5.953364365872893665e-03 1.839351286547620187e-02 1.118185646246031353e-03 -3.785339525079367985e-04 2.395393531746040213e-04 1.000000000000000000e+00 4.459972000000000492e+01 1.989135024078400420e+03 8.871486511608993169e+04 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.585156749757550756e+02 1.131722623416632167e+01 3.749442739682529169e-02 -1.501305634127106381e-03 1.711901486547620296e-02 1.368664136246031289e-03 -5.395318625079368116e-04 1.879513531746040403e-04 1.000000000000000000e+00 4.662698000000000320e+01 2.174075263920400175e+03 1.013705638493112347e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.589431610190735000e+02 1.345714208625528263e+00 3.218309039682527850e-02 -7.129233634127103703e-03 2.217183586547620197e-02 1.429032466246031368e-03 -5.373530925079368203e-04 1.592906031746042046e-04 1.000000000000000000e+00 4.865424000000000149e+01 2.367235069977600233e+03 1.151760232311069558e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.588859099636547398e+02 -3.786197907636791982e+00 2.637535539682528754e-02 -1.390411634127106111e-03 1.310852586547620047e-02 1.517677216246031326e-03 -5.291699825079366776e-04 1.052765531746040640e-04 1.000000000000000000e+00 5.068149999999999977e+01 2.568614442250000138e+03 1.301812328548933729e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.584379032107140688e+02 -4.100675927724760861e+00 2.384725139682528430e-02 -1.080336163412710590e-02 -4.173090134523799177e-03 1.358116916246031227e-03 -4.800622625079367331e-04 5.590095317460413646e-05 1.000000000000000000e+00 5.270875999999999806e+01 2.778213380737599891e+03 1.464361823140867637e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.595410206851418025e+02 -2.630373115496683400e+00 1.004822839682528376e-02 9.314062365872892435e-03 -9.878861345237952007e-04 1.325770276246031245e-03 -4.428060525079367620e-04 -2.427069682539584328e-05 1.000000000000000000e+00 5.473602000000000345e+01 2.996031885440400401e+03 1.639908612021034642e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.607257898907158165e+02 -1.286200190109046071e+00 2.464792639682528499e-02 2.035648336587289609e-02 -6.855731345237967012e-04 1.419879466246031343e-03 -6.113658025079368383e-04 1.115435631746041455e-04 1.000000000000000000e+00 5.676328000000000173e+01 3.222069956358400304e+03 1.828952591123596649e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.588783224743828555e+02 -5.223127938147428262e-01 2.786826139682528278e-02 1.117468365872894415e-03 -1.241363713452380002e-02 1.415631896246031260e-03 -4.147048725079367825e-04 -1.723451682539593396e-05 1.000000000000000000e+00 5.879054000000000002e+01 3.456327593491600055e+03 2.031993656382716435e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.588356428472260973e+02 -1.842356108573543483e-01 2.425059939682528559e-02 -4.276288634127104610e-03 -1.091986813452380106e-02 1.392750786246031280e-03 -4.490394525079367555e-04 -1.003586682539589405e-05 1.000000000000000000e+00 6.081780000000000541e+01 3.698804796840000563e+03 2.249531703732558235e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.597484695395635299e+02 -5.807770807625862314e-02 1.325085839682528521e-02 -3.310795634127106785e-03 2.611598386547619999e-02 1.344393666246031368e-03 -5.894356525079367040e-04 -4.194197682539594491e-05 1.000000000000000000e+00 6.284506000000000370e+01 3.949501566403600464e+03 2.482066629107282788e+05 -0.000000000000000000e+00 -0.000000000000000000e+00 -0.000000000000000000e+00
2.586971711680070598e+02 -1.669108953593786623e-02 1.520021739682528641e-02 -6.521448634127104127e-03 1.323596186547620207e-02 1.018124536246031329e-03 -5.651434125079368188e-04 -1.186629568253958888e-04 1.000000000000000000e+00 6.487232000000000198e+01 4.208417902182400212e+03 2.730098328441053745e+05 6.637850845511725772e-01 -0.000000000000000000e+00 -0.000000000000000000e+00
2.578038305276642745e+02 -4.438179736810902651e-03 1.418104939682528556e-02 -1.458225563412710556e-02 2.076608686547620070e-02 7.166574462460313308e-04 -6.010164225079367385e-04 -2.031235568253959454e-04 1.000000000000000000e+00 6.689958000000000027e+01 4.475553804176400263e+03 2.994126697668033885e+05 2.437840493460591773e+01 -0.000000000000000000e+00 -0.000000000000000000e+00
2.575900012845362994e+02 -1.104415351769467155e-03 1.171448539682528461e-02 -6.411356341271060022e-04 2.179420786547620059e-02 7.711998362460313790e-04 -5.958785525079367436e-04 -1.778974268253958766e-04 1.000000000000000000e+00 6.892683999999999855e+01 4.750909272385600161e+03 3.274651632722386275e+05 1.195928942034693989e+02 -0.000000000000000000e+00 -0.000000000000000000e+00
2.592522649854503811e+02 -2.595046810460775255e-04 5.653468396825284473e-03 -3.306909634127105230e-03 3.415740386547620050e-02 7.991702162460313699e-04 -5.105784425079367903e-04 -2.023469768253959109e-04 1.000000000000000000e+00 7.095409999999999684e+01 5.034484306809999907e+03 3.572173029538273695e+05 3.362968463074205374e+02 -0.000000000000000000e+00 -0.000000000000000000e+00
2.579003985729477790e+02 -5.799523371039054791e-05 4.075954396825285861e-03 -5.813851634127106816e-03 3.851734186547620120e-02 8.126851062460313437e-04 -4.455600825079367448e-04 -3.203095468253959032e-04 1.000000000000000000e+00 7.298135999999999513e+01 5.326278907449599501e+03 3.887190784049858339e+05 7.244798546627382620e+02 -0.000000000000000000e+00 -0.000000000000000000e+00
2.577110562270163996e+02 -1.240153176296573148e-05 1.982903996825284912e-03 -5.751847634127105896e-03 1.817295686547620165e-02 6.980794162460313449e-04 -3.607846825079367298e-04 -3.361090868253959027e-04 1.000000000000000000e+00 7.500862000000000762e+01 5.626293074304400761e+03 4.220204792191306478e+05 1.334131512685706639e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.593279674352351662e+02 -2.549813093163372416e-06 2.012354196825284422e-03 -3.176191634127106811e-03 2.634695186547620152e-02 5.562481362460312394e-04 -4.909143225079367614e-04 -2.835488168253958450e-04 1.000000000000000000e+00 7.703588000000000591e+01 5.934526807374400960e+03 4.571714949896775070e+05 2.215241413792596632e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.604506571831263386e+02 -5.061830223553558920e-07 3.248753396825284495e-03 5.653695365872894729e-03 3.363641326547620047e-02 4.461581362460312686e-04 -5.631164925079367844e-04 -1.737951468253959427e-04 1.000000000000000000e+00 7.906314000000000419e+01 6.250980106659601006e+03 4.942221153100429801e+05 3.417799151399689890e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.602953723174513812e+02 -9.736609629218024716e-08 6.825325968252849568e-04 1.423937136587289515e-02 3.023103586547620097e-02 7.006392762460313377e-04 -5.004090925079366942e-04 -1.539339168253958537e-04 1.000000000000000000e+00 8.109040000000000248e+01 6.575652972159999990e+03 5.332223297736432869e+05 4.991794318923266474e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.606369817776421769e+02 -1.820092841812642467e-08 -1.136286590317471534e-02 3.619031336587289621e-02 1.424289986547620096e-02 5.533487362460313193e-04 -4.338583525079367596e-04 -1.890155468253958962e-04 1.000000000000000000e+00 8.311766000000000076e+01 6.908545403875599732e+03 5.742221279738948215e+05 6.987216509779604166e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.593342616024719973e+02 -3.315296284192901071e-09 -5.857725263174714918e-03 2.357598136587289728e-02 1.897169486547620187e-02 7.518108062460313089e-04 -5.384554125079367383e-04 -1.363035768253958785e-04 1.000000000000000000e+00 8.514491999999999905e+01 7.249657401806400230e+03 6.172714995042138034e+05 9.454055317384982118e+03 -0.000000000000000000e+00 -0.000000000000000000e+00
2.586453520651357962e+02 -5.897348231645986935e-10 1.111030896825284872e-03 2.246285136587289344e-02 2.219625186547620130e-02 6.593569362460313795e-04 -4.778790125079367536e-04 -7.630101682539586726e-05 1.000000000000000000e+00 8.717217999999999734e+01 7.598988965952399667e+03 6.624204339580164524e+05 1.244230033515567993e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.590023951682784400e+02 -1.026285419063397840e-10 1.848365996825284893e-03 1.420209336587289345e-02 2.652135286547620263e-02 9.330586362460312937e-04 -5.569034125079367487e-04 -8.223069682539586433e-05 1.000000000000000000e+00 8.919944000000000983e+01 7.956540096313601680e+03 7.097189209287194535e+05 1.600194115650800268e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.602716614134758402e+02 -1.749597348817579870e-11 -4.677688603174715194e-03 1.815530536587289800e-02 5.745579865476198311e-03 6.605902962460313572e-04 -5.903785325079367440e-04 -1.106166468253958835e-04 1.000000000000000000e+00 9.122670000000000812e+01 8.322310792890000812e+03 7.592169500097383279e+05 2.018296737485818085e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.604482429819940421e+02 -2.915444713749153634e-12 9.829689682528536254e-05 1.461135536587289396e-02 1.032855886547619922e-02 6.060708362460314087e-04 -5.028199025079367092e-04 9.170133174604125012e-06 1.000000000000000000e+00 9.325396000000000640e+01 8.696301055681600701e+03 8.109645107944898773e+05 2.503536858362251587e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.603872844034092395e+02 -4.662988791519401875e-13 -8.091808403174714781e-03 2.668391636587289645e-02 9.499642865476200237e-03 6.190488562460314068e-04 -5.573827825079367406e-04 -1.419941268253958845e-04 1.000000000000000000e+00 9.528122000000000469e+01 9.078510884688401347e+03 8.650115928763902048e+05 3.060913437621728735e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.604142717232071504e+02 -8.815554369269722195e-14 7.379531968252847629e-04 1.966617536587289550e-02 5.218423865476204404e-03 7.821939762460313177e-04 -6.720836925079368140e-04 -1.368856682539584639e-05 1.000000000000000000e+00 9.730848000000000297e+01 9.468940279910400932e+03 9.214081858488556463e+05 3.695425434605876944e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.596725946468405937e+02 -4.790854546882667301e-14 3.729469396825285318e-03 1.677155036587289760e-02 9.729758654761985759e-04 7.744619962460313600e-04 -6.579227325079367063e-04 3.219561317460413550e-05 1.000000000000000000e+00 9.933574000000000126e+01 9.867589241347599454e+03 9.802042793053025380e+05 4.412071808656324720e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.581306961166553151e+02 -1.980567423292065087e-14 1.638672296825285576e-03 -6.475722634127104721e-03 1.390103865476201295e-03 4.816735362460312572e-04 -6.694806825079367436e-04 -9.350514682539593728e-05 1.000000000000000000e+00 1.013629999999999995e+02 1.027445776900000055e+04 1.041449862839146983e+06 5.215851519114699477e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.583217628919668982e+02 -1.405585884156201381e-14 7.728531396825284727e-03 -3.856817634127103489e-03 5.960830865476204887e-03 3.423149362460312529e-04 -7.660289725079367888e-04 2.281447317460411506e-05 1.000000000000000000e+00 1.033902599999999978e+02 1.068954586286759877e+04 1.105194926043805433e+06 6.111763525322629721e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.598821128165189407e+02 -1.691532273721650273e-14 2.477927296825284398e-03 1.116856365872893886e-03 9.179691865476201362e-03 7.097850162460313164e-04 -8.175605915079367601e-04 -5.294306825395908231e-06 1.000000000000000000e+00 1.054175199999999961e+02 1.111285352295039957e+04 1.171489458512694109e+06 7.104806786621743231e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.585404205373588979e+02 5.840602392497974451e-15 -3.963158031747146190e-04 7.451014365872893341e-03 3.865376865476201351e-03 5.380693362460314128e-04 -7.396422825079367394e-04 -2.474268682539594241e-05 1.000000000000000000e+00 1.074447800000000086e+02 1.154438074924840112e+04 1.240383449839229695e+06 8.199980262353675789e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.574534264776349914e+02 1.712000128500727594e-14 -8.327767103174715108e-03 6.492053658728944021e-04 -4.315605134523795017e-03 4.314180362460313858e-04 -5.235343725079368016e-04 -1.426233668253959388e-04 1.000000000000000000e+00 1.094720400000000069e+02 1.198412754176160161e+04 1.311926889616827713e+06 9.402282911860039167e+04 -0.000000000000000000e+00 -0.000000000000000000e+00
2.587782193488699249e+02 1.886219388254293476e-14 4.761096396825285557e-03 -7.202196341271061009e-04 -2.113392134523800481e-03 4.052769362460314270e-04 -7.262424025079366931e-04 -9.712075682539588351e-05 1.000000000000000000e+00 1.114993000000000052e+02 1.243209390049000103e+04 1.386169767438904848e+06 1.071671369448246987e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.598656445159390387e+02 2.347227720962027251e-14 -4.165797203174715496e-03 1.295209736587289717e-02 -1.783551213452379963e-02 4.884648362460312747e-04 -5.813059725079367619e-04 -7.004130682539588988e-05 1.000000000000000000e+00 1.135265600000000035e+02 1.288827982543360122e+04 1.463162072898877319e+06 1.214827156956259423e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.600989598110621728e+02 7.813072328225483567e-15 1.221070796825285756e-03 1.337387336587289588e-02 -1.252786513452380096e-02 2.161711362460314121e-04 -5.074466025079367101e-04 2.142214317460411615e-05 1.000000000000000000e+00 1.155538200000000018e+02 1.335268531659240034e+04 1.542953795590161346e+06 1.370195549644204148e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.608026132195976174e+02 -8.925257391752444914e-15 1.228668539682528649e-02 1.208959736587289502e-02 -2.235864113452379343e-02 1.684635362460312931e-04 -2.464530425079367254e-04 1.124107331746041069e-04 1.000000000000000000e+00 1.175810800000000000e+02 1.382531037396640022e+04 1.625594925106173148e+06 1.538276443446243939e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.599810088414655525e+02 -1.025966746681070654e-14 2.181112039682528425e-02 -1.205161763412710557e-02 -1.086435413452380150e-02 -5.987476375396861422e-05 -3.407551025079368036e-04 1.726038431746041530e-04 1.000000000000000000e+00 1.196083399999999983e+02 1.430615499755559904e+04 1.711135451040329412e+06 1.719569734296541719e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.590798220474434288e+02 -1.906044947566650386e-14 1.003784239682528612e-02 6.137143365872895634e-03 3.477642546547619895e-02 -2.676582637539685807e-04 -2.744146425079367797e-04 7.012074317460411776e-05 1.000000000000000000e+00 1.216356000000000108e+02 1.479521918736000225e+04 1.799625362986046588e+06 1.914575318129261141e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.594494534850605305e+02 -2.578066919736734499e-14 -2.027138103174714809e-03 -6.372505634127105523e-03 2.919624086547620290e-02 -3.534829637539685723e-04 -3.414351725079367138e-04 -9.094636825395874605e-06 1.000000000000000000e+00 1.236628600000000091e+02 1.529250294337960258e+04 1.891114650536739733e+06 2.123793090878562944e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.580147282408462956e+02 -1.888533906750968316e-14 -1.798189060317471888e-02 8.892993658728941264e-04 1.529699586547620185e-02 -1.785335637539686715e-04 -3.668640225079367609e-04 -1.523243868253959478e-04 1.000000000000000000e+00 1.256901200000000074e+02 1.579800626561440185e+04 1.985653303285826230e+06 2.347722948478611070e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.573028837927551535e+02 -1.384973992480027394e-14 -2.226030160317471474e-02 -1.401617563412710550e-02 9.232429865476204922e-03 -2.170017637539685754e-04 -6.020543625079367335e-04 -1.919957668253958593e-04 1.000000000000000000e+00 1.277173800000000057e+02 1.631172915406440188e+04 2.083291310826721834e+06 2.586864786863568006e+05 -0.000000000000000000e+00 -0.000000000000000000e+00
2.578661686497533765e+02 6.466134453156046314e-15 -2.225478460317471471e-02 -4.346986634127105592e-03 4.281016865476203193e-03 7.098093624603144143e-05 -4.939255525079367390e-04 -1.331850268253959284e-04 1.000000000000000000e+00 1.297446400000000040e+02 1.683367160872960085e+04 2.184078662752842996e+06 0.000000000000000000e+00 5.310280676409380618e+00 -0.000000000000000000e+00
2.591778558577004219e+02 3.881029512404210243e-14 -3.350587260317471061e-02 3.708508365872893731e-03 3.303729865476202898e-03 4.290136246031276898e-06 -4.810798125079367789e-04 -1.990675968253958908e-04 1.000000000000000000e+00 1.317719000000000165e+02 1.736383362961000603e+04 2.288065348657606635e+06 0.000000000000000000e+00 5.366368736595970290e+01 -0.000000000000000000e+00
2.586265557848932417e+02 2.496704000974017557e-14 -4.055766460317471178e-02 2.324536365872896526e-03 -1.314141813452379681e-02 -2.111011637539687423e-04 -4.601720925079367608e-04 -2.897881768253959302e-04 1.000000000000000000e+00 1.337991600000000005e+02 1.790221521670560105e+04 2.395301358134427108e+06 0.000000000000000000e+00 1.950272394768473418e+02 -0.000000000000000000e+00
2.590581804587237684e+02 1.020916719346166513e-14 -3.380893360317471785e-02 4.358221365872893410e-03 -1.662428913452379531e-02 -3.211422637539687076e-04 -4.006317125079367453e-04 -1.464107968253959514e-04 1.000000000000000000e+00 1.358264200000000130e+02 1.844881637001640411e+04 2.505836680776723661e+06 0.000000000000000000e+00 4.793905304253556778e+02 -0.000000000000000000e+00
2.583827214705520987e+02 2.027411219651766781e-14 -3.782695560317471395e-02 1.149229936587289197e-02 -1.630400713452379718e-02 -2.047094637539685711e-04 -2.136010125079367472e-04 -1.059907068253958815e-04 1.000000000000000000e+00 1.378536799999999971e+02 1.900363708954240064e+04 2.619721306177909020e+06 0.000000000000000000e+00 9.567431536277551913e+02 -0.000000000000000000e+00
2.588504729398947006e+02 -2.959667608385212738e-16 -2.737255860317471326e-02 2.306047836587289679e-02 -1.175693013452380059e-02 -1.525203637539687424e-04 -2.631168025079367104e-04 -9.378550682539587170e-05 1.000000000000000000e+00 1.398809400000000096e+02 1.956667737528360158e+04 2.737005223931403365e+06 0.000000000000000000e+00 1.677074702500338617e+03 -0.000000000000000000e+00
2.575437003556809259e+02 -1.510831198685233849e-14 -2.817193160317471579e-02 -8.620721634127109789e-03 -1.014567713452380060e-02 -2.024390637539686885e-04 2.442606749206328864e-05 -1.798543568253958532e-04 1.000000000000000000e+00 1.419081999999999937e+02 2.013793722723999963e+04 2.857738423630618956e+06 0.000000000000000000e+00 2.690374770459364299e+03 -0.000000000000000000e+00
2.571050917428615890e+02 -1.222166334909334567e-14 -3.965530660317471978e-02 -7.093271634127106678e-03 -2.676973013452380035e-02 -2.402326637539686175e-04 -1.294388825079367941e-04 -2.178491468253959109e-04 1.000000000000000000e+00 1.439354600000000062e+02 2.071741664541160208e+04 2.981970894868975971e+06 0.000000000000000000e+00 4.046632950921140036e+03 -0.000000000000000000e+00
2.578799640847943806e+02 -2.669000963219823434e-14 -4.578354560317471345e-02 -1.935690153412710640e-02 -1.530625134523795616e-03 -3.285852637539686972e-04 -2.997716825079367771e-04 -1.772051168253958690e-04 1.000000000000000000e+00 1.459627199999999903e+02 2.130511562979839800e+04 3.109752627239886671e+06 0.000000000000000000e+00 5.795838837301906096e+03 -0.000000000000000000e+00
2.593543512047501167e+02 -1.557425937872241460e-14 -5.462931060317471887e-02 -1.786486341271049938e-04 -2.675493513452380234e-02 -3.041632637539686251e-04 -2.994083325079367969e-04 -2.266904168253959084e-04 1.000000000000000000e+00 1.479899800000000027e+02 2.190103418040040197e+04 3.241133610336771701e+06 0.000000000000000000e+00 7.987982023017991196e+03 -0.000000000000000000e+00
2.592103613515139955e+02 1.025550116606464834e-14 -5.747345460317471177e-02 -2.301652634127106245e-03 -3.055690313452380513e-02 -1.852517637539686981e-04 -7.782878250793675776e-05 -2.941239768253959370e-04 1.000000000000000000e+00 1.500172400000000152e+02 2.250517229721760305e+04 3.376163833753045183e+06 0.000000000000000000e+00 1.067305210148565311e+04 -0.000000000000000000e+00
2.593191728453962241e+02 7.314866478049465998e-15 -4.823187060317471464e-02 1.890068236587289646e-02 -4.777992713452379470e-02 -3.452388637539688387e-04 -1.024134925079367604e-04 -3.109670268253958468e-04 1.000000000000000000e+00 1.520444999999999993e+02 2.311752998025000124e+04 3.514893287082121242e+06 0.000000000000000000e+00 1.390103866612112324e+04 -0.000000000000000000e+00
2.592008769899240974e+02 2.221644269997001270e-14 -4.118386960317471646e-02 1.733267436587289378e-02 -4.355931913452379400e-02 -3.705732637539686618e-04 -2.771284925079367436e-04 -1.953945868253958865e-04 1.000000000000000000e+00 1.540717600000000118e+02 2.373810722949760384e+04 3.657371959917420056e+06 0.000000000000000000e+00 1.772193131034077305e+04 -0.000000000000000000e+00
2.594888566963954304e+02 3.423980053733376596e-14 -3.876614060317470911e-02 1.016017036587289757e-02 -5.628503713452380486e-02 -3.304482637539686487e-04 -1.241367425079367053e-04 -9.316598682539590105e-05 1.000000000000000000e+00 1.560990199999999959e+02 2.436690404496039991e+04 3.803649841852353886e+06 0.000000000000000000e+00 2.218571962756076755e+04 -0.000000000000000000e+00
2.592471779187787320e+02 2.090355192478126067e+00 -4.206244260317471007e-02 1.105673136587289468e-02 -4.754148013452379196e-02 -2.150553637539685901e-04 -3.158812625079367815e-04 -1.838400068253959359e-04 1.000000000000000000e+00 1.581262800000000084e+02 2.500392042663840402e+04 3.953776922480343841e+06 0.000000000000000000e+00 2.734239321119751912e+04 -0.000000000000000000e+00
2.595144644726888146e+02 1.390900631135700216e+01 -4.419308660317471105e-02 2.374663636587289600e-02 -5.757486113452379983e-02 -3.322781637539685886e-04 3.979992749206327091e-05 -1.636741968253958715e-04 1.000000000000000000e+00 1.601535399999999925e+02 2.564915637453159798e+04 4.107803191394800786e+06 0.000000000000000000e+00 3.324194165466715640e+04 -0.000000000000000000e+00
2.594934264342520578e+02 1.557696618507103814e+01 -4.037264960317471507e-02 1.567967136587289367e-02 -6.731542113452379517e-02 -3.889040637539684965e-04 6.342409749206321789e-05 -1.471519668253958805e-04 1.000000000000000000e+00 1.621808000000000050e+02 2.630261188863999996e+04 4.265778638189146295e+06 0.000000000000000000e+00 3.993435455138613179e+04 -0.000000000000000000e+00
2.599376394425571561e+02 6.004799507075502696e+00 -2.857072660317471618e-02 1.227729936587289294e-02 -4.839276813452379755e-02 -4.437891637539687073e-04 9.347311749206322923e-05 -8.371388682539590391e-05 1.000000000000000000e+00 1.642080600000000175e+02 2.696428696896360634e+04 4.427753252456794493e+06 0.000000000000000000e+00 4.746962149477063213e+04 -0.000000000000000000e+00
2.583130545235978275e+02 -1.994345614804481137e+00 -3.650895860317471264e-02 6.498904365872894273e-03 -2.158240113452379594e-02 -4.707137637539686968e-04 5.781790749206320440e-05 -1.285526168253958669e-04 1.000000000000000000e+00 1.662353200000000015e+02 2.763418161550239893e+04 4.593777023791158572e+06 0.000000000000000000e+00 5.589773207823683333e+04 -0.000000000000000000e+00
2.576154366178921009e+02 -4.354781600224979066e+00 -3.754501060317471522e-02 -1.127231463412710355e-02 -2.067503813452380157e-02 -4.761822637539686598e-04 1.106139774920631693e-04 -2.297192168253959360e-04 1.000000000000000000e+00 1.682625800000000140e+02 2.831229582825640318e+04 4.763899941785659641e+06 0.000000000000000000e+00 6.526867589520123147e+04 -0.000000000000000000e+00
2.582447671201464914e+02 -3.421137938612250018e+00 -3.709204260317471025e-02 -2.815319033412710253e-02 -3.209472813452379780e-02 -4.201502637539685295e-04 1.587400974920632384e-04 -1.486038868253958603e-04 1.000000000000000000e+00 1.702898399999999981e+02 2.899862960722560092e+04 4.938171996033710428e+06 0.000000000000000000e+00 7.563244253907985694e+04 -0.000000000000000000e+00
2.578366809073544346e+02 -1.884871573445409121e+00 -4.559719660317471113e-02 -2.012774773412710425e-02 -4.258769413452380415e-02 -5.238649637539687445e-04 1.121453374920631770e-04 -3.780857468253959143e-04 1.000000000000000000e+00 1.723171000000000106e+02 2.969318295241000305e+04 5.116643176128730178e+06 0.000000000000000000e+00 8.703902160328927857e+04 -0.000000000000000000e+00
2.575530122743335824e+02 -8.333182129281194728e-01 -5.434145660317471482e-02 -3.934316634127105194e-03 -2.802218613452379936e-02 -6.544949637539688135e-04 -4.547183250793677273e-05 -4.325602468253959159e-04 1.000000000000000000e+00 1.743443599999999947e+02 3.039595586380959867e+04 5.299363471664131619e+06 0.000000000000000000e+00 9.953840268124543945e+04 -0.000000000000000000e+00
2.582481297574381642e+02 -3.138703752643597356e-01 -7.287733960317471782e-02 -5.080906634127104610e-03 -3.453186913452380158e-02 -4.803973637539688153e-04 1.780781974920633099e-04 -5.289674068253958326e-04 1.000000000000000000e+00 1.763716200000000072e+02 3.110694834142440232e+04 5.486382872233334929e+06 0.000000000000000000e+00 1.131805753663649812e+05 -0.000000000000000000e+00
2.585080012650139452e+02 -1.043136344467513188e-01 -6.379200960317471525e-02 -1.374258063412710576e-02 -2.450723213452379867e-02 -4.271225637539686863e-04 1.437427974920632524e-04 -5.507417468253958956e-04 1.000000000000000000e+00 1.783988800000000197e+02 3.182616038525440672e+04 5.677751367429755628e+06 0.000000000000000000e+00 1.280155292520640214e+05 -0.000000000000000000e+00
2.589810122439655515e+02 -3.132084140102636693e-02 -7.903580860317471757e-02 -1.139652463412710488e-02 -3.978782313452379482e-02 -7.604801637539687284e-04 7.115520749206329099e-05 -5.629854968253959323e-04 1.000000000000000000e+00 1.804261400000000037e+02 3.255359199529960097e+04 5.873518946846805513e+06 0.000000000000000000e+00 1.440932539317586052e+05 -0.000000000000000000e+00
2.592973588137387537e+02 -8.642004611292256402e-03 -7.117489360317472147e-02 6.563063658728933436e-04 -1.220494713452379559e-02 -1.040174863753968570e-03 1.255475674920631573e-04 -4.912412868253959080e-04 1.000000000000000000e+00 1.824534000000000162e+02 3.328924317156000325e+04 6.073735600077906623e+06 0.000000000000000000e+00 1.614637389988654468e+05 -0.000000000000000000e+00
2.597658000548336190e+02 -2.219631667718219379e-03 -7.383044660317471253e-02 1.410036136587289324e-02 7.094414865476204868e-03 -1.154148363753968592e-03 1.615656974920631457e-04 -4.779029468253959113e-04 1.000000000000000000e+00 1.844806600000000003e+02 3.403311391403560265e+04 6.278451316716470756e+06 0.000000000000000000e+00 1.801769740468003438e+05 -0.000000000000000000e+00
2.600563664053747743e+02 -5.360794304548768905e-04 -6.586775660317471803e-02 1.655973336587289457e-02 -3.370324134523795812e-03 -1.024209463753968704e-03 2.467476974920631986e-04 -3.545917268253958969e-04 1.000000000000000000e+00 1.865079200000000128e+02 3.478520422272640280e+04 6.487716086355919018e+06 0.000000000000000000e+00 2.002829486689801270e+05 -0.000000000000000000e+00
2.602901128078428314e+02 -1.227230843113079188e-04 -7.049731160317471157e-02 2.987536736587289438e-02 -2.844490013452380395e-02 -1.023074663753968574e-03 3.075099974920633130e-04 -4.335420468253959139e-04 1.000000000000000000e+00 1.885351799999999969e+02 3.554551409763239644e+04 6.701579898589661345e+06 0.000000000000000000e+00 2.218316524588204629e+05 -0.000000000000000000e+00
2.598264999690118202e+02 -2.680547757517946417e-05 -6.999111960317472292e-02 3.268389136587289412e-02 -1.014415313452379785e-02 -1.264280463753968825e-03 3.754828974920633141e-04 -4.390807868253959116e-04 1.000000000000000000e+00 1.905624400000000094e+02 3.631404353875360539e+04 6.920092743011121638e+06 0.000000000000000000e+00 2.448730750097382988e+05 -0.000000000000000000e+00
2.593596969358279694e+02 -5.616675076130314912e-06 -7.061410360317471602e-02 1.946565236587289444e-02 1.298353186547620067e-02 -1.411778063753968502e-03 3.464828974920631513e-04 -4.315975268253958975e-04 1.000000000000000000e+00 1.925896999999999935e+02 3.709079254609000054e+04 7.143304609213708900e+06 0.000000000000000000e+00 0.000000000000000000e+00 2.050658692729931676e-01
2.590111897581169842e+02 -1.134042666258773269e-06 -6.430232360317471307e-02 9.387877365872897284e-03 1.989402986547620170e-02 -1.206621563753968590e-03 3.619273974920631963e-04 -2.785400868253959123e-04 1.000000000000000000e+00 1.946169600000000059e+02 3.787576111964160373e+04 7.371265486790845171e+06 0.000000000000000000e+00 0.000000000000000000e+00 1.792219728288165825e+01
2.592187248340080146e+02 -2.214888922805985211e-07 -6.659902160317471287e-02 2.656261365872894520e-03 3.570156865476202535e-03 -1.165767363753968631e-03 2.659288974920633259e-04 -4.230125468253958989e-04 1.000000000000000000e+00 1.966442200000000184e+02 3.866894925940840767e+04 7.604025365335945040e+06 0.000000000000000000e+00 0.000000000000000000e+00 1.001701567040806395e+02
2.568569463550700789e+02 -4.198101968002987330e-08 -7.323089060317471144e-02 -1.627500873412710686e-02 -1.577161013452380023e-02 -1.102308463753968581e-03 3.161176974920632205e-04 -5.168182368253959342e-04 1.000000000000000000e+00 1.986714800000000025e+02 3.947035696539039782e+04 7.841634234442420304e+06 0.000000000000000000e+00 0.000000000000000000e+00 2.969385375491461332e+02
2.567240790713361207e+02 -7.743329643075380204e-09 -6.897513560317471148e-02 -1.611915993412710302e-02 -2.616736134523796331e-03 -1.200253663753968539e-03 1.929508974920633073e-04 -5.036689468253959488e-04 1.000000000000000000e+00 2.006987400000000150e+02 4.027998423758760327e+04 8.084142083703693934e+06 0.000000000000000000e+00 0.000000000000000000e+00 6.582169332343630686e+02
2.586193994029661667e+02 -1.393367178829911838e-09 -5.634666760317472156e-02 -1.460659013412710441e-02 -7.355770134523799408e-03 -1.106632863753968712e-03 2.397183974920633369e-04 -2.815068668253959110e-04 1.000000000000000000e+00 2.027259999999999991e+02 4.109783107600000221e+04 8.331598902713175863e+06 0.000000000000000000e+00 0.000000000000000000e+00 1.233994937175999667e+03
2.583841010140561707e+02 -2.451015754665610884e-10 -5.365690960317472114e-02 -3.176442634127106535e-03 -1.357681813452379926e-02 -1.359759063753968565e-03 3.363015974920633170e-04 -3.537066968253959058e-04 1.000000000000000000e+00 2.047532600000000116e+02 4.192389748062760191e+04 8.584054681064289063e+06 0.000000000000000000e+00 0.000000000000000000e+00 2.074262142790351845e+03
2.580570629821268653e+02 -4.223306201495954454e-11 -6.084147260317471217e-02 -3.061121634127104973e-03 -7.098403134523798008e-03 -1.362605363753968592e-03 3.012735974920632180e-04 -3.800203668253959157e-04 1.000000000000000000e+00 2.067805199999999957e+02 4.275818345147039508e+04 8.841559408350443467e+06 0.000000000000000000e+00 0.000000000000000000e+00 3.229008143493673742e+03
2.579470443876787158e+02 -7.156816105084053148e-12 -5.542562460317471129e-02 -7.138105634127103749e-03 -8.081440134523797114e-03 -1.391156263753968482e-03 4.135093974920632997e-04 -3.021409368253959084e-04 1.000000000000000000e+00 2.088077800000000082e+02 4.360068898852840357e+04 9.104163074165061116e+06 0.000000000000000000e+00 0.000000000000000000e+00 4.748222532702277931e+03
2.599391914289993224e+02 -1.207054311834787104e-12 -3.929887060317471814e-02 -1.462635634127105316e-03 8.798824865476201351e-03 -1.426038263753968451e-03 4.433595974920632532e-04 -2.323506468253958533e-04 1.000000000000000000e+00 2.108350399999999922e+02 4.445141409180159826e+04 9.371915668101552874e+06 0.000000000000000000e+00 0.000000000000000000e+00 6.681894903832399905e+03
2.598000299780277942e+02 -2.147932142301532625e-13 -3.044666260317471437e-02 3.492261365872894457e-03 1.358364186547620159e-02 -1.401727663753968636e-03 6.015499974920632828e-04 -3.316270368253958971e-04 1.000000000000000000e+00 2.128623000000000047e+02 4.531035876129000098e+04 9.644867179753340781e+06 0.000000000000000000e+00 0.000000000000000000e+00 9.080014850300372927e+03
2.586574230707965967e+02 -4.005466736646009179e-14 -2.476531660317471406e-02 -5.519612634127105122e-03 -4.220371134523795420e-03 -1.356848963753968466e-03 6.767285974920632342e-04 -1.464826568253959098e-04 1.000000000000000000e+00 2.148895600000000172e+02 4.617752299699360447e+04 9.923067598713837564e+06 0.000000000000000000e+00 0.000000000000000000e+00 1.199257196552245478e+04
2.580585287471001266e+02 5.576182320169238469e-15 -2.481630060317471451e-02 -2.066747903412710641e-02 -8.855922134523797062e-03 -1.513199763753968717e-03 7.608736974920631479e-04 -2.132055668253959218e-04 1.000000000000000000e+00 2.169168200000000013e+02 4.705290679891240143e+04 1.020656691457645781e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.546955584291486957e+04
2.596834585519351322e+02 2.568277743391322110e-14 -1.720004460317471617e-02 -3.208105634127104283e-03 6.712917865476203394e-03 -1.568501263753968675e-03 8.467745974920633202e-04 -4.144829682539588698e-05 1.000000000000000000e+00 2.189440800000000138e+02 4.793651016704640642e+04 1.049541511693462171e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.956095607589399515e+04
2.602373452688122484e+02 2.864205519224055026e-14 -6.253708943174715595e-03 -9.945376634127107290e-03 4.145976865476200257e-03 -1.333169263753968686e-03 9.531898974920633176e-04 3.677603317460409731e-05 1.000000000000000000e+00 2.209713399999999979e+02 4.882833310139559762e+04 1.078966219538174197e+07 0.000000000000000000e+00 0.000000000000000000e+00 2.431676225787599833e+04
2.597039792615581746e+02 4.243141342386595795e-14 1.595174939682528562e-02 -1.055324663412710549e-02 -1.035979134523801193e-03 -1.308523563753968659e-03 9.431188974920631492e-04 1.944633531746040077e-04 1.000000000000000000e+00 2.229986000000000104e+02 4.972837560196000413e+04 1.108935813951123878e+07 0.000000000000000000e+00 0.000000000000000000e+00 2.978696398227728423e+04
2.587581297465923740e+02 5.218249358888024564e-14 1.167652339682528559e-02 -3.154518813412710704e-02 -1.016064713452379670e-02 -1.509701063753968696e-03 1.229371797492063208e-03 3.136779317460414856e-05 1.000000000000000000e+00 2.250258599999999944e+02 5.063663766873959685e+04 1.139455293891652301e+07 0.000000000000000000e+00 0.000000000000000000e+00 3.602155084251398512e+04
2.603926301344870922e+02 3.008952168432361289e-14 5.463445239682528098e-02 1.774224736587289714e-02 -2.389310713452379165e-02 -1.121983163753968726e-03 1.694905097492063056e-03 6.866294531746040862e-04 1.000000000000000000e+00 2.270531200000000069e+02 5.155311930173440487e+04 1.170529658319101855e+07 0.000000000000000000e+00 0.000000000000000000e+00 4.307051243200255703e+04
2.602516580326667963e+02 -6.752827934503707979e-15 4.810599839682529189e-02 -1.496451783412710429e-02 -1.839609713452379502e-02 -1.091134563753968535e-03 2.077571597492063049e-03 6.492220531746040147e-04 1.000000000000000000e+00 2.290803800000000194e+02 5.247782050094440638e+04 1.202163906192813627e+07 0.000000000000000000e+00 0.000000000000000000e+00 5.098383834415919409e+04
2.592875295662722124e+02 -5.123726758011432889e-14 5.266199139682528618e-02 -3.588364363412710478e-02 -2.446927813452379197e-02 -1.311484563753968768e-03 2.101589897492062934e-03 4.946715531746040178e-04 1.000000000000000000e+00 2.311076400000000035e+02 5.341074126636960136e+04 1.234363036472129077e+07 0.000000000000000000e+00 0.000000000000000000e+00 5.981151817240004311e+04
2.592308820611373790e+02 -4.564269168653986185e-14 6.118641339682527602e-02 -3.072892273412710512e-02 -5.481309213452380258e-02 -1.255843163753968695e-03 2.399166697492063124e-03 8.114795531746040380e-04 1.000000000000000000e+00 2.331349000000000160e+02 5.435188159801000438e+04 1.267132048116390407e+07 0.000000000000000000e+00 0.000000000000000000e+00 6.960354151014162926e+04
2.595619725021097111e+02 -7.003595564454634878e-15 7.061374139682527473e-02 -2.235936743412710384e-02 -9.913570134523798372e-03 -1.268399263753968668e-03 2.261032397492063195e-03 8.009784531746040040e-04 1.000000000000000000e+00 2.351621600000000001e+02 5.530124149586560088e+04 1.300475940084938519e+07 0.000000000000000000e+00 0.000000000000000000e+00 8.040989795079996111e+04
2.590598186666346123e+02 6.860929725324630356e-16 7.201945939682527498e-02 -3.425773313412710380e-02 -2.546723413452380014e-02 -1.075733563753968579e-03 2.168366397492063266e-03 7.489506531746041021e-04 1.000000000000000000e+00 2.371894200000000126e+02 5.625882095993640542e+04 1.334399711337115988e+07 0.000000000000000000e+00 0.000000000000000000e+00 9.228057708779163659e+04
2.587420925533573950e+02 1.294219066619279127e-14 7.647773339682528704e-02 -3.709144563412710566e-02 -2.783978113452380276e-02 -1.551370363753968619e-03 2.241157097492063165e-03 8.262895531746041215e-04 1.000000000000000000e+00 2.392166799999999967e+02 5.722461999022239615e+04 1.368908360832263529e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.052655685145325697e+05
2.593739234782133281e+02 2.489479921874677366e-14 8.438259339682528670e-02 -4.554863663412710151e-02 -2.931516913452379691e-02 -1.568884063753968675e-03 2.324322897492063066e-03 1.015189053174604152e-03 1.000000000000000000e+00 2.412439400000000091e+02 5.819863858672360220e+04 1.404006887529723532e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.194148618244394165e+05
2.595694737699126904e+02 1.555171091878242343e-14 1.248369423968252873e-01 -2.113519634127106195e-03 -3.012072513452379585e-02 -1.548244163753968554e-03 1.854907397492062959e-03 1.469042453174604010e-03 1.000000000000000000e+00 2.432712000000000216e+02 5.918087674944000901e+04 1.439700290388837270e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.347784466109283094e+05
2.589335042145449961e+02 2.528064545633945617e-14 1.386098423968253057e-01 -1.554300763412710430e-02 -4.327043134523797518e-03 -1.380193763753968534e-03 1.668424897492063255e-03 1.876337353174604202e-03 1.000000000000000000e+00 2.452984600000000057e+02 6.017133447837160202e+04 1.475993568368945830e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.514063124674152350e+05
2.585203309350814038e+02 1.708358532944420453e-14 1.239822423968252735e-01 -5.095475363412710346e-02 2.847714586547619997e-02 -1.985254663753968554e-03 1.673778397492063034e-03 1.607149453174604170e-03 1.000000000000000000e+00 2.473257200000000182e+02 6.117001177351841034e+04 1.512891720429391786e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.693484489873169805e+05
2.577332151445505701e+02 -1.306066815792015049e-14 1.361771723968253078e-01 -4.787148863412710176e-02 4.220229221547620174e-02 -1.763331063753968610e-03 1.475171397492063084e-03 1.748681753174604041e-03 1.000000000000000000e+00 2.493529800000000023e+02 6.217690863488039759e+04 1.550399745529516041e+07 0.000000000000000000e+00 0.000000000000000000e+00 1.886548457640492998e+05
2.560893166165346315e+02 -2.617243662451785409e-14 1.327212623968253014e-01 -5.556048863412710315e-02 3.156961486547620044e-02 -1.981807663753968815e-03 1.528626197492063011e-03 1.673368953174604080e-03 1.000000000000000000e+00 2.513802400000000148e+02 6.319202506245760742e+04 1.588522642628660984e+07 0.000000000000000000e+00 0.000000000000000000e+00 2.093754923910290236e+05
2.583923782750806595e+02 -3.599677893875522221e-14 1.378797223968253050e-01 -4.097437163412710748e-02 4.345609296547620071e-02 -1.820374863753968561e-03 1.497760797492063162e-03 1.892659753174604376e-03 1.000000000000000000e+00 2.534074999999999989e+02 6.421536105624999618e+04 1.627265410686167143e+07 0.000000000000000000e+00 0.000000000000000000e+00 2.315603784616718476e+05
'''
