"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenInjectableDockerBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
/**
 * A CDK construct to build and push Docker images to an ECR repository using
 * CodeBuild and Lambda custom resources, **then** retrieve the final image tag
 * so that ECS/Lambda references use the exact digest.
 */
class TokenInjectableDockerBuilder extends constructs_1.Construct {
    /**
     * Creates a new `TokenInjectableDockerBuilder`.
     *
     * @param scope The scope in which to define this construct.
     * @param id The scoped construct ID.
     * @param props Configuration for building and pushing the Docker image.
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { path: sourcePath, buildArgs, dockerLoginSecretArn, vpc, securityGroups, subnetSelection, installCommands, preBuildCommands, } = props;
        // Generate an ephemeral tag for CodeBuild
        const imageTag = crypto.randomUUID();
        // Define a KMS key for ECR encryption
        const encryptionKey = new aws_kms_1.Key(this, 'EcrEncryptionKey', {
            enableKeyRotation: true,
        });
        // Create an ECR repository with encryption, lifecycle rules, and image scanning
        this.ecrRepository = new aws_ecr_1.Repository(this, 'ECRRepository', {
            lifecycleRules: [
                {
                    rulePriority: 1,
                    description: 'Remove untagged images after 1 day',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            encryption: aws_ecr_1.RepositoryEncryption.KMS,
            encryptionKey,
            imageScanOnPush: true,
        });
        // Wrap the source folder as an S3 asset for CodeBuild to use
        const sourceAsset = new aws_s3_assets_1.Asset(this, 'SourceAsset', {
            path: sourcePath,
        });
        // Create an S3 bucket to store the CodeBuild artifacts
        const artifactBucket = new aws_s3_1.Bucket(this, 'ArtifactBucket', {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
        });
        // Convert buildArgs to a CLI-friendly string
        const buildArgsString = buildArgs
            ? Object.entries(buildArgs)
                .map(([k, v]) => `--build-arg ${k}=${v}`)
                .join(' ')
            : '';
        // Optional DockerHub login, if a secret ARN is provided
        const dockerLoginCommands = dockerLoginSecretArn
            ? [
                'echo "Retrieving Docker credentials..."',
                'apt-get update -y && apt-get install -y jq',
                `DOCKER_USERNAME=$(aws secretsmanager get-secret-value --secret-id ${dockerLoginSecretArn} --query SecretString --output text | jq -r .username)`,
                `DOCKER_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${dockerLoginSecretArn} --query SecretString --output text | jq -r .password)`,
                'echo "Logging in to Docker Hub..."',
                'echo $DOCKER_PASSWORD | docker login --username $DOCKER_USERNAME --password-stdin',
            ]
            : ['echo "No Docker credentials. Skipping Docker Hub login."'];
        const buildSpecObj = {
            version: '0.2',
            phases: {
                install: {
                    commands: [
                        'echo "Beginning install phase..."',
                        ...(installCommands ?? []),
                    ],
                },
                pre_build: {
                    commands: [
                        ...(preBuildCommands ?? []),
                        ...dockerLoginCommands,
                        'echo "Retrieving AWS Account ID..."',
                        'export ACCOUNT_ID=$(aws sts get-caller-identity --query Account --output text)',
                        'echo "Logging into Amazon ECR..."',
                        'aws ecr get-login-password --region $AWS_DEFAULT_REGION | docker login --username AWS --password-stdin $ACCOUNT_ID.dkr.ecr.$AWS_DEFAULT_REGION.amazonaws.com',
                    ],
                },
                build: {
                    commands: [
                        `echo "Building Docker image with tag ${imageTag}..."`,
                        `docker build ${buildArgsString} -t $ECR_REPO_URI:${imageTag} $CODEBUILD_SRC_DIR`,
                    ],
                },
                post_build: {
                    commands: [
                        `echo "Pushing Docker image with tag ${imageTag}..."`,
                        `docker push $ECR_REPO_URI:${imageTag}`,
                    ],
                },
            },
        };
        // Create the CodeBuild project
        const codeBuildProject = new aws_codebuild_1.Project(this, 'CodeBuildProject', {
            source: aws_codebuild_1.Source.s3({
                bucket: sourceAsset.bucket,
                path: sourceAsset.s3ObjectKey,
            }),
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_7_0,
                privileged: true,
            },
            environmentVariables: {
                ECR_REPO_URI: { value: this.ecrRepository.repositoryUri },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject(buildSpecObj),
            vpc,
            securityGroups,
            subnetSelection,
        });
        // Grant CodeBuild the ability to interact with ECR
        this.ecrRepository.grantPullPush(codeBuildProject);
        codeBuildProject.role?.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ecr:GetAuthorizationToken',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchCheckLayerAvailability',
            ],
            resources: [this.ecrRepository.repositoryArn],
        }));
        if (dockerLoginSecretArn) {
            codeBuildProject.role?.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: [dockerLoginSecretArn],
            }));
        }
        encryptionKey.grantEncryptDecrypt(codeBuildProject.role);
        // Define the Lambda functions for custom resource event and completion handling
        const onEventHandlerFunction = new aws_lambda_1.Function(this, 'OnEventHandlerFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.resolve(__dirname, '../onEvent')),
            handler: 'onEvent.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(15),
        });
        onEventHandlerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['codebuild:StartBuild'],
            resources: [codeBuildProject.projectArn],
        }));
        const isCompleteHandlerFunction = new aws_lambda_1.Function(this, 'IsCompleteHandlerFunction', {
            runtime: aws_lambda_1.Runtime.NODEJS_18_X,
            code: aws_lambda_1.Code.fromAsset(path.resolve(__dirname, '../isComplete')),
            environment: {
                IMAGE_TAG: imageTag,
            },
            handler: 'isComplete.handler',
            timeout: aws_cdk_lib_1.Duration.minutes(15),
        });
        isCompleteHandlerFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:ListBuildsForProject',
                'logs:GetLogEvents',
                'logs:DescribeLogStreams',
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        artifactBucket.grantReadWrite(isCompleteHandlerFunction);
        encryptionKey.grantEncryptDecrypt(onEventHandlerFunction);
        encryptionKey.grantEncryptDecrypt(isCompleteHandlerFunction);
        this.ecrRepository.grantPullPush(onEventHandlerFunction);
        this.ecrRepository.grantPullPush(isCompleteHandlerFunction);
        // Create a custom resource provider that uses the above Lambdas
        const provider = new custom_resources_1.Provider(this, 'CustomResourceProvider', {
            onEventHandler: onEventHandlerFunction,
            isCompleteHandler: isCompleteHandlerFunction,
            queryInterval: aws_cdk_lib_1.Duration.seconds(30),
        });
        // Custom Resource that triggers the CodeBuild and waits for completion
        const buildTriggerResource = new aws_cdk_lib_1.CustomResource(this, 'BuildTriggerResource', {
            serviceToken: provider.serviceToken,
            properties: {
                ProjectName: codeBuildProject.projectName,
                ImageTag: imageTag,
                Trigger: crypto.randomUUID(),
            },
        });
        buildTriggerResource.node.addDependency(codeBuildProject);
        // Retrieve the final Docker image tag from Data.ImageTag
        // This creates a dependency on the Custom Resource...
        const imageTagRef = buildTriggerResource.getAttString('ImageTag');
        this.containerImage = aws_ecs_1.ContainerImage.fromEcrRepository(this.ecrRepository, imageTagRef);
        this.dockerImageCode = aws_lambda_1.DockerImageCode.fromEcr(this.ecrRepository, {
            tagOrDigest: imageTagRef,
        });
    }
}
exports.TokenInjectableDockerBuilder = TokenInjectableDockerBuilder;
_a = JSII_RTTI_SYMBOL_1;
TokenInjectableDockerBuilder[_a] = { fqn: "token-injectable-docker-builder.TokenInjectableDockerBuilder", version: "1.3.2" };
//# sourceMappingURL=data:application/json;base64,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