# SPDX-FileCopyrightText: 2025 M5Stack Technology CO LTD
#
# SPDX-License-Identifier: MIT

from . import reg
import time
from collections import namedtuple

_coeff_div = namedtuple(
    "_coeff_div",
    [
        "mclk",
        "rate",
        "pre_div",
        "pre_multi",
        "adc_div",
        "dac_div",
        "fs_mode",
        "lrck_h",
        "lrck_l",
        "bclk_div",
        "adc_osr",
        "dac_osr",
    ],
)

coeff_div = (
    # 8k
    _coeff_div(12288000, 8000, 0x06, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 8000, 0x03, 0x01, 0x03, 0x03, 0x00, 0x05, 0xFF, 0x18, 0x10, 0x10),
    _coeff_div(16384000, 8000, 0x08, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(8192000, 8000, 0x04, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 8000, 0x03, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(4096000, 8000, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 8000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2048000, 8000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 8000, 0x03, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1024000, 8000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 11.025k
    _coeff_div(11289600, 11025, 0x04, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(5644800, 11025, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2822400, 11025, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1411200, 11025, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 12k
    _coeff_div(12288000, 12000, 0x04, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 12000, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 12000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 12000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 16k
    _coeff_div(12288000, 16000, 0x03, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 16000, 0x03, 0x01, 0x03, 0x03, 0x00, 0x02, 0xFF, 0x0C, 0x10, 0x10),
    _coeff_div(16384000, 16000, 0x04, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(8192000, 16000, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 16000, 0x03, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(4096000, 16000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 16000, 0x03, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2048000, 16000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 16000, 0x03, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1024000, 16000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 22.05k
    _coeff_div(11289600, 22050, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(5644800, 22050, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2822400, 22050, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1411200, 22050, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(705600, 22050, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 24k
    _coeff_div(12288000, 24000, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 24000, 0x03, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 24000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 24000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 24000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 32k
    _coeff_div(12288000, 32000, 0x03, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 32000, 0x03, 0x02, 0x03, 0x03, 0x00, 0x02, 0xFF, 0x0C, 0x10, 0x10),
    _coeff_div(16384000, 32000, 0x02, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(8192000, 32000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 32000, 0x03, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(4096000, 32000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 32000, 0x03, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2048000, 32000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 32000, 0x03, 0x03, 0x01, 0x01, 0x01, 0x00, 0x7F, 0x02, 0x10, 0x10),
    _coeff_div(1024000, 32000, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 44.1k
    _coeff_div(11289600, 44100, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(5644800, 44100, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2822400, 44100, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1411200, 44100, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 48k
    _coeff_div(12288000, 48000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 48000, 0x03, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 48000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 48000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 48000, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    # 64k
    _coeff_div(12288000, 64000, 0x03, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 64000, 0x03, 0x02, 0x03, 0x03, 0x01, 0x01, 0x7F, 0x06, 0x10, 0x10),
    _coeff_div(16384000, 64000, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(8192000, 64000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 64000, 0x01, 0x02, 0x03, 0x03, 0x01, 0x01, 0x7F, 0x06, 0x10, 0x10),
    _coeff_div(4096000, 64000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 64000, 0x01, 0x03, 0x03, 0x03, 0x01, 0x01, 0x7F, 0x06, 0x10, 0x10),
    _coeff_div(2048000, 64000, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 64000, 0x01, 0x03, 0x01, 0x01, 0x01, 0x00, 0xBF, 0x03, 0x18, 0x18),
    _coeff_div(1024000, 64000, 0x01, 0x03, 0x01, 0x01, 0x01, 0x00, 0x7F, 0x02, 0x10, 0x10),
    # 88.2k
    _coeff_div(11289600, 88200, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(5644800, 88200, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(2822400, 88200, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1411200, 88200, 0x01, 0x03, 0x01, 0x01, 0x01, 0x00, 0x7F, 0x02, 0x10, 0x10),
    # 96k
    _coeff_div(12288000, 96000, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(18432000, 96000, 0x03, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(6144000, 96000, 0x01, 0x02, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(3072000, 96000, 0x01, 0x03, 0x01, 0x01, 0x00, 0x00, 0xFF, 0x04, 0x10, 0x10),
    _coeff_div(1536000, 96000, 0x01, 0x03, 0x01, 0x01, 0x01, 0x00, 0x7F, 0x02, 0x10, 0x10),
)


def BIT(nr):  # noqa: N802
    return 1 << nr


def get_coeff(mclk: int, rate: int):
    for i in range(len(coeff_div)):
        if coeff_div[i].rate == rate and coeff_div[i].mclk == mclk:
            return i
    return -1


class es8311_clock_config_t:
    mclk_inverted: bool
    sclk_inverted: bool
    mclk_from_mclk_pin: int
    mclk_frequency: int
    sample_frequency: int


class ES8311:
    ES8311_RESOLUTION_16 = 16
    ES8311_RESOLUTION_18 = 18
    ES8311_RESOLUTION_20 = 20
    ES8311_RESOLUTION_24 = 24
    ES8311_RESOLUTION_32 = 32

    res_map = {
        ES8311_RESOLUTION_16: (3 << 2),
        ES8311_RESOLUTION_18: (2 << 2),
        ES8311_RESOLUTION_20: (1 << 2),
        ES8311_RESOLUTION_24: (0 << 2),
        ES8311_RESOLUTION_32: (4 << 2),
    }

    def __init__(self, i2c, address: int = 0x18):
        self._i2c = i2c
        self._address = address

    def init(self, clk_cfg, res_in, res_out):
        self._i2c.writeto_mem(self._address, reg.ES8311_RESET_REG00, b"\x1f")
        time.sleep(0.02)
        self._i2c.writeto_mem(self._address, reg.ES8311_RESET_REG00, b"\x00")
        self._i2c.writeto_mem(self._address, reg.ES8311_RESET_REG00, b"\x80")

        self.clock_config(clk_cfg, res_out)

        self.fmt_config(res_in, res_out)

        self._i2c.writeto_mem(self._address, reg.ES8311_SYSTEM_REG0D, b"\x01")
        self._i2c.writeto_mem(self._address, reg.ES8311_SYSTEM_REG0E, b"\x02")
        self._i2c.writeto_mem(self._address, reg.ES8311_SYSTEM_REG12, b"\x00")
        self._i2c.writeto_mem(self._address, reg.ES8311_SYSTEM_REG13, b"\x10")
        self._i2c.writeto_mem(self._address, reg.ES8311_ADC_REG1C, b"\x6a")
        self._i2c.writeto_mem(self._address, reg.ES8311_DAC_REG37, b"\x08")

    def clock_config(self, clk_cfg: es8311_clock_config_t, res):
        reg06 = 0
        reg01 = 0x3F
        mclk_hz = 0

        if clk_cfg.mclk_from_mclk_pin:
            mclk_hz = clk_cfg.mclk_frequency
        else:
            mclk_hz = clk_cfg.sample_frequency * res * 2
            reg01 |= BIT(7)  # Select BCLK (a.k.a. SCK) pin

        if clk_cfg.mclk_inverted:
            reg01 |= BIT(6)  # Invert MCLK pin

        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG01, reg01.to_bytes(1, "big")
        )

        reg06 = self._i2c.readfrom_mem(self._address, reg.ES8311_CLK_MANAGER_REG06, 1)[0]
        if clk_cfg.sclk_inverted:
            reg06 |= BIT(5)
        else:
            reg06 &= ~BIT(5)

        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG06, reg06.to_bytes(1, "big")
        )

        self.sample_frequency_config(mclk_hz, clk_cfg.sample_frequency)

    def fmt_config(self, res_in, res_out):
        reg09 = 0
        reg0a = 0

        reg00 = self._i2c.readfrom_mem(self._address, reg.ES8311_RESET_REG00, 1)[0]
        reg00 &= 0xBF
        self._i2c.writeto_mem(self._address, reg.ES8311_RESET_REG00, reg00.to_bytes(1, "big"))

        reg09 = self.resolution_config(res_in, reg09)
        reg0a = self.resolution_config(res_out, reg0a)
        self._i2c.writeto_mem(self._address, reg.ES8311_SDPIN_REG09, reg09.to_bytes(1, "big"))
        self._i2c.writeto_mem(self._address, reg.ES8311_SDPOUT_REG0A, reg0a.to_bytes(1, "big"))

    def sample_frequency_config(self, mclk_frequency: int, sample_frequency: int):
        regv = 0

        # /* Get clock coefficients from coefficient table */
        coeff = get_coeff(mclk_frequency, sample_frequency)

        if coeff < 0:
            print(
                "Unable to configure sample rate {}Hz with {}Hz MCLK".format(
                    sample_frequency, mclk_frequency
                )
            )
            return

        selected_coeff = coeff_div[coeff]

        # /* register 0x02 */
        regv = self._i2c.readfrom_mem(self._address, reg.ES8311_CLK_MANAGER_REG02, 1)[0]
        regv &= 0x07
        regv |= (selected_coeff.pre_div - 1) << 5
        regv |= selected_coeff.pre_multi << 3
        self._i2c.writeto_mem(self._address, reg.ES8311_CLK_MANAGER_REG02, regv.to_bytes(1, "big"))

        # /* register 0x03 */
        reg03 = (selected_coeff.fs_mode << 6) | selected_coeff.adc_osr
        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG03, reg03.to_bytes(1, "big")
        )

        # /* register 0x04 */
        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG04, selected_coeff.dac_osr.to_bytes(1, "big")
        )

        # /* register 0x05 */
        reg05 = ((selected_coeff.adc_div - 1) << 4) | (selected_coeff.dac_div - 1)
        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG05, reg05.to_bytes(1, "big")
        )

        # /* register 0x06 */
        regv = self._i2c.readfrom_mem(self._address, reg.ES8311_CLK_MANAGER_REG06, 1)[0]
        regv &= 0xE0

        if selected_coeff.bclk_div < 19:
            regv |= (selected_coeff.bclk_div - 1) << 0
        else:
            regv |= (selected_coeff.bclk_div) << 0

        self._i2c.writeto_mem(self._address, reg.ES8311_CLK_MANAGER_REG06, regv.to_bytes(1, "big"))

        # /* register 0x07 */
        regv = self._i2c.readfrom_mem(self._address, reg.ES8311_CLK_MANAGER_REG07, 1)[0]
        regv &= 0xC0
        regv |= selected_coeff.lrck_h << 0
        self._i2c.writeto_mem(self._address, reg.ES8311_CLK_MANAGER_REG07, regv.to_bytes(1, "big"))

        # /* register 0x08 */
        self._i2c.writeto_mem(
            self._address, reg.ES8311_CLK_MANAGER_REG08, selected_coeff.lrck_l.to_bytes(1, "big")
        )

        return

    def voice_volume_set(self, volume: int):
        if volume < 0:
            volume = 0
        elif volume > 100:
            volume = 100

        reg32 = 0
        if volume == 0:
            reg32 = 0
        else:
            reg32 = int((volume) * 256 / 100) - 1

        self._i2c.writeto_mem(self._address, reg.ES8311_DAC_REG32, reg32.to_bytes(1, "big"))

    def microphone_config(self, digital_mic: bool):
        reg14 = 0x1A  # enable analog MIC and max PGA gain

        # PDM digital microphone enable or disable
        if digital_mic:
            reg14 |= BIT(6)
        self._i2c.writeto_mem(
            self._address, reg.ES8311_ADC_REG17, b"\xff"
        )  # Set ADC gain @todo move this to ADC config section

        self._i2c.writeto_mem(self._address, reg.ES8311_SYSTEM_REG14, reg14.to_bytes(1, "big"))

    def resolution_config(self, res, reg):
        return reg | self.res_map.get(res)
