"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionsBoundary = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const iam_generated_1 = require("./iam.generated");
/**
 * Modify the Permissions Boundaries of Users and Roles in a construct tree
 *
 * ```ts
 * const policy = iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess');
 * iam.PermissionsBoundary.of(this).apply(policy);
 * ```
 */
class PermissionsBoundary {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Access the Permissions Boundaries of a construct tree
     */
    static of(scope) {
        return new PermissionsBoundary(scope);
    }
    /**
     * Apply the given policy as Permissions Boundary to all Roles and Users in
     * the scope.
     *
     * Will override any Permissions Boundaries configured previously; in case
     * a Permission Boundary is applied in multiple scopes, the Boundary applied
     * closest to the Role wins.
     */
    apply(boundaryPolicy) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IManagedPolicy(boundaryPolicy);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.apply);
            }
            throw error;
        }
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyOverride('PermissionsBoundary', boundaryPolicy.managedPolicyArn);
                }
            },
        });
    }
    /**
     * Remove previously applied Permissions Boundaries
     */
    clear() {
        constructs_1.Node.of(this.scope).applyAspect({
            visit(node) {
                if (core_1.CfnResource.isCfnResource(node) &&
                    (node.cfnResourceType == iam_generated_1.CfnRole.CFN_RESOURCE_TYPE_NAME || node.cfnResourceType == iam_generated_1.CfnUser.CFN_RESOURCE_TYPE_NAME)) {
                    node.addPropertyDeletionOverride('PermissionsBoundary');
                }
            },
        });
    }
}
exports.PermissionsBoundary = PermissionsBoundary;
_a = JSII_RTTI_SYMBOL_1;
PermissionsBoundary[_a] = { fqn: "@aws-cdk/aws-iam.PermissionsBoundary", version: "1.184.1" };
//# sourceMappingURL=data:application/json;base64,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