"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const kms = require("aws-cdk-lib/aws-kms");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const construct = new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    // By default, we don't create a VPC
    expect(construct.vpc).toBeUndefined();
});
test('New VPC is created when deployVpc flag is true', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        deployVpc: true
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC created by the construct
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Isolated");
});
test('Existing VPC is used when specified', () => {
    const stack = new cdk.Stack();
    const existingVpc = new ec2.Vpc(stack, 'test-vpc', { vpcName: 'my-vpc-name' });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingVpc
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC by its name
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Private");
});
test('New VPC is created from user-provided vpcProps', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        vpcProps: {
            vpcName: 'my-vpc-name'
        },
        deployVpc: true
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get the VPC by its name
    template.resourceCountIs('AWS::EC2::VPC', 1);
    const vpcResource = template.findResources('AWS::EC2::VPC');
    const [vpcId] = Object.keys(vpcResource);
    verifyLambdaFunctionVpcProps(template, vpcId, "Isolated");
});
test('Lambda Function has default stream environment variable', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                KINESIS_DATASTREAM_NAME: {
                    Ref: kinesisStreamId
                }
            }
        }
    });
});
test('Lambda Function stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        streamEnvironmentVariableName: 'CUSTOM_ENV_VAR_NAME'
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                CUSTOM_ENV_VAR_NAME: {
                    Ref: kinesisStreamId
                }
            }
        }
    });
});
test('Kinesis Stream is encrypted with AWS-managed CMK by default', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
});
test('CloudWatch Alarms are created for Kinesis Stream by default', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'GetRecords.IteratorAgeMilliseconds'
    });
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'ReadProvisionedThroughputExceeded'
    });
});
test('CloudWatch Alarms are not created when createCloudWatchAlarms property is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        createCloudWatchAlarms: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::CloudWatch::Alarm', 0);
});
test('Error is thrown when vpc is specified and existing lambda function is not associated with it', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'test-lamba', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler'
    });
    const existingVpc = new ec2.Vpc(stack, 'test-vpc', {});
    const app = () => {
        new lib_1.LambdaToKinesisStreams(stack, 'lambda-to-sqs-stack', {
            existingLambdaObj,
            existingVpc
        });
    };
    expect(app).toThrowError();
});
test('Construct uses existing Lambda Function', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingLambdaObj = new lambda.Function(stack, 'test-lamba', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_18_X,
        handler: 'index.handler',
        functionName: 'my-lambda-function'
    });
    new lib_1.LambdaToKinesisStreams(stack, 'lambda-to-sqs-stack', {
        existingLambdaObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'my-lambda-function'
    });
});
test('Construct uses existing Kinesis Stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct uses unencrypted existing stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    defaults.expectNonexistence(stack, 'AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct uses unencrypted streams from stream props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        kinesisStreamProps: {
            streamName: 'my-stream',
            encryption: kinesis.StreamEncryption.UNENCRYPTED
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    defaults.expectNonexistence(stack, 'AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct grants PutRecord permission for the Lambda Function to write to the Kinesis Stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords'
                    ],
                    Effect: 'Allow',
                    Resource: '*'
                },
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kinesisStreamId,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
test('When a Customer-manged CMK is used on an existing stream, construct grants the Lambda Function permission to use the encryption key so it can publish messages to it', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        description: 'my-key-description'
    });
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
        encryptionKey
    });
    new lib_1.LambdaToKinesisStreams(stack, 'test-lambda-kinesisstreams', {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_18_X,
            handler: 'index.handler'
        },
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    const resource = template.findResources('AWS::KMS::Key', {
        Properties: {
            Description: assertions_1.Match.exact('my-key-description')
        }
    });
    const [kmsKey] = Object.keys(resource);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'xray:PutTraceSegments',
                        'xray:PutTelemetryRecords'
                    ],
                    Effect: 'Allow',
                    Resource: '*'
                },
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'teststream04374A09',
                            'Arn'
                        ]
                    }
                },
                {
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kmsKey,
                            'Arn'
                        ]
                    }
                },
                {
                    Action: [
                        'kms:Decrypt',
                        'kms:GenerateDataKey*'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kmsKey,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
function verifyLambdaFunctionVpcProps(template, vpcId, subnetType) {
    // Get the Security Group associated with the VPC
    const securityGroupResource = template.findResources('AWS::EC2::SecurityGroup', {
        Properties: {
            VpcId: {
                Ref: assertions_1.Match.exact(vpcId)
            }
        }
    });
    const [securityGroupId] = Object.keys(securityGroupResource);
    // Get the Private or Isolated Subnets associated with the VPC
    const subnetResources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: subnetType
                }
            ])
        }
    });
    const [subnet1, subnet2] = Object.keys(subnetResources);
    // Verify the Lambda Function has the same Security Group
    template.hasResourceProperties('AWS::Lambda::Function', {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    'Fn::GetAtt': [
                        securityGroupId,
                        'GroupId'
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: subnet1
                },
                {
                    Ref: subnet2
                }
            ]
        }
    });
    // Verify the VPC has an interface endpoint for Kinesis Streams
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-streams'
                ]
            ]
        },
        VpcId: {
            Ref: vpcId
        },
    });
    // Verify the VPC has dns hostnames and support enabled
    template.hasResourceProperties('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
}
//# sourceMappingURL=data:application/json;base64,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