"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Account = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [account](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsaccountmanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Account extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept the process to update the primary email address of an account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEmailTargetDomain()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_AcceptPrimaryEmailUpdate.html
     */
    toAcceptPrimaryEmailUpdate() {
        return this.to('AcceptPrimaryEmailUpdate');
    }
    /**
     * Grants permission to close an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_account-permissions-ref.html
     */
    toCloseAccount() {
        return this.to('CloseAccount');
    }
    /**
     * Grants permission to delete the alternate contacts for an account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAlternateContactTypes()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_DeleteAlternateContact.html
     */
    toDeleteAlternateContact() {
        return this.to('DeleteAlternateContact');
    }
    /**
     * Grants permission to disable use of a Region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_DisableRegion.html
     */
    toDisableRegion() {
        return this.to('DisableRegion');
    }
    /**
     * Grants permission to enable use of a Region
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_EnableRegion.html
     */
    toEnableRegion() {
        return this.to('EnableRegion');
    }
    /**
     * Grants permission to retrieve the account information for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_GetAccountInformation.html
     */
    toGetAccountInformation() {
        return this.to('GetAccountInformation');
    }
    /**
     * Grants permission to retrieve the alternate contacts for an account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAlternateContactTypes()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_GetAlternateContact.html
     */
    toGetAlternateContact() {
        return this.to('GetAlternateContact');
    }
    /**
     * Grants permission to retrieve the primary contact information for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_GetContactInformation.html
     */
    toGetContactInformation() {
        return this.to('GetContactInformation');
    }
    /**
     * Grants permission to retrieve the primary email address of an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_GetPrimaryEmail.html
     */
    toGetPrimaryEmail() {
        return this.to('GetPrimaryEmail');
    }
    /**
     * Grants permission to get the opt-in status of a Region
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_GetRegionOptStatus.html
     */
    toGetRegionOptStatus() {
        return this.to('GetRegionOptStatus');
    }
    /**
     * Grants permission to list the available Regions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_ListRegions.html
     */
    toListRegions() {
        return this.to('ListRegions');
    }
    /**
     * Grants permission to update the name for an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_PutAccountName.html
     */
    toPutAccountName() {
        return this.to('PutAccountName');
    }
    /**
     * Grants permission to modify the alternate contacts for an account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAlternateContactTypes()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_PutAlternateContact.html
     */
    toPutAlternateContact() {
        return this.to('PutAlternateContact');
    }
    /**
     * Grants permission to update the primary contact information for an account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_PutContactInformation.html
     */
    toPutContactInformation() {
        return this.to('PutContactInformation');
    }
    /**
     * Grants permission to start the process to update the primary email address of an account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEmailTargetDomain()
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/API_StartPrimaryEmailUpdate.html
     */
    toStartPrimaryEmailUpdate() {
        return this.to('StartPrimaryEmailUpdate');
    }
    /**
     * Adds a resource of type account to the statement
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:account::${account ?? this.defaultAccount}:account`);
    }
    /**
     * Adds a resource of type accountInOrganization to the statement
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-resources
     *
     * @param managementAccountId - Identifier for the managementAccountId.
     * @param organizationId - Identifier for the organizationId.
     * @param memberAccountId - Identifier for the memberAccountId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccountInOrganization(managementAccountId, organizationId, memberAccountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:account::${managementAccountId}:account/o-${organizationId}/${memberAccountId}`);
    }
    /**
     * Filters access by the resource path for an account in an organization
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountResourceOrgPaths(value, operator) {
        return this.if(`AccountResourceOrgPaths`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by resource tags for an account in an organization
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountResourceOrgTags(tagKey, value, operator) {
        return this.if(`AccountResourceOrgTags/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by alternate contact types
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toDeleteAlternateContact()
     * - .toGetAlternateContact()
     * - .toPutAlternateContact()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAlternateContactTypes(value, operator) {
        return this.if(`AlternateContactTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by email domain of the target email address
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toAcceptPrimaryEmailUpdate()
     * - .toStartPrimaryEmailUpdate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEmailTargetDomain(value, operator) {
        return this.if(`EmailTargetDomain`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of Regions. Enables or disables all the Regions specified here
     *
     * https://docs.aws.amazon.com/accounts/latest/reference/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toDisableRegion()
     * - .toEnableRegion()
     * - .toGetRegionOptStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetRegion(value, operator) {
        return this.if(`TargetRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [account](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsaccountmanagement.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'account';
        this.accessLevelList = {
            Write: [
                'AcceptPrimaryEmailUpdate',
                'CloseAccount',
                'DeleteAlternateContact',
                'DisableRegion',
                'EnableRegion',
                'PutAccountName',
                'PutAlternateContact',
                'PutContactInformation',
                'StartPrimaryEmailUpdate'
            ],
            Read: [
                'GetAccountInformation',
                'GetAlternateContact',
                'GetContactInformation',
                'GetPrimaryEmail',
                'GetRegionOptStatus'
            ],
            List: [
                'ListRegions'
            ]
        };
    }
}
exports.Account = Account;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWNjb3VudG1hbmFnZW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhY2NvdW50bWFuYWdlbWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsT0FBUSxTQUFRLHdCQUFlO0lBRzFDOzs7Ozs7Ozs7T0FTRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBMEJEOzs7Ozs7O09BT0c7SUFDSSxTQUFTLENBQUMsT0FBZ0IsRUFBRSxTQUFrQjtRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixhQUFjLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxVQUFVLENBQUMsQ0FBQztJQUNySCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCLENBQUMsbUJBQTJCLEVBQUUsY0FBc0IsRUFBRSxlQUF1QixFQUFFLFNBQWtCO1FBQzdILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGFBQWMsbUJBQW9CLGNBQWUsY0FBZSxJQUFLLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ3JKLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0kseUJBQXlCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNyRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM3RSxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSx3QkFBd0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUNwRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTJCLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDeEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLHVCQUF1QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksbUJBQW1CLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMvRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksY0FBYyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDMUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQWpVUixrQkFBYSxHQUFHLFNBQVMsQ0FBQztRQStMdkIsb0JBQWUsR0FBb0I7WUFDM0MsS0FBSyxFQUFFO2dCQUNMLDBCQUEwQjtnQkFDMUIsY0FBYztnQkFDZCx3QkFBd0I7Z0JBQ3hCLGVBQWU7Z0JBQ2YsY0FBYztnQkFDZCxnQkFBZ0I7Z0JBQ2hCLHFCQUFxQjtnQkFDckIsdUJBQXVCO2dCQUN2Qix5QkFBeUI7YUFDMUI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osdUJBQXVCO2dCQUN2QixxQkFBcUI7Z0JBQ3JCLHVCQUF1QjtnQkFDdkIsaUJBQWlCO2dCQUNqQixvQkFBb0I7YUFDckI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osYUFBYTthQUNkO1NBQ0YsQ0FBQztJQTZHRixDQUFDO0NBQ0Y7QUFwVUQsMEJBb1VDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthY2NvdW50XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NhY2NvdW50bWFuYWdlbWVudC5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEFjY291bnQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdhY2NvdW50JztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWNjZXB0IHRoZSBwcm9jZXNzIHRvIHVwZGF0ZSB0aGUgcHJpbWFyeSBlbWFpbCBhZGRyZXNzIG9mIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkVtYWlsVGFyZ2V0RG9tYWluKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjY291bnRzL2xhdGVzdC9yZWZlcmVuY2UvQVBJX0FjY2VwdFByaW1hcnlFbWFpbFVwZGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BY2NlcHRQcmltYXJ5RW1haWxVcGRhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FjY2VwdFByaW1hcnlFbWFpbFVwZGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNsb3NlIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9zZWN1cml0eV9hY2NvdW50LXBlcm1pc3Npb25zLXJlZi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DbG9zZUFjY291bnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Nsb3NlQWNjb3VudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgYWx0ZXJuYXRlIGNvbnRhY3RzIGZvciBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBbHRlcm5hdGVDb250YWN0VHlwZXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9BUElfRGVsZXRlQWx0ZXJuYXRlQ29udGFjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBbHRlcm5hdGVDb250YWN0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVBbHRlcm5hdGVDb250YWN0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYWJsZSB1c2Ugb2YgYSBSZWdpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlRhcmdldFJlZ2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL0FQSV9EaXNhYmxlUmVnaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2FibGVSZWdpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rpc2FibGVSZWdpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgdXNlIG9mIGEgUmVnaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZUYXJnZXRSZWdpb24oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9BUElfRW5hYmxlUmVnaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0VuYWJsZVJlZ2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRW5hYmxlUmVnaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGFjY291bnQgaW5mb3JtYXRpb24gZm9yIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL0FQSV9HZXRBY2NvdW50SW5mb3JtYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QWNjb3VudEluZm9ybWF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRBY2NvdW50SW5mb3JtYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgYWx0ZXJuYXRlIGNvbnRhY3RzIGZvciBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkFsdGVybmF0ZUNvbnRhY3RUeXBlcygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL0FQSV9HZXRBbHRlcm5hdGVDb250YWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEFsdGVybmF0ZUNvbnRhY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEFsdGVybmF0ZUNvbnRhY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgcHJpbWFyeSBjb250YWN0IGluZm9ybWF0aW9uIGZvciBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9BUElfR2V0Q29udGFjdEluZm9ybWF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldENvbnRhY3RJbmZvcm1hdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0Q29udGFjdEluZm9ybWF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIHByaW1hcnkgZW1haWwgYWRkcmVzcyBvZiBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9BUElfR2V0UHJpbWFyeUVtYWlsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFByaW1hcnlFbWFpbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UHJpbWFyeUVtYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IHRoZSBvcHQtaW4gc3RhdHVzIG9mIGEgUmVnaW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlRhcmdldFJlZ2lvbigpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL0FQSV9HZXRSZWdpb25PcHRTdGF0dXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmVnaW9uT3B0U3RhdHVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRSZWdpb25PcHRTdGF0dXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBhdmFpbGFibGUgUmVnaW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjY291bnRzL2xhdGVzdC9yZWZlcmVuY2UvQVBJX0xpc3RSZWdpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZWdpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UmVnaW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgbmFtZSBmb3IgYW4gYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL0FQSV9QdXRBY2NvdW50TmFtZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRBY2NvdW50TmFtZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0QWNjb3VudE5hbWUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBtb2RpZnkgdGhlIGFsdGVybmF0ZSBjb250YWN0cyBmb3IgYW4gYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQWx0ZXJuYXRlQ29udGFjdFR5cGVzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjY291bnRzL2xhdGVzdC9yZWZlcmVuY2UvQVBJX1B1dEFsdGVybmF0ZUNvbnRhY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0QWx0ZXJuYXRlQ29udGFjdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0QWx0ZXJuYXRlQ29udGFjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgcHJpbWFyeSBjb250YWN0IGluZm9ybWF0aW9uIGZvciBhbiBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjY291bnRzL2xhdGVzdC9yZWZlcmVuY2UvQVBJX1B1dENvbnRhY3RJbmZvcm1hdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRDb250YWN0SW5mb3JtYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dENvbnRhY3RJbmZvcm1hdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IHRoZSBwcm9jZXNzIHRvIHVwZGF0ZSB0aGUgcHJpbWFyeSBlbWFpbCBhZGRyZXNzIG9mIGFuIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkVtYWlsVGFyZ2V0RG9tYWluKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjY291bnRzL2xhdGVzdC9yZWZlcmVuY2UvQVBJX1N0YXJ0UHJpbWFyeUVtYWlsVXBkYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0UHJpbWFyeUVtYWlsVXBkYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFByaW1hcnlFbWFpbFVwZGF0ZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQWNjZXB0UHJpbWFyeUVtYWlsVXBkYXRlJyxcbiAgICAgICdDbG9zZUFjY291bnQnLFxuICAgICAgJ0RlbGV0ZUFsdGVybmF0ZUNvbnRhY3QnLFxuICAgICAgJ0Rpc2FibGVSZWdpb24nLFxuICAgICAgJ0VuYWJsZVJlZ2lvbicsXG4gICAgICAnUHV0QWNjb3VudE5hbWUnLFxuICAgICAgJ1B1dEFsdGVybmF0ZUNvbnRhY3QnLFxuICAgICAgJ1B1dENvbnRhY3RJbmZvcm1hdGlvbicsXG4gICAgICAnU3RhcnRQcmltYXJ5RW1haWxVcGRhdGUnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0QWNjb3VudEluZm9ybWF0aW9uJyxcbiAgICAgICdHZXRBbHRlcm5hdGVDb250YWN0JyxcbiAgICAgICdHZXRDb250YWN0SW5mb3JtYXRpb24nLFxuICAgICAgJ0dldFByaW1hcnlFbWFpbCcsXG4gICAgICAnR2V0UmVnaW9uT3B0U3RhdHVzJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0xpc3RSZWdpb25zJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYWNjb3VudCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtcmVzb3VyY2VzXG4gICAqXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkFjY291bnQoYWNjb3VudD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YWNjb3VudDo6JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06YWNjb3VudGApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFjY291bnRJbk9yZ2FuaXphdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtcmVzb3VyY2VzXG4gICAqXG4gICAqIEBwYXJhbSBtYW5hZ2VtZW50QWNjb3VudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIG1hbmFnZW1lbnRBY2NvdW50SWQuXG4gICAqIEBwYXJhbSBvcmdhbml6YXRpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBvcmdhbml6YXRpb25JZC5cbiAgICogQHBhcmFtIG1lbWJlckFjY291bnRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBtZW1iZXJBY2NvdW50SWQuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25BY2NvdW50SW5Pcmdhbml6YXRpb24obWFuYWdlbWVudEFjY291bnRJZDogc3RyaW5nLCBvcmdhbml6YXRpb25JZDogc3RyaW5nLCBtZW1iZXJBY2NvdW50SWQ6IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YWNjb3VudDo6JHsgbWFuYWdlbWVudEFjY291bnRJZCB9OmFjY291bnQvby0keyBvcmdhbml6YXRpb25JZCB9LyR7IG1lbWJlckFjY291bnRJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHJlc291cmNlIHBhdGggZm9yIGFuIGFjY291bnQgaW4gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtY29uZGl0aW9ua2V5c1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkFjY291bnRSZXNvdXJjZU9yZ1BhdGhzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBBY2NvdW50UmVzb3VyY2VPcmdQYXRoc2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHJlc291cmNlIHRhZ3MgZm9yIGFuIGFjY291bnQgaW4gYW4gb3JnYW5pemF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hY2NvdW50cy9sYXRlc3QvcmVmZXJlbmNlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtY29uZGl0aW9ua2V5c1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQWNjb3VudFJlc291cmNlT3JnVGFncyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYEFjY291bnRSZXNvdXJjZU9yZ1RhZ3MvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhbHRlcm5hdGUgY29udGFjdCB0eXBlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLWlkLWJhc2VkLXBvbGljaWVzLWNvbmRpdGlvbmtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0RlbGV0ZUFsdGVybmF0ZUNvbnRhY3QoKVxuICAgKiAtIC50b0dldEFsdGVybmF0ZUNvbnRhY3QoKVxuICAgKiAtIC50b1B1dEFsdGVybmF0ZUNvbnRhY3QoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkFsdGVybmF0ZUNvbnRhY3RUeXBlcyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgQWx0ZXJuYXRlQ29udGFjdFR5cGVzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgZW1haWwgZG9tYWluIG9mIHRoZSB0YXJnZXQgZW1haWwgYWRkcmVzc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLWlkLWJhc2VkLXBvbGljaWVzLWNvbmRpdGlvbmtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0FjY2VwdFByaW1hcnlFbWFpbFVwZGF0ZSgpXG4gICAqIC0gLnRvU3RhcnRQcmltYXJ5RW1haWxVcGRhdGUoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkVtYWlsVGFyZ2V0RG9tYWluKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBFbWFpbFRhcmdldERvbWFpbmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEgbGlzdCBvZiBSZWdpb25zLiBFbmFibGVzIG9yIGRpc2FibGVzIGFsbCB0aGUgUmVnaW9ucyBzcGVjaWZpZWQgaGVyZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNjb3VudHMvbGF0ZXN0L3JlZmVyZW5jZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sI3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLWlkLWJhc2VkLXBvbGljaWVzLWNvbmRpdGlvbmtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0Rpc2FibGVSZWdpb24oKVxuICAgKiAtIC50b0VuYWJsZVJlZ2lvbigpXG4gICAqIC0gLnRvR2V0UmVnaW9uT3B0U3RhdHVzKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZUYXJnZXRSZWdpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFRhcmdldFJlZ2lvbmAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYWNjb3VudF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYWNjb3VudG1hbmFnZW1lbnQuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19