"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Amplifybackend = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [amplifybackend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsamplifyadmin.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Amplifybackend extends shared_1.PolicyStatement {
    /**
     * Grants permission to clone an existing Amplify Admin backend environment into a new Amplify Admin backend enviroment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-environments-backendenvironmentname-clone.html#CloneBackend
     */
    toCloneBackend() {
        return this.to('CloneBackend');
    }
    /**
     * Grants permission to create a new Amplify Admin backend environment by Amplify appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend.html#CreateBackend
     */
    toCreateBackend() {
        return this.to('CreateBackend');
    }
    /**
     * Grants permission to create an API for an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api.html#CreateBackendAPI
     */
    toCreateBackendAPI() {
        return this.to('CreateBackendAPI');
    }
    /**
     * Grants permission to create an auth resource for an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth.html#CreateBackendAuth
     */
    toCreateBackendAuth() {
        return this.to('CreateBackendAuth');
    }
    /**
     * Grants permission to create a new Amplify Admin backend config by Amplify appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-config.html#CreateBackendConfig
     */
    toCreateBackendConfig() {
        return this.to('CreateBackendConfig');
    }
    /**
     * Grants permission to create a backend storage resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#CreateBackendStorage
     */
    toCreateBackendStorage() {
        return this.to('CreateBackendStorage');
    }
    /**
     * Grants permission to create an Amplify Admin challenge token by appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-challenge.html#CreateToken
     */
    toCreateToken() {
        return this.to('CreateToken');
    }
    /**
     * Grants permission to delete an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-environments-backendenvironmentname-remove.html#DeleteBackend
     */
    toDeleteBackend() {
        return this.to('DeleteBackend');
    }
    /**
     * Grants permission to delete an API of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname-remove.html#DeleteBackendAPI
     */
    toDeleteBackendAPI() {
        return this.to('DeleteBackendAPI');
    }
    /**
     * Grants permission to delete an auth resource of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth-backendenvironmentname-remove.html#DeleteBackendAuth
     */
    toDeleteBackendAuth() {
        return this.to('DeleteBackendAuth');
    }
    /**
     * Grants permission to delete a backend storage resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#DeleteBackendStorage
     */
    toDeleteBackendStorage() {
        return this.to('DeleteBackendStorage');
    }
    /**
     * Grants permission to delete an Amplify Admin challenge token by appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-challenge-sessionid-remove.html#DeleteToken
     */
    toDeleteToken() {
        return this.to('DeleteToken');
    }
    /**
     * Grants permission to generate models for an API of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname-generatemodels.html#GenerateBackendAPIModels
     */
    toGenerateBackendAPIModels() {
        return this.to('GenerateBackendAPIModels');
    }
    /**
     * Grants permission to retrieve an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-details.html#GetBackend
     */
    toGetBackend() {
        return this.to('GetBackend');
    }
    /**
     * Grants permission to retrieve an API of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname-details.html#GetBackendAPI
     */
    toGetBackendAPI() {
        return this.to('GetBackendAPI');
    }
    /**
     * Grants permission to retrieve models for an API of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname-getmodels.html#GetBackendAPIModels
     */
    toGetBackendAPIModels() {
        return this.to('GetBackendAPIModels');
    }
    /**
     * Grants permission to retrieve an auth resource of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth-backendenvironmentname-details.html#GetBackendAuth
     */
    toGetBackendAuth() {
        return this.to('GetBackendAuth');
    }
    /**
     * Grants permission to retrieve a job of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-job-backendenvironmentname-jobid.html#GetBackendJob
     */
    toGetBackendJob() {
        return this.to('GetBackendJob');
    }
    /**
     * Grants permission to retrieve an existing backend storage resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#GetBackendStorage
     */
    toGetBackendStorage() {
        return this.to('GetBackendStorage');
    }
    /**
     * Grants permission to retrieve an Amplify Admin challenge token by appId
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-challenge-sessionid.html#GetToken
     */
    toGetToken() {
        return this.to('GetToken');
    }
    /**
     * Grants permission to import an existing auth resource of an Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth-backendenvironmentname.html#ImportBackendAuth
     */
    toImportBackendAuth() {
        return this.to('ImportBackendAuth');
    }
    /**
     * Grants permission to import an existing backend storage resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#ImportBackendStorage
     */
    toImportBackendStorage() {
        return this.to('ImportBackendStorage');
    }
    /**
     * Grants permission to retrieve the jobs of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-job-backendenvironmentname.html#ListBackendJobs
     */
    toListBackendJobs() {
        return this.to('ListBackendJobs');
    }
    /**
     * Grants permission to retrieve s3 buckets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - s3:ListAllMyBuckets
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#ListS3Buckets
     */
    toListS3Buckets() {
        return this.to('ListS3Buckets');
    }
    /**
     * Grants permission to delete all existing Amplify Admin backend environments by appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-remove.html#RemoveAllBackends
     */
    toRemoveAllBackends() {
        return this.to('RemoveAllBackends');
    }
    /**
     * Grants permission to delete an Amplify Admin backend config by Amplify appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-config-remove.html#RemoveBackendConfig
     */
    toRemoveBackendConfig() {
        return this.to('RemoveBackendConfig');
    }
    /**
     * Grants permission to update an API of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname.html#UpdateBackendAPI
     */
    toUpdateBackendAPI() {
        return this.to('UpdateBackendAPI');
    }
    /**
     * Grants permission to update an auth resource of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth-backendenvironmentname.html#UpdateBackendAuth
     */
    toUpdateBackendAuth() {
        return this.to('UpdateBackendAuth');
    }
    /**
     * Grants permission to update an Amplify Admin backend config by Amplify appId
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-config-update.html#UpdateBackendConfig
     */
    toUpdateBackendConfig() {
        return this.to('UpdateBackendConfig');
    }
    /**
     * Grants permission to update a job of an existing Amplify Admin backend environment by appId and backendEnvironmentName
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-job-backendenvironmentname-jobid.html#UpdateBackendJob
     */
    toUpdateBackendJob() {
        return this.to('UpdateBackendJob');
    }
    /**
     * Grants permission to update a backend storage resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html#UpdateBackendStorage
     */
    toUpdateBackendStorage() {
        return this.to('UpdateBackendStorage');
    }
    /**
     * Adds a resource of type created-backend to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCreatedBackend(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${resourceName}`);
    }
    /**
     * Adds a resource of type backend to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBackend(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/*`);
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api-backendenvironmentname-details.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEnvironment(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/environments/${resourceName}`);
    }
    /**
     * Adds a resource of type api to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-api.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApi(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/api/${resourceName}`);
    }
    /**
     * Adds a resource of type auth to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-auth.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAuth(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/auth/${resourceName}`);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-job-backendenvironmentname.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJob(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/job/${resourceName}`);
    }
    /**
     * Adds a resource of type config to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-config.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConfig(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/config/${resourceName}`);
    }
    /**
     * Adds a resource of type token to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-token.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onToken(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/challenge/${resourceName}`);
    }
    /**
     * Adds a resource of type storage to the statement
     *
     * https://docs.aws.amazon.com/amplify-admin-ui/latest/APIReference/backend-appid-storage.html
     *
     * @param appId - Identifier for the appId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStorage(appId, resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplifybackend:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/backend/${appId}/storage/${resourceName}`);
    }
    /**
     * Statement provider for service [amplifybackend](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsamplifyadmin.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'amplifybackend';
        this.accessLevelList = {
            Write: [
                'CloneBackend',
                'CreateBackend',
                'CreateBackendAPI',
                'CreateBackendAuth',
                'CreateBackendConfig',
                'CreateBackendStorage',
                'CreateToken',
                'DeleteBackend',
                'DeleteBackendAPI',
                'DeleteBackendAuth',
                'DeleteBackendStorage',
                'DeleteToken',
                'GenerateBackendAPIModels',
                'ImportBackendAuth',
                'ImportBackendStorage',
                'RemoveAllBackends',
                'RemoveBackendConfig',
                'UpdateBackendAPI',
                'UpdateBackendAuth',
                'UpdateBackendConfig',
                'UpdateBackendJob',
                'UpdateBackendStorage'
            ],
            Read: [
                'GetBackend',
                'GetBackendAPI',
                'GetBackendAPIModels',
                'GetBackendAuth',
                'GetBackendJob',
                'GetBackendStorage',
                'GetToken'
            ],
            List: [
                'ListBackendJobs',
                'ListS3Buckets'
            ]
        };
    }
}
exports.Amplifybackend = Amplifybackend;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW1wbGlmeWFkbWluLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYW1wbGlmeWFkbWluLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUErQztBQUcvQzs7OztHQUlHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsd0JBQWU7SUFHakQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBMENEOzs7Ozs7Ozs7T0FTRztJQUNJLGdCQUFnQixDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsbUJBQW9CLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDaEwsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFNBQVMsQ0FBQyxLQUFhLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsbUJBQW9CLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLEtBQU0sSUFBSSxDQUFDLENBQUM7SUFDM0ssQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhLENBQUMsS0FBYSxFQUFFLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0csT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsbUJBQW9CLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLEtBQU0saUJBQWtCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDeE0sQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxLQUFLLENBQUMsS0FBYSxFQUFFLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsbUJBQW9CLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLEtBQU0sUUFBUyxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQy9MLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksTUFBTSxDQUFDLEtBQWEsRUFBRSxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLG1CQUFvQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsYUFBYyxLQUFNLFNBQVUsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNoTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLEtBQUssQ0FBQyxLQUFhLEVBQUUsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixtQkFBb0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsS0FBTSxRQUFTLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDL0wsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxRQUFRLENBQUMsS0FBYSxFQUFFLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDeEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsbUJBQW9CLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxhQUFjLEtBQU0sV0FBWSxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQ2xNLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksT0FBTyxDQUFDLEtBQWEsRUFBRSxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLG1CQUFvQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsYUFBYyxLQUFNLGNBQWUsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNyTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFNBQVMsQ0FBQyxLQUFhLEVBQUUsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixtQkFBb0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsS0FBTSxZQUFhLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDbk0sQ0FBQztJQUVEOzs7T0FHRztJQUNILFlBQVksS0FBZ0M7UUFDMUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBNWdCUixrQkFBYSxHQUFHLGdCQUFnQixDQUFDO1FBMFY5QixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsY0FBYztnQkFDZCxlQUFlO2dCQUNmLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixxQkFBcUI7Z0JBQ3JCLHNCQUFzQjtnQkFDdEIsYUFBYTtnQkFDYixlQUFlO2dCQUNmLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixzQkFBc0I7Z0JBQ3RCLGFBQWE7Z0JBQ2IsMEJBQTBCO2dCQUMxQixtQkFBbUI7Z0JBQ25CLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQixxQkFBcUI7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIsbUJBQW1CO2dCQUNuQixxQkFBcUI7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIsc0JBQXNCO2FBQ3ZCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLFlBQVk7Z0JBQ1osZUFBZTtnQkFDZixxQkFBcUI7Z0JBQ3JCLGdCQUFnQjtnQkFDaEIsZUFBZTtnQkFDZixtQkFBbUI7Z0JBQ25CLFVBQVU7YUFDWDtZQUNELElBQUksRUFBRTtnQkFDSixpQkFBaUI7Z0JBQ2pCLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBNklGLENBQUM7Q0FDRjtBQS9nQkQsd0NBK2dCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthbXBsaWZ5YmFja2VuZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYW1wbGlmeWFkbWluLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQW1wbGlmeWJhY2tlbmQgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdhbXBsaWZ5YmFja2VuZCc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNsb25lIGFuIGV4aXN0aW5nIEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9ubWVudCBpbnRvIGEgbmV3IEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWVudmlyb25tZW50cy1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLWNsb25lLmh0bWwjQ2xvbmVCYWNrZW5kXG4gICAqL1xuICBwdWJsaWMgdG9DbG9uZUJhY2tlbmQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Nsb25lQmFja2VuZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgQW1wbGlmeSBhcHBJZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC5odG1sI0NyZWF0ZUJhY2tlbmRcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUJhY2tlbmQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUJhY2tlbmQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQVBJIGZvciBhbiBleGlzdGluZyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgYXBwSWQgYW5kIGJhY2tlbmRFbnZpcm9ubWVudE5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXBpLmh0bWwjQ3JlYXRlQmFja2VuZEFQSVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQmFja2VuZEFQSSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQmFja2VuZEFQSScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBhdXRoIHJlc291cmNlIGZvciBhbiBleGlzdGluZyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgYXBwSWQgYW5kIGJhY2tlbmRFbnZpcm9ubWVudE5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXV0aC5odG1sI0NyZWF0ZUJhY2tlbmRBdXRoXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCYWNrZW5kQXV0aCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQmFja2VuZEF1dGgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGNvbmZpZyBieSBBbXBsaWZ5IGFwcElkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWNvbmZpZy5odG1sI0NyZWF0ZUJhY2tlbmRDb25maWdcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUJhY2tlbmRDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUJhY2tlbmRDb25maWcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBiYWNrZW5kIHN0b3JhZ2UgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtc3RvcmFnZS5odG1sI0NyZWF0ZUJhY2tlbmRTdG9yYWdlXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCYWNrZW5kU3RvcmFnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQmFja2VuZFN0b3JhZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYW4gQW1wbGlmeSBBZG1pbiBjaGFsbGVuZ2UgdG9rZW4gYnkgYXBwSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtY2hhbGxlbmdlLmh0bWwjQ3JlYXRlVG9rZW5cbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVRva2VuKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVUb2tlbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBleGlzdGluZyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgYXBwSWQgYW5kIGJhY2tlbmRFbnZpcm9ubWVudE5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtZW52aXJvbm1lbnRzLWJhY2tlbmRlbnZpcm9ubWVudG5hbWUtcmVtb3ZlLmh0bWwjRGVsZXRlQmFja2VuZFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQmFja2VuZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQmFja2VuZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBUEkgb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWFwaS1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLXJlbW92ZS5odG1sI0RlbGV0ZUJhY2tlbmRBUElcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUJhY2tlbmRBUEkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUJhY2tlbmRBUEknKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gYXV0aCByZXNvdXJjZSBvZiBhbiBleGlzdGluZyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgYXBwSWQgYW5kIGJhY2tlbmRFbnZpcm9ubWVudE5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXV0aC1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLXJlbW92ZS5odG1sI0RlbGV0ZUJhY2tlbmRBdXRoXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVCYWNrZW5kQXV0aCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQmFja2VuZEF1dGgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBiYWNrZW5kIHN0b3JhZ2UgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtc3RvcmFnZS5odG1sI0RlbGV0ZUJhY2tlbmRTdG9yYWdlXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVCYWNrZW5kU3RvcmFnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQmFja2VuZFN0b3JhZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQW1wbGlmeSBBZG1pbiBjaGFsbGVuZ2UgdG9rZW4gYnkgYXBwSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtY2hhbGxlbmdlLXNlc3Npb25pZC1yZW1vdmUuaHRtbCNEZWxldGVUb2tlblxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlVG9rZW4oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVRva2VuJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2VuZXJhdGUgbW9kZWxzIGZvciBhbiBBUEkgb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWFwaS1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLWdlbmVyYXRlbW9kZWxzLmh0bWwjR2VuZXJhdGVCYWNrZW5kQVBJTW9kZWxzXG4gICAqL1xuICBwdWJsaWMgdG9HZW5lcmF0ZUJhY2tlbmRBUElNb2RlbHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dlbmVyYXRlQmFja2VuZEFQSU1vZGVscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIGV4aXN0aW5nIEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9ubWVudCBieSBhcHBJZCBhbmQgYmFja2VuZEVudmlyb25tZW50TmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWRldGFpbHMuaHRtbCNHZXRCYWNrZW5kXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCYWNrZW5kKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCYWNrZW5kJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gQVBJIG9mIGFuIGV4aXN0aW5nIEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9ubWVudCBieSBhcHBJZCBhbmQgYmFja2VuZEVudmlyb25tZW50TmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWFwaS1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLWRldGFpbHMuaHRtbCNHZXRCYWNrZW5kQVBJXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCYWNrZW5kQVBJKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCYWNrZW5kQVBJJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgbW9kZWxzIGZvciBhbiBBUEkgb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXBpLWJhY2tlbmRlbnZpcm9ubWVudG5hbWUtZ2V0bW9kZWxzLmh0bWwjR2V0QmFja2VuZEFQSU1vZGVsc1xuICAgKi9cbiAgcHVibGljIHRvR2V0QmFja2VuZEFQSU1vZGVscygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QmFja2VuZEFQSU1vZGVscycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIGF1dGggcmVzb3VyY2Ugb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXV0aC1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLWRldGFpbHMuaHRtbCNHZXRCYWNrZW5kQXV0aFxuICAgKi9cbiAgcHVibGljIHRvR2V0QmFja2VuZEF1dGgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEJhY2tlbmRBdXRoJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBqb2Igb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtam9iLWJhY2tlbmRlbnZpcm9ubWVudG5hbWUtam9iaWQuaHRtbCNHZXRCYWNrZW5kSm9iXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCYWNrZW5kSm9iKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCYWNrZW5kSm9iJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gZXhpc3RpbmcgYmFja2VuZCBzdG9yYWdlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtc3RvcmFnZS5odG1sI0dldEJhY2tlbmRTdG9yYWdlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRCYWNrZW5kU3RvcmFnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0QmFja2VuZFN0b3JhZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBBbXBsaWZ5IEFkbWluIGNoYWxsZW5nZSB0b2tlbiBieSBhcHBJZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWNoYWxsZW5nZS1zZXNzaW9uaWQuaHRtbCNHZXRUb2tlblxuICAgKi9cbiAgcHVibGljIHRvR2V0VG9rZW4oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFRva2VuJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gaW1wb3J0IGFuIGV4aXN0aW5nIGF1dGggcmVzb3VyY2Ugb2YgYW4gQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWF1dGgtYmFja2VuZGVudmlyb25tZW50bmFtZS5odG1sI0ltcG9ydEJhY2tlbmRBdXRoXG4gICAqL1xuICBwdWJsaWMgdG9JbXBvcnRCYWNrZW5kQXV0aCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnSW1wb3J0QmFja2VuZEF1dGgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBpbXBvcnQgYW4gZXhpc3RpbmcgYmFja2VuZCBzdG9yYWdlIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLXN0b3JhZ2UuaHRtbCNJbXBvcnRCYWNrZW5kU3RvcmFnZVxuICAgKi9cbiAgcHVibGljIHRvSW1wb3J0QmFja2VuZFN0b3JhZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0ltcG9ydEJhY2tlbmRTdG9yYWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGpvYnMgb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtam9iLWJhY2tlbmRlbnZpcm9ubWVudG5hbWUuaHRtbCNMaXN0QmFja2VuZEpvYnNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RCYWNrZW5kSm9icygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEJhY2tlbmRKb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgczMgYnVja2V0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gczM6TGlzdEFsbE15QnVja2V0c1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtc3RvcmFnZS5odG1sI0xpc3RTM0J1Y2tldHNcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RTM0J1Y2tldHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTM0J1Y2tldHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYWxsIGV4aXN0aW5nIEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9ubWVudHMgYnkgYXBwSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtcmVtb3ZlLmh0bWwjUmVtb3ZlQWxsQmFja2VuZHNcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZUFsbEJhY2tlbmRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVBbGxCYWNrZW5kcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgY29uZmlnIGJ5IEFtcGxpZnkgYXBwSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtY29uZmlnLXJlbW92ZS5odG1sI1JlbW92ZUJhY2tlbmRDb25maWdcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZUJhY2tlbmRDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlbW92ZUJhY2tlbmRDb25maWcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gQVBJIG9mIGFuIGV4aXN0aW5nIEFtcGxpZnkgQWRtaW4gYmFja2VuZCBlbnZpcm9ubWVudCBieSBhcHBJZCBhbmQgYmFja2VuZEVudmlyb25tZW50TmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC1hcHBpZC1hcGktYmFja2VuZGVudmlyb25tZW50bmFtZS5odG1sI1VwZGF0ZUJhY2tlbmRBUElcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUJhY2tlbmRBUEkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUJhY2tlbmRBUEknKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYW4gYXV0aCByZXNvdXJjZSBvZiBhbiBleGlzdGluZyBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgZW52aXJvbm1lbnQgYnkgYXBwSWQgYW5kIGJhY2tlbmRFbnZpcm9ubWVudE5hbWVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXV0aC1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLmh0bWwjVXBkYXRlQmFja2VuZEF1dGhcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUJhY2tlbmRBdXRoKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVCYWNrZW5kQXV0aCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBBbXBsaWZ5IEFkbWluIGJhY2tlbmQgY29uZmlnIGJ5IEFtcGxpZnkgYXBwSWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtY29uZmlnLXVwZGF0ZS5odG1sI1VwZGF0ZUJhY2tlbmRDb25maWdcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUJhY2tlbmRDb25maWcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUJhY2tlbmRDb25maWcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBqb2Igb2YgYW4gZXhpc3RpbmcgQW1wbGlmeSBBZG1pbiBiYWNrZW5kIGVudmlyb25tZW50IGJ5IGFwcElkIGFuZCBiYWNrZW5kRW52aXJvbm1lbnROYW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FtcGxpZnktYWRtaW4tdWkvbGF0ZXN0L0FQSVJlZmVyZW5jZS9iYWNrZW5kLWFwcGlkLWpvYi1iYWNrZW5kZW52aXJvbm1lbnRuYW1lLWpvYmlkLmh0bWwjVXBkYXRlQmFja2VuZEpvYlxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQmFja2VuZEpvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQmFja2VuZEpvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGJhY2tlbmQgc3RvcmFnZSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC1hcHBpZC1zdG9yYWdlLmh0bWwjVXBkYXRlQmFja2VuZFN0b3JhZ2VcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUJhY2tlbmRTdG9yYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVCYWNrZW5kU3RvcmFnZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQ2xvbmVCYWNrZW5kJyxcbiAgICAgICdDcmVhdGVCYWNrZW5kJyxcbiAgICAgICdDcmVhdGVCYWNrZW5kQVBJJyxcbiAgICAgICdDcmVhdGVCYWNrZW5kQXV0aCcsXG4gICAgICAnQ3JlYXRlQmFja2VuZENvbmZpZycsXG4gICAgICAnQ3JlYXRlQmFja2VuZFN0b3JhZ2UnLFxuICAgICAgJ0NyZWF0ZVRva2VuJyxcbiAgICAgICdEZWxldGVCYWNrZW5kJyxcbiAgICAgICdEZWxldGVCYWNrZW5kQVBJJyxcbiAgICAgICdEZWxldGVCYWNrZW5kQXV0aCcsXG4gICAgICAnRGVsZXRlQmFja2VuZFN0b3JhZ2UnLFxuICAgICAgJ0RlbGV0ZVRva2VuJyxcbiAgICAgICdHZW5lcmF0ZUJhY2tlbmRBUElNb2RlbHMnLFxuICAgICAgJ0ltcG9ydEJhY2tlbmRBdXRoJyxcbiAgICAgICdJbXBvcnRCYWNrZW5kU3RvcmFnZScsXG4gICAgICAnUmVtb3ZlQWxsQmFja2VuZHMnLFxuICAgICAgJ1JlbW92ZUJhY2tlbmRDb25maWcnLFxuICAgICAgJ1VwZGF0ZUJhY2tlbmRBUEknLFxuICAgICAgJ1VwZGF0ZUJhY2tlbmRBdXRoJyxcbiAgICAgICdVcGRhdGVCYWNrZW5kQ29uZmlnJyxcbiAgICAgICdVcGRhdGVCYWNrZW5kSm9iJyxcbiAgICAgICdVcGRhdGVCYWNrZW5kU3RvcmFnZSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdHZXRCYWNrZW5kJyxcbiAgICAgICdHZXRCYWNrZW5kQVBJJyxcbiAgICAgICdHZXRCYWNrZW5kQVBJTW9kZWxzJyxcbiAgICAgICdHZXRCYWNrZW5kQXV0aCcsXG4gICAgICAnR2V0QmFja2VuZEpvYicsXG4gICAgICAnR2V0QmFja2VuZFN0b3JhZ2UnLFxuICAgICAgJ0dldFRva2VuJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0xpc3RCYWNrZW5kSm9icycsXG4gICAgICAnTGlzdFMzQnVja2V0cydcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNyZWF0ZWQtYmFja2VuZCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkNyZWF0ZWRCYWNrZW5kKHJlc291cmNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFtcGxpZnliYWNrZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovYmFja2VuZC8keyByZXNvdXJjZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGJhY2tlbmQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQmFja2VuZChhcHBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFtcGxpZnliYWNrZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovYmFja2VuZC8keyBhcHBJZCB9LypgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBlbnZpcm9ubWVudCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC1hcHBpZC1hcGktYmFja2VuZGVudmlyb25tZW50bmFtZS1kZXRhaWxzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcElkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25FbnZpcm9ubWVudChhcHBJZDogc3RyaW5nLCByZXNvdXJjZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphbXBsaWZ5YmFja2VuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2JhY2tlbmQvJHsgYXBwSWQgfS9lbnZpcm9ubWVudHMvJHsgcmVzb3VyY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBhcGkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtYXBpLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcElkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25BcGkoYXBwSWQ6IHN0cmluZywgcmVzb3VyY2VOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YW1wbGlmeWJhY2tlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9iYWNrZW5kLyR7IGFwcElkIH0vYXBpLyR7IHJlc291cmNlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXV0aCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hbXBsaWZ5LWFkbWluLXVpL2xhdGVzdC9BUElSZWZlcmVuY2UvYmFja2VuZC1hcHBpZC1hdXRoLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcElkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25BdXRoKGFwcElkOiBzdHJpbmcsIHJlc291cmNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFtcGxpZnliYWNrZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovYmFja2VuZC8keyBhcHBJZCB9L2F1dGgvJHsgcmVzb3VyY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBqb2IgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtam9iLWJhY2tlbmRlbnZpcm9ubWVudG5hbWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwSWQuXG4gICAqIEBwYXJhbSByZXNvdXJjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkpvYihhcHBJZDogc3RyaW5nLCByZXNvdXJjZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphbXBsaWZ5YmFja2VuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2JhY2tlbmQvJHsgYXBwSWQgfS9qb2IvJHsgcmVzb3VyY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjb25maWcgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtY29uZmlnLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcElkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25Db25maWcoYXBwSWQ6IHN0cmluZywgcmVzb3VyY2VOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YW1wbGlmeWJhY2tlbmQ6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9iYWNrZW5kLyR7IGFwcElkIH0vY29uZmlnLyR7IHJlc291cmNlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgdG9rZW4gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtdG9rZW4uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwSWQuXG4gICAqIEBwYXJhbSByZXNvdXJjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblRva2VuKGFwcElkOiBzdHJpbmcsIHJlc291cmNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFtcGxpZnliYWNrZW5kOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTovYmFja2VuZC8keyBhcHBJZCB9L2NoYWxsZW5nZS8keyByZXNvdXJjZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHN0b3JhZ2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYW1wbGlmeS1hZG1pbi11aS9sYXRlc3QvQVBJUmVmZXJlbmNlL2JhY2tlbmQtYXBwaWQtc3RvcmFnZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcHBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBJZC5cbiAgICogQHBhcmFtIHJlc291cmNlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uU3RvcmFnZShhcHBJZDogc3RyaW5nLCByZXNvdXJjZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphbXBsaWZ5YmFja2VuZDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2JhY2tlbmQvJHsgYXBwSWQgfS9zdG9yYWdlLyR7IHJlc291cmNlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthbXBsaWZ5YmFja2VuZF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzYW1wbGlmeWFkbWluLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==