"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExecuteApi = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [execute-api](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ExecuteApi extends shared_1.PolicyStatement {
    /**
     * Used to invalidate API cache upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/api-gateway-caching.html
     */
    toInvalidateCache() {
        return this.to('InvalidateCache');
    }
    /**
     * Used to invoke an API upon a client request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/how-to-call-api.html
     */
    toInvoke() {
        return this.to('Invoke');
    }
    /**
     * ManageConnections controls access to the @connections API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apigateway/api-reference/apigateway-websocket-control-access-iam.html
     */
    toManageConnections() {
        return this.to('ManageConnections');
    }
    /**
     * Adds a resource of type execute-api-general to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param apiId - Identifier for the apiId.
     * @param stage - Identifier for the stage.
     * @param method - Identifier for the method.
     * @param apiSpecificResourcePath - Identifier for the apiSpecificResourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifViaDomainArn()
     */
    onExecuteApiGeneral(apiId, stage, method, apiSpecificResourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:execute-api:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${apiId}/${stage}/${method}/${apiSpecificResourcePath}`);
    }
    /**
     * Adds a resource of type execute-api-domain to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExecuteApiDomain(domainName, domainIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:execute-api:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnames/${domainName}+${domainIdentifier}`);
    }
    /**
     * Filters access by the domain name ARN the API is called from
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - execute-api-general
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaDomainArn(value, operator) {
        return this.if(`viaDomainArn`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [execute-api](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigateway.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'execute-api';
        this.accessLevelList = {
            Write: [
                'InvalidateCache',
                'Invoke',
                'ManageConnections'
            ]
        };
    }
}
exports.ExecuteApi = ExecuteApi;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpZ2F0ZXdheS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwaWdhdGV3YXkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQUc3Qzs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFFBQVE7UUFDYixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDM0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBVUQ7Ozs7Ozs7Ozs7Ozs7OztPQWVHO0lBQ0ksbUJBQW1CLENBQUMsS0FBYSxFQUFFLEtBQWEsRUFBRSxNQUFjLEVBQUUsdUJBQStCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDN0osT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxJQUFLLEtBQU0sSUFBSyxLQUFNLElBQUssTUFBTyxJQUFLLHVCQUF3QixFQUFFLENBQUMsQ0FBQztJQUNqTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtCQUFrQixDQUFDLFVBQWtCLEVBQUUsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0gsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZ0JBQWlCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxpQkFBa0IsVUFBVyxJQUFLLGdCQUFpQixFQUFFLENBQUMsQ0FBQztJQUNyTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGNBQWMsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzFFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFsR1Isa0JBQWEsR0FBRyxhQUFhLENBQUM7UUFtQzNCLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxpQkFBaUI7Z0JBQ2pCLFFBQVE7Z0JBQ1IsbUJBQW1CO2FBQ3BCO1NBQ0YsQ0FBQztJQTBERixDQUFDO0NBQ0Y7QUFyR0QsZ0NBcUdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtleGVjdXRlLWFwaV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uYXBpZ2F0ZXdheS5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEV4ZWN1dGVBcGkgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdleGVjdXRlLWFwaSc7XG5cbiAgLyoqXG4gICAqIFVzZWQgdG8gaW52YWxpZGF0ZSBBUEkgY2FjaGUgdXBvbiBhIGNsaWVudCByZXF1ZXN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvYXBpLXJlZmVyZW5jZS9hcGktZ2F0ZXdheS1jYWNoaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0ludmFsaWRhdGVDYWNoZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnSW52YWxpZGF0ZUNhY2hlJyk7XG4gIH1cblxuICAvKipcbiAgICogVXNlZCB0byBpbnZva2UgYW4gQVBJIHVwb24gYSBjbGllbnQgcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2FwaS1yZWZlcmVuY2UvaG93LXRvLWNhbGwtYXBpLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0ludm9rZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnSW52b2tlJyk7XG4gIH1cblxuICAvKipcbiAgICogTWFuYWdlQ29ubmVjdGlvbnMgY29udHJvbHMgYWNjZXNzIHRvIHRoZSBAY29ubmVjdGlvbnMgQVBJXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvYXBpLXJlZmVyZW5jZS9hcGlnYXRld2F5LXdlYnNvY2tldC1jb250cm9sLWFjY2Vzcy1pYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTWFuYWdlQ29ubmVjdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ01hbmFnZUNvbm5lY3Rpb25zJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdJbnZhbGlkYXRlQ2FjaGUnLFxuICAgICAgJ0ludm9rZScsXG4gICAgICAnTWFuYWdlQ29ubmVjdGlvbnMnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBleGVjdXRlLWFwaS1nZW5lcmFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwaUlkLlxuICAgKiBAcGFyYW0gc3RhZ2UgLSBJZGVudGlmaWVyIGZvciB0aGUgc3RhZ2UuXG4gICAqIEBwYXJhbSBtZXRob2QgLSBJZGVudGlmaWVyIGZvciB0aGUgbWV0aG9kLlxuICAgKiBAcGFyYW0gYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGggLSBJZGVudGlmaWVyIGZvciB0aGUgYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlZpYURvbWFpbkFybigpXG4gICAqL1xuICBwdWJsaWMgb25FeGVjdXRlQXBpR2VuZXJhbChhcGlJZDogc3RyaW5nLCBzdGFnZTogc3RyaW5nLCBtZXRob2Q6IHN0cmluZywgYXBpU3BlY2lmaWNSZXNvdXJjZVBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpleGVjdXRlLWFwaTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06JHsgYXBpSWQgfS8keyBzdGFnZSB9LyR7IG1ldGhvZCB9LyR7IGFwaVNwZWNpZmljUmVzb3VyY2VQYXRoIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBleGVjdXRlLWFwaS1kb21haW4gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gZG9tYWluSWRlbnRpZmllciAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5JZGVudGlmaWVyLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkV4ZWN1dGVBcGlEb21haW4oZG9tYWluTmFtZTogc3RyaW5nLCBkb21haW5JZGVudGlmaWVyOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06ZXhlY3V0ZS1hcGk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH0rJHsgZG9tYWluSWRlbnRpZmllciB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGRvbWFpbiBuYW1lIEFSTiB0aGUgQVBJIGlzIGNhbGxlZCBmcm9tXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gZXhlY3V0ZS1hcGktZ2VuZXJhbFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlZpYURvbWFpbkFybih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgdmlhRG9tYWluQXJuYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtleGVjdXRlLWFwaV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uYXBpZ2F0ZXdheS5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=