"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Apigateway = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [apigateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigatewaymanagement.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Apigateway extends shared_1.PolicyStatement {
    /**
     * Grants permission to add certificates for mutual TLS authentication to a domain name. This is an additional authorization control for managing the DomainName resource due to the sensitive nature of mTLS
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toAddCertificateToDomain() {
        return this.to('AddCertificateToDomain');
    }
    /**
     * Grants permission to create an access association from an access association source to a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toCreateAccessAssociation() {
        return this.to('CreateAccessAssociation');
    }
    /**
     * Grants permission to delete a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toDELETE() {
        return this.to('DELETE');
    }
    /**
     * Grants permission to read a particular resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toGET() {
        return this.to('GET');
    }
    /**
     * Grants permission to update a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPATCH() {
        return this.to('PATCH');
    }
    /**
     * Grants permission to create a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPOST() {
        return this.to('POST');
    }
    /**
     * Grants permission to update a particular resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toPUT() {
        return this.to('PUT');
    }
    /**
     * Grants permission to reject an existing access association owned by another account to a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toRejectAccessAssociation() {
        return this.to('RejectAccessAssociation');
    }
    /**
     * Grants permission to remove certificates for mutual TLS authentication from a domain name. This is an additional authorization control for managing the DomainName resource due to the sensitive nature of mTLS
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toRemoveCertificateFromDomain() {
        return this.to('RemoveCertificateFromDomain');
    }
    /**
     * Grants permission to set a WAF access control list (ACL). This is an additional authorization control for managing the Stage resource due to the sensitive nature of WebAcl's
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toSetWebACL() {
        return this.to('SetWebACL');
    }
    /**
     * Grants permission to update the management policy of a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateDomainNameManagementPolicy() {
        return this.to('UpdateDomainNameManagementPolicy');
    }
    /**
     * Grants permission to update the invoke policy of a custom domain name for private APIs
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateDomainNamePolicy() {
        return this.to('UpdateDomainNamePolicy');
    }
    /**
     * Grants permission to manage the IAM resource policy for an API. This is an additional authorization control for managing an API due to the sensitive nature of the resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Operations.html
     */
    toUpdateRestApiPolicy() {
        return this.to('UpdateRestApiPolicy');
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/account`);
    }
    /**
     * Adds a resource of type ApiKey to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ApiKey.html
     *
     * @param apiKeyId - Identifier for the apiKeyId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApiKey(apiKeyId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/apikeys/${apiKeyId}`);
    }
    /**
     * Adds a resource of type ApiKeys to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ApiKey.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApiKeys(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/apikeys`);
    }
    /**
     * Adds a resource of type Authorizer to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Authorizer.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param authorizerId - Identifier for the authorizerId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestCognitoUserPoolProviderArn()
     * - .ifResourceAuthorizerType()
     * - .ifResourceAuthorizerUri()
     * - .ifResourceCognitoUserPoolProviderArn()
     * - .ifAwsResourceTag()
     */
    onAuthorizer(restApiId, authorizerId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/authorizers/${authorizerId}`);
    }
    /**
     * Adds a resource of type Authorizers to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Authorizer.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestCognitoUserPoolProviderArn()
     * - .ifAwsResourceTag()
     */
    onAuthorizers(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/authorizers`);
    }
    /**
     * Adds a resource of type BasePathMapping to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param basePath - Identifier for the basePath.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBasePathMapping(domainName, basePath, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}/basepathmappings/${basePath}`);
    }
    /**
     * Adds a resource of type BasePathMappings to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBasePathMappings(domainName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}/basepathmappings`);
    }
    /**
     * Adds a resource of type ClientCertificate to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ClientCertificate.html
     *
     * @param clientCertificateId - Identifier for the clientCertificateId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClientCertificate(clientCertificateId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/clientcertificates/${clientCertificateId}`);
    }
    /**
     * Adds a resource of type ClientCertificates to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_ClientCertificate.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onClientCertificates(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/clientcertificates`);
    }
    /**
     * Adds a resource of type Deployment to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Deployment.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param deploymentId - Identifier for the deploymentId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeployment(restApiId, deploymentId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/deployments/${deploymentId}`);
    }
    /**
     * Adds a resource of type Deployments to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Deployment.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestStageName()
     * - .ifAwsResourceTag()
     */
    onDeployments(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/deployments`);
    }
    /**
     * Adds a resource of type DocumentationPart to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPart.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param documentationPartId - Identifier for the documentationPartId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationPart(restApiId, documentationPartId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/parts/${documentationPartId}`);
    }
    /**
     * Adds a resource of type DocumentationParts to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationPart.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationParts(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/parts`);
    }
    /**
     * Adds a resource of type DocumentationVersion to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationVersion.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param documentationVersionId - Identifier for the documentationVersionId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationVersion(restApiId, documentationVersionId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/versions/${documentationVersionId}`);
    }
    /**
     * Adds a resource of type DocumentationVersions to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DocumentationVersion.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDocumentationVersions(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/documentation/versions`);
    }
    /**
     * Adds a resource of type DomainName to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param domainName - Identifier for the domainName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifRequestMtlsTrustStoreUri()
     * - .ifRequestMtlsTrustStoreVersion()
     * - .ifRequestSecurityPolicy()
     * - .ifResourceEndpointType()
     * - .ifResourceMtlsTrustStoreUri()
     * - .ifResourceMtlsTrustStoreVersion()
     * - .ifResourceSecurityPolicy()
     * - .ifAwsResourceTag()
     */
    onDomainName(domainName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}`);
    }
    /**
     * Adds a resource of type DomainNames to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifRequestMtlsTrustStoreUri()
     * - .ifRequestMtlsTrustStoreVersion()
     * - .ifRequestSecurityPolicy()
     * - .ifAwsResourceTag()
     */
    onDomainNames(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames`);
    }
    /**
     * Adds a resource of type DomainNameAccessAssociation to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainNameAccessAssociation.html
     *
     * @param domainName - Identifier for the domainName.
     * @param sourceType - Identifier for the sourceType.
     * @param sourceId - Identifier for the sourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDomainNameAccessAssociation(domainName, sourceType, sourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnameaccessassociations/domainname/${domainName}/${sourceType}/${sourceId}`);
    }
    /**
     * Adds a resource of type DomainNameAccessAssociations to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainNameAccessAssociation.html
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessAssociationSource()
     * - .ifRequestDomainNameArn()
     * - .ifAwsResourceTag()
     */
    onDomainNameAccessAssociations(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnameaccessassociations`);
    }
    /**
     * Adds a resource of type GatewayResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param responseType - Identifier for the responseType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGatewayResponse(restApiId, responseType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/gatewayresponses/${responseType}`);
    }
    /**
     * Adds a resource of type GatewayResponses to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_GatewayResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGatewayResponses(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/gatewayresponses`);
    }
    /**
     * Adds a resource of type Integration to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Integration.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegration(restApiId, resourceId, httpMethodType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/integration`);
    }
    /**
     * Adds a resource of type IntegrationResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_IntegrationResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param statusCode - Identifier for the statusCode.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrationResponse(restApiId, resourceId, httpMethodType, statusCode, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/integration/responses/${statusCode}`);
    }
    /**
     * Adds a resource of type Method to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Method.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestRouteAuthorizationType()
     * - .ifResourceApiKeyRequired()
     * - .ifResourceRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onMethod(restApiId, resourceId, httpMethodType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}`);
    }
    /**
     * Adds a resource of type MethodResponse to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_MethodResponse.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param httpMethodType - Identifier for the httpMethodType.
     * @param statusCode - Identifier for the statusCode.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMethodResponse(restApiId, resourceId, httpMethodType, statusCode, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}/methods/${httpMethodType}/responses/${statusCode}`);
    }
    /**
     * Adds a resource of type Model to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Model.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param modelName - Identifier for the modelName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(restApiId, modelName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/models/${modelName}`);
    }
    /**
     * Adds a resource of type Models to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Model.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModels(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/models`);
    }
    /**
     * Adds a resource of type PrivateBasePathMapping to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param basePath - Identifier for the basePath.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrivateBasePathMapping(domainName, domainIdentifier, basePath, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}+${domainIdentifier}/basepathmappings/${basePath}`);
    }
    /**
     * Adds a resource of type PrivateBasePathMappings to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_BasePathMapping.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrivateBasePathMappings(domainName, domainIdentifier, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/domainnames/${domainName}+${domainIdentifier}/basepathmappings`);
    }
    /**
     * Adds a resource of type PrivateDomainName to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_DomainName.html
     *
     * @param domainName - Identifier for the domainName.
     * @param domainIdentifier - Identifier for the domainIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestEndpointType()
     * - .ifResourceEndpointType()
     * - .ifAwsResourceTag()
     */
    onPrivateDomainName(domainName, domainIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/domainnames/${domainName}+${domainIdentifier}`);
    }
    /**
     * Adds a resource of type RequestValidator to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RequestValidator.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param requestValidatorId - Identifier for the requestValidatorId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRequestValidator(restApiId, requestValidatorId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/requestvalidators/${requestValidatorId}`);
    }
    /**
     * Adds a resource of type RequestValidators to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RequestValidator.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRequestValidators(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/requestvalidators`);
    }
    /**
     * Adds a resource of type Resource to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Resource.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param resourceId - Identifier for the resourceId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResource(restApiId, resourceId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources/${resourceId}`);
    }
    /**
     * Adds a resource of type Resources to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Resource.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onResources(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/resources`);
    }
    /**
     * Adds a resource of type RestApi to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RestApi.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestApiName()
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestDisableExecuteApiEndpoint()
     * - .ifRequestEndpointType()
     * - .ifRequestRouteAuthorizationType()
     * - .ifResourceApiKeyRequired()
     * - .ifResourceApiName()
     * - .ifResourceAuthorizerType()
     * - .ifResourceAuthorizerUri()
     * - .ifResourceDisableExecuteApiEndpoint()
     * - .ifResourceEndpointType()
     * - .ifResourceRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onRestApi(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}`);
    }
    /**
     * Adds a resource of type RestApis to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_RestApi.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestApiKeyRequired()
     * - .ifRequestApiName()
     * - .ifRequestAuthorizerType()
     * - .ifRequestAuthorizerUri()
     * - .ifRequestDisableExecuteApiEndpoint()
     * - .ifRequestEndpointType()
     * - .ifRequestRouteAuthorizationType()
     * - .ifAwsResourceTag()
     */
    onRestApis(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis`);
    }
    /**
     * Adds a resource of type Sdk to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param stageName - Identifier for the stageName.
     * @param sdkType - Identifier for the sdkType.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSdk(restApiId, stageName, sdkType, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages/${stageName}/sdks/${sdkType}`);
    }
    /**
     * Adds a resource of type Stage to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param stageName - Identifier for the stageName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessLoggingDestination()
     * - .ifRequestAccessLoggingFormat()
     * - .ifResourceAccessLoggingDestination()
     * - .ifResourceAccessLoggingFormat()
     * - .ifAwsResourceTag()
     */
    onStage(restApiId, stageName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages/${stageName}`);
    }
    /**
     * Adds a resource of type Stages to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_Stage.html
     *
     * @param restApiId - Identifier for the restApiId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifRequestAccessLoggingDestination()
     * - .ifRequestAccessLoggingFormat()
     * - .ifAwsResourceTag()
     */
    onStages(restApiId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/${restApiId}/stages`);
    }
    /**
     * Adds a resource of type Template to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * @param modelName - Identifier for the modelName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(modelName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/restapis/models/${modelName}/template`);
    }
    /**
     * Adds a resource of type UsagePlan to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlan(usagePlanId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}`);
    }
    /**
     * Adds a resource of type UsagePlans to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlan.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlans(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans`);
    }
    /**
     * Adds a resource of type UsagePlanKey to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlanKey.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param id - Identifier for the id.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlanKey(usagePlanId, id, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}/keys/${id}`);
    }
    /**
     * Adds a resource of type UsagePlanKeys to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_UsagePlanKey.html
     *
     * @param usagePlanId - Identifier for the usagePlanId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUsagePlanKeys(usagePlanId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/usageplans/${usagePlanId}/keys`);
    }
    /**
     * Adds a resource of type VpcLink to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_VpcLink.html
     *
     * @param vpcLinkId - Identifier for the vpcLinkId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcLink(vpcLinkId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/vpclinks/${vpcLinkId}`);
    }
    /**
     * Adds a resource of type VpcLinks to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/api/API_VpcLink.html
     *
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcLinks(region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/vpclinks`);
    }
    /**
     * Adds a resource of type Tags to the statement
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * @param urlEncodedResourceARN - Identifier for the urlEncodedResourceARN.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTags(urlEncodedResourceARN, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}::/tags/${urlEncodedResourceARN}`);
    }
    /**
     * Filters access by access association source. Available during the CreateDomainNameAccessAssociation operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainNameAccessAssociations
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessAssociationSource(value, operator) {
        return this.if(`Request/AccessAssociationSource`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log destination. Available during the CreateStage and UpdateStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     * - Stages
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessLoggingDestination(value, operator) {
        return this.if(`Request/AccessLoggingDestination`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log format. Available during the CreateStage and UpdateStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     * - Stages
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAccessLoggingFormat(value, operator) {
        return this.if(`Request/AccessLoggingFormat`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether an API key is required or not. Available during the CreateMethod and PutMethod operations. Also available as a collection during import and reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     * - RestApis
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestApiKeyRequired(value) {
        return this.if(`Request/ApiKeyRequired`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by API name. Available during the CreateRestApi and UpdateRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestApiName(value, operator) {
        return this.if(`Request/ApiName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by type of authorizer in the request, for example TOKEN, REQUEST, JWT. Available during CreateAuthorizer and UpdateAuthorizer. Also available during import and reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAuthorizerType(value, operator) {
        return this.if(`Request/AuthorizerType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of a Lambda authorizer function. Available during CreateAuthorizer and UpdateAuthorizer. Also available during import and reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAuthorizerUri(value, operator) {
        return this.if(`Request/AuthorizerUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Cognito user pool provider ARN. Available during CreateAuthorizer and UpdateAuthorizer
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - Authorizers
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRequestCognitoUserPoolProviderArn(value, operator) {
        return this.if(`Request/CognitoUserPoolProviderArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by status of the default execute-api endpoint. Available during the CreateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     * - RestApis
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifRequestDisableExecuteApiEndpoint(value) {
        return this.if(`Request/DisableExecuteApiEndpoint`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by domain name ARN. Available during the CreateDomainNameAccessAssociation operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainNameAccessAssociations
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRequestDomainNameArn(value, operator) {
        return this.if(`Request/DomainNameArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by endpoint type. Available during the CreateDomainName, UpdateDomainName, CreateRestApi, and UpdateRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     * - PrivateDomainName
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestEndpointType(value, operator) {
        return this.if(`Request/EndpointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of the truststore used for mutual TLS authentication. Available during the CreateDomainName and UpdateDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestMtlsTrustStoreUri(value, operator) {
        return this.if(`Request/MtlsTrustStoreUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by version of the truststore used for mutual TLS authentication. Available during the CreateDomainName and UpdateDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestMtlsTrustStoreVersion(value, operator) {
        return this.if(`Request/MtlsTrustStoreVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by authorization type, for example NONE, AWS_IAM, CUSTOM, JWT, COGNITO_USER_POOLS. Available during the CreateMethod and PutMethod operations Also available as a collection during import
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     * - RestApis
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestRouteAuthorizationType(value, operator) {
        return this.if(`Request/RouteAuthorizationType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by TLS version. Available during the CreateDomain and UpdateDomain operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - DomainNames
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestSecurityPolicy(value, operator) {
        return this.if(`Request/SecurityPolicy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by stage name of the deployment that you attempt to create. Available during the CreateDeployment operation
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Deployments
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestStageName(value, operator) {
        return this.if(`Request/StageName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log destination of the current Stage resource. Available during the UpdateStage and DeleteStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccessLoggingDestination(value, operator) {
        return this.if(`Resource/AccessLoggingDestination`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by access log format of the current Stage resource. Available during the UpdateStage and DeleteStage operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Stage
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAccessLoggingFormat(value, operator) {
        return this.if(`Resource/AccessLoggingFormat`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether an API key is required or not for the existing Method resource. Available during the PutMethod and DeleteMethod operations. Also available as a collection during reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifResourceApiKeyRequired(value) {
        return this.if(`Resource/ApiKeyRequired`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by API name of the existing RestApi resource. Available during UpdateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceApiName(value, operator) {
        return this.if(`Resource/ApiName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the current type of authorizer, for example TOKEN, REQUEST, JWT. Available during UpdateAuthorizer and DeleteAuthorizer operations. Also available during reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAuthorizerType(value, operator) {
        return this.if(`Resource/AuthorizerType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of a Lambda authorizer function. Available during UpdateAuthorizer and DeleteAuthorizer operations. Also available during reimport as an ArrayOfString
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAuthorizerUri(value, operator) {
        return this.if(`Resource/AuthorizerUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by Cognito user pool provider ARN. Available during CreateAuthorizer and UpdateAuthorizer
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Authorizer
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifResourceCognitoUserPoolProviderArn(value, operator) {
        return this.if(`Resource/CognitoUserPoolProviderArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by status of the default execute-api endpoint of the current RestApi resource. Available during UpdateRestApi and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - RestApi
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifResourceDisableExecuteApiEndpoint(value) {
        return this.if(`Resource/DisableExecuteApiEndpoint`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by endpoint type. Available during the UpdateDomainName, DeleteDomainName, UpdateRestApi, and DeleteRestApi operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     * - PrivateDomainName
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceEndpointType(value, operator) {
        return this.if(`Resource/EndpointType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by URI of the truststore used for mutual TLS authentication. Available during UpdateDomainName and DeleteDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceMtlsTrustStoreUri(value, operator) {
        return this.if(`Resource/MtlsTrustStoreUri`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by version of the truststore used for mutual TLS authentication. Available during UpdateDomainName and DeleteDomainName operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceMtlsTrustStoreVersion(value, operator) {
        return this.if(`Resource/MtlsTrustStoreVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by authorization type of the existing Method resource, for example NONE, AWS_IAM, CUSTOM, JWT, COGNITO_USER_POOLS. Available during the PutMethod and DeleteMethod operations. Also available as a collection during reimport
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - Method
     * - RestApi
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceRouteAuthorizationType(value, operator) {
        return this.if(`Resource/RouteAuthorizationType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by TLS version. Available during UpdateDomain and DeleteDomain operations
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/security_iam_service-with-iam.html
     *
     * Applies to resource types:
     * - DomainName
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceSecurityPolicy(value, operator) {
        return this.if(`Resource/SecurityPolicy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to actions:
     * - .toDELETE()
     * - .toPATCH()
     * - .toPOST()
     * - .toPUT()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to resource types:
     * - ApiKey
     * - ApiKeys
     * - Authorizer
     * - Authorizers
     * - BasePathMapping
     * - BasePathMappings
     * - ClientCertificate
     * - ClientCertificates
     * - Deployment
     * - Deployments
     * - DocumentationPart
     * - DocumentationParts
     * - DocumentationVersion
     * - DocumentationVersions
     * - DomainName
     * - DomainNames
     * - DomainNameAccessAssociations
     * - GatewayResponse
     * - GatewayResponses
     * - Integration
     * - IntegrationResponse
     * - Method
     * - MethodResponse
     * - Model
     * - Models
     * - PrivateBasePathMapping
     * - PrivateBasePathMappings
     * - PrivateDomainName
     * - RequestValidator
     * - RequestValidators
     * - Resource
     * - Resources
     * - RestApi
     * - RestApis
     * - Sdk
     * - Stage
     * - Stages
     * - Template
     * - UsagePlan
     * - UsagePlans
     * - UsagePlanKey
     * - UsagePlanKeys
     * - VpcLink
     * - VpcLinks
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request
     *
     * https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-tagging.html
     *
     * Applies to actions:
     * - .toDELETE()
     * - .toPATCH()
     * - .toPOST()
     * - .toPUT()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [apigateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonapigatewaymanagement.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'apigateway';
        this.accessLevelList = {
            'Permissions management': [
                'AddCertificateToDomain',
                'CreateAccessAssociation',
                'RejectAccessAssociation',
                'RemoveCertificateFromDomain',
                'SetWebACL',
                'UpdateDomainNameManagementPolicy',
                'UpdateDomainNamePolicy',
                'UpdateRestApiPolicy'
            ],
            Write: [
                'DELETE',
                'PATCH',
                'POST',
                'PUT'
            ],
            Read: [
                'GET'
            ]
        };
    }
}
exports.Apigateway = Apigateway;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpZ2F0ZXdheW1hbmFnZW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGlnYXRld2F5bWFuYWdlbWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLHdCQUFlO0lBRzdDOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUTtRQUNiLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUMzQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksS0FBSztRQUNWLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE9BQU87UUFDWixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksS0FBSztRQUNWLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN4QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQ0FBa0M7UUFDdkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUF3QkQ7Ozs7Ozs7T0FPRztJQUNJLFNBQVMsQ0FBQyxNQUFlLEVBQUUsU0FBa0I7UUFDbEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLFlBQVksQ0FBQyxDQUFDO0lBQ3ZILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFFBQVEsQ0FBQyxRQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsY0FBZSxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3JJLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksU0FBUyxDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUNsRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsWUFBWSxDQUFDLENBQUM7SUFDdkgsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQkc7SUFDSSxZQUFZLENBQUMsU0FBaUIsRUFBRSxZQUFvQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxnQkFBaUIsWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNySyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxhQUFhLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsY0FBYyxDQUFDLENBQUM7SUFDbkosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQixDQUFDLFVBQWtCLEVBQUUsUUFBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDaEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGtCQUFtQixVQUFXLHFCQUFzQixRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQzFLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGtCQUFrQixDQUFDLFVBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxrQkFBbUIsVUFBVyxtQkFBbUIsQ0FBQyxDQUFDO0lBQzVKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQixDQUFDLG1CQUEyQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMseUJBQTBCLG1CQUFvQixFQUFFLENBQUMsQ0FBQztJQUMzSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9CQUFvQixDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUM3RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsdUJBQXVCLENBQUMsQ0FBQztJQUNsSSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLFNBQWlCLEVBQUUsWUFBb0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsZ0JBQWlCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDckssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFjLENBQUMsQ0FBQztJQUNuSixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksbUJBQW1CLENBQUMsU0FBaUIsRUFBRSxtQkFBMkIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsd0JBQXlCLG1CQUFvQixFQUFFLENBQUMsQ0FBQztJQUNwTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxvQkFBb0IsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxzQkFBc0IsQ0FBQyxDQUFDO0lBQzNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxzQkFBc0IsQ0FBQyxTQUFpQixFQUFFLHNCQUE4QixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNsSCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSwyQkFBNEIsc0JBQXVCLEVBQUUsQ0FBQyxDQUFDO0lBQzFMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLHVCQUF1QixDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLHlCQUF5QixDQUFDLENBQUM7SUFDOUosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BbUJHO0lBQ0ksWUFBWSxDQUFDLFVBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxrQkFBbUIsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUMzSSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxhQUFhLENBQUMsTUFBZSxFQUFFLFNBQWtCO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQzNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDZCQUE2QixDQUFDLFVBQWtCLEVBQUUsVUFBa0IsRUFBRSxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2xKLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSw2Q0FBOEMsVUFBVyxJQUFLLFVBQVcsSUFBSyxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3hPLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksOEJBQThCLENBQUMsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGdDQUFnQyxDQUFDLENBQUM7SUFDL0ssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGlCQUFpQixDQUFDLFNBQWlCLEVBQUUsWUFBb0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDbkcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUscUJBQXNCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDMUssQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksa0JBQWtCLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsbUJBQW1CLENBQUMsQ0FBQztJQUN4SixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGFBQWEsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsY0FBc0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsY0FBZSxVQUFXLFlBQWEsY0FBZSxjQUFjLENBQUMsQ0FBQztJQUN6TSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxxQkFBcUIsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsY0FBc0IsRUFBRSxVQUFrQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNqSixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxjQUFlLFVBQVcsWUFBYSxjQUFlLDBCQUEyQixVQUFXLEVBQUUsQ0FBQyxDQUFDO0lBQ25PLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxRQUFRLENBQUMsU0FBaUIsRUFBRSxVQUFrQixFQUFFLGNBQXNCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2hILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLGNBQWUsVUFBVyxZQUFhLGNBQWUsRUFBRSxDQUFDLENBQUM7SUFDN0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksZ0JBQWdCLENBQUMsU0FBaUIsRUFBRSxVQUFrQixFQUFFLGNBQXNCLEVBQUUsVUFBa0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsY0FBZSxVQUFXLFlBQWEsY0FBZSxjQUFlLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDdk4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE9BQU8sQ0FBQyxTQUFpQixFQUFFLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLFdBQVksU0FBVSxFQUFFLENBQUMsQ0FBQztJQUM3SixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxRQUFRLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsU0FBUyxDQUFDLENBQUM7SUFDOUksQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSx3QkFBd0IsQ0FBQyxVQUFrQixFQUFFLGdCQUF3QixFQUFFLFFBQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pJLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxrQkFBbUIsVUFBVyxJQUFLLGdCQUFpQixxQkFBc0IsUUFBUyxFQUFFLENBQUMsQ0FBQztJQUNoTSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kseUJBQXlCLENBQUMsVUFBa0IsRUFBRSxnQkFBd0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDaEgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGtCQUFtQixVQUFXLElBQUssZ0JBQWlCLG1CQUFtQixDQUFDLENBQUM7SUFDbEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLG1CQUFtQixDQUFDLFVBQWtCLEVBQUUsZ0JBQXdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDNUgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGlCQUFrQixVQUFXLElBQUssZ0JBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ3BNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQkFBa0IsQ0FBQyxTQUFpQixFQUFFLGtCQUEwQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMxRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxzQkFBdUIsa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQ2pMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLG1CQUFtQixDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLG9CQUFvQixDQUFDLENBQUM7SUFDekosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFVBQVUsQ0FBQyxTQUFpQixFQUFFLFVBQWtCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzFGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLGNBQWUsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUNqSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxXQUFXLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGVBQWdCLFNBQVUsWUFBWSxDQUFDLENBQUM7SUFDakosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BeUJHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZJLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxVQUFVLENBQUMsTUFBZSxFQUFFLFNBQWtCO1FBQ25ELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxhQUFhLENBQUMsQ0FBQztJQUN4SCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLEtBQUssQ0FBQyxTQUFpQixFQUFFLFNBQWlCLEVBQUUsT0FBZSxFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxXQUFZLFNBQVUsU0FBVSxPQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQy9LLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLE9BQU8sQ0FBQyxTQUFpQixFQUFFLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3RGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLFdBQVksU0FBVSxFQUFFLENBQUMsQ0FBQztJQUM3SixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFFBQVEsQ0FBQyxTQUFpQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNwRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsZUFBZ0IsU0FBVSxTQUFTLENBQUMsQ0FBQztJQUM5SSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxVQUFVLENBQUMsU0FBaUIsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsZUFBZ0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLHNCQUF1QixTQUFVLFdBQVcsQ0FBQyxDQUFDO0lBQ3ZKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFdBQVcsQ0FBQyxXQUFtQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsaUJBQWtCLFdBQVksRUFBRSxDQUFDLENBQUM7SUFDM0ksQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxZQUFZLENBQUMsTUFBZSxFQUFFLFNBQWtCO1FBQ3JELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFlLENBQUMsQ0FBQztJQUMxSCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksY0FBYyxDQUFDLFdBQW1CLEVBQUUsRUFBVSxFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN4RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsaUJBQWtCLFdBQVksU0FBVSxFQUFHLEVBQUUsQ0FBQyxDQUFDO0lBQ3hKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGVBQWUsQ0FBQyxXQUFtQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsaUJBQWtCLFdBQVksT0FBTyxDQUFDLENBQUM7SUFDaEosQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxlQUFnQixTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZJLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksVUFBVSxDQUFDLE1BQWUsRUFBRSxTQUFrQjtRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsYUFBYSxDQUFDLENBQUM7SUFDeEgsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksTUFBTSxDQUFDLHFCQUE2QixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsV0FBWSxxQkFBc0IsRUFBRSxDQUFDLENBQUM7SUFDL0ksQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxnQ0FBZ0MsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3JGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGlDQUFpQyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDN0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDdEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksNEJBQTRCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSx1QkFBdUIsQ0FBQyxLQUFlO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxDQUFDLE9BQU8sS0FBSyxLQUFLLFdBQVcsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNsRyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQkFBZ0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzVFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksdUJBQXVCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNuRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHNCQUFzQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDbEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDM0UsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksbUNBQW1DLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMvRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtDQUFrQyxDQUFDLEtBQWU7UUFDdkQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQzdHLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksc0JBQXNCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNsRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxxQkFBcUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzFFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLDBCQUEwQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDdEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDL0UsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksOEJBQThCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNuRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksK0JBQStCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNwRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSx1QkFBdUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksa0JBQWtCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM5RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN2RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGtDQUFrQyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDOUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDdkYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSw2QkFBNkIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksd0JBQXdCLENBQUMsS0FBZTtRQUM3QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDbkcsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUIsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzdFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3RFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLHdCQUF3QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDcEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksdUJBQXVCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNuRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9DQUFvQyxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDaEcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDdEYsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1DQUFtQyxDQUFDLEtBQWU7UUFDeEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLENBQUMsT0FBTyxLQUFLLEtBQUssV0FBVyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQzlHLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxzQkFBc0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ2xGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzNFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksMkJBQTJCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN2RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLCtCQUErQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDcEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0NBQWdDLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNyRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLHdCQUF3QixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDcEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDN0UsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXNERztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUEvaERSLGtCQUFhLEdBQUcsWUFBWSxDQUFDO1FBaUsxQixvQkFBZSxHQUFvQjtZQUMzQyx3QkFBd0IsRUFBRTtnQkFDeEIsd0JBQXdCO2dCQUN4Qix5QkFBeUI7Z0JBQ3pCLHlCQUF5QjtnQkFDekIsNkJBQTZCO2dCQUM3QixXQUFXO2dCQUNYLGtDQUFrQztnQkFDbEMsd0JBQXdCO2dCQUN4QixxQkFBcUI7YUFDdEI7WUFDRCxLQUFLLEVBQUU7Z0JBQ0wsUUFBUTtnQkFDUixPQUFPO2dCQUNQLE1BQU07Z0JBQ04sS0FBSzthQUNOO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLEtBQUs7YUFDTjtTQUNGLENBQUM7SUEyMkNGLENBQUM7Q0FDRjtBQWxpREQsZ0NBa2lEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYXBpZ2F0ZXdheV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uYXBpZ2F0ZXdheW1hbmFnZW1lbnQuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBBcGlnYXRld2F5IGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnYXBpZ2F0ZXdheSc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBjZXJ0aWZpY2F0ZXMgZm9yIG11dHVhbCBUTFMgYXV0aGVudGljYXRpb24gdG8gYSBkb21haW4gbmFtZS4gVGhpcyBpcyBhbiBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24gY29udHJvbCBmb3IgbWFuYWdpbmcgdGhlIERvbWFpbk5hbWUgcmVzb3VyY2UgZHVlIHRvIHRoZSBzZW5zaXRpdmUgbmF0dXJlIG9mIG1UTFNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWRkQ2VydGlmaWNhdGVUb0RvbWFpbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWRkQ2VydGlmaWNhdGVUb0RvbWFpbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBhY2Nlc3MgYXNzb2NpYXRpb24gZnJvbSBhbiBhY2Nlc3MgYXNzb2NpYXRpb24gc291cmNlIHRvIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGZvciBwcml2YXRlIEFQSXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQWNjZXNzQXNzb2NpYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUFjY2Vzc0Fzc29jaWF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgcGFydGljdWxhciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvREVMRVRFKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdERUxFVEUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZWFkIGEgcGFydGljdWxhciByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HRVQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dFVCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHBhcnRpY3VsYXIgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9PcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1BBVENIKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQQVRDSCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHBhcnRpY3VsYXIgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9PcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1BPU1QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1BPU1QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBwYXJ0aWN1bGFyIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfT3BlcmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QVVQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1BVVCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlamVjdCBhbiBleGlzdGluZyBhY2Nlc3MgYXNzb2NpYXRpb24gb3duZWQgYnkgYW5vdGhlciBhY2NvdW50IHRvIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGZvciBwcml2YXRlIEFQSXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmVqZWN0QWNjZXNzQXNzb2NpYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlamVjdEFjY2Vzc0Fzc29jaWF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIGNlcnRpZmljYXRlcyBmb3IgbXV0dWFsIFRMUyBhdXRoZW50aWNhdGlvbiBmcm9tIGEgZG9tYWluIG5hbWUuIFRoaXMgaXMgYW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIGNvbnRyb2wgZm9yIG1hbmFnaW5nIHRoZSBEb21haW5OYW1lIHJlc291cmNlIGR1ZSB0byB0aGUgc2Vuc2l0aXZlIG5hdHVyZSBvZiBtVExTXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9PcGVyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlbW92ZUNlcnRpZmljYXRlRnJvbURvbWFpbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVtb3ZlQ2VydGlmaWNhdGVGcm9tRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2V0IGEgV0FGIGFjY2VzcyBjb250cm9sIGxpc3QgKEFDTCkuIFRoaXMgaXMgYW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIGNvbnRyb2wgZm9yIG1hbmFnaW5nIHRoZSBTdGFnZSByZXNvdXJjZSBkdWUgdG8gdGhlIHNlbnNpdGl2ZSBuYXR1cmUgb2YgV2ViQWNsJ3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2V0V2ViQUNMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTZXRXZWJBQ0wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIG1hbmFnZW1lbnQgcG9saWN5IG9mIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGZvciBwcml2YXRlIEFQSXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRG9tYWluTmFtZU1hbmFnZW1lbnRQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZURvbWFpbk5hbWVNYW5hZ2VtZW50UG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBpbnZva2UgcG9saWN5IG9mIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGZvciBwcml2YXRlIEFQSXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRG9tYWluTmFtZVBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRG9tYWluTmFtZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIG1hbmFnZSB0aGUgSUFNIHJlc291cmNlIHBvbGljeSBmb3IgYW4gQVBJLiBUaGlzIGlzIGFuIGFkZGl0aW9uYWwgYXV0aG9yaXphdGlvbiBjb250cm9sIGZvciBtYW5hZ2luZyBhbiBBUEkgZHVlIHRvIHRoZSBzZW5zaXRpdmUgbmF0dXJlIG9mIHRoZSByZXNvdXJjZSBwb2xpY3lcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX09wZXJhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUmVzdEFwaVBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUmVzdEFwaVBvbGljeScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgICdQZXJtaXNzaW9ucyBtYW5hZ2VtZW50JzogW1xuICAgICAgJ0FkZENlcnRpZmljYXRlVG9Eb21haW4nLFxuICAgICAgJ0NyZWF0ZUFjY2Vzc0Fzc29jaWF0aW9uJyxcbiAgICAgICdSZWplY3RBY2Nlc3NBc3NvY2lhdGlvbicsXG4gICAgICAnUmVtb3ZlQ2VydGlmaWNhdGVGcm9tRG9tYWluJyxcbiAgICAgICdTZXRXZWJBQ0wnLFxuICAgICAgJ1VwZGF0ZURvbWFpbk5hbWVNYW5hZ2VtZW50UG9saWN5JyxcbiAgICAgICdVcGRhdGVEb21haW5OYW1lUG9saWN5JyxcbiAgICAgICdVcGRhdGVSZXN0QXBpUG9saWN5J1xuICAgIF0sXG4gICAgV3JpdGU6IFtcbiAgICAgICdERUxFVEUnLFxuICAgICAgJ1BBVENIJyxcbiAgICAgICdQT1NUJyxcbiAgICAgICdQVVQnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR0VUJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQWNjb3VudCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkFjY291bnQocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2FjY291bnRgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBBcGlLZXkgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9BcGlLZXkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBpS2V5SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBpS2V5SWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BcGlLZXkoYXBpS2V5SWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2FwaWtleXMvJHsgYXBpS2V5SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEFwaUtleXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9BcGlLZXkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQXBpS2V5cyhyZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovYXBpa2V5c2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEF1dGhvcml6ZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9BdXRob3JpemVyLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSBhdXRob3JpemVySWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXV0aG9yaXplcklkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0QXV0aG9yaXplclVyaSgpXG4gICAqIC0gLmlmUmVxdWVzdENvZ25pdG9Vc2VyUG9vbFByb3ZpZGVyQXJuKClcbiAgICogLSAuaWZSZXNvdXJjZUF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXNvdXJjZUF1dGhvcml6ZXJVcmkoKVxuICAgKiAtIC5pZlJlc291cmNlQ29nbml0b1VzZXJQb29sUHJvdmlkZXJBcm4oKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkF1dGhvcml6ZXIocmVzdEFwaUlkOiBzdHJpbmcsIGF1dGhvcml6ZXJJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vYXV0aG9yaXplcnMvJHsgYXV0aG9yaXplcklkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBBdXRob3JpemVycyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0F1dGhvcml6ZXIuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RBdXRob3JpemVyVHlwZSgpXG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJVcmkoKVxuICAgKiAtIC5pZlJlcXVlc3RDb2duaXRvVXNlclBvb2xQcm92aWRlckFybigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQXV0aG9yaXplcnMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9hdXRob3JpemVyc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEJhc2VQYXRoTWFwcGluZyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0Jhc2VQYXRoTWFwcGluZy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBiYXNlUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSBiYXNlUGF0aC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkJhc2VQYXRoTWFwcGluZyhkb21haW5OYW1lOiBzdHJpbmcsIGJhc2VQYXRoOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH0vYmFzZXBhdGhtYXBwaW5ncy8keyBiYXNlUGF0aCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQmFzZVBhdGhNYXBwaW5ncyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0Jhc2VQYXRoTWFwcGluZy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25CYXNlUGF0aE1hcHBpbmdzKGRvbWFpbk5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L2RvbWFpbm5hbWVzLyR7IGRvbWFpbk5hbWUgfS9iYXNlcGF0aG1hcHBpbmdzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQ2xpZW50Q2VydGlmaWNhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9DbGllbnRDZXJ0aWZpY2F0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjbGllbnRDZXJ0aWZpY2F0ZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsaWVudENlcnRpZmljYXRlSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25DbGllbnRDZXJ0aWZpY2F0ZShjbGllbnRDZXJ0aWZpY2F0ZUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9jbGllbnRjZXJ0aWZpY2F0ZXMvJHsgY2xpZW50Q2VydGlmaWNhdGVJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgQ2xpZW50Q2VydGlmaWNhdGVzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQ2xpZW50Q2VydGlmaWNhdGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2xpZW50Q2VydGlmaWNhdGVzKHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9jbGllbnRjZXJ0aWZpY2F0ZXNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEZXBsb3ltZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRGVwbG95bWVudC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gZGVwbG95bWVudElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRlcGxveW1lbnRJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRlcGxveW1lbnQocmVzdEFwaUlkOiBzdHJpbmcsIGRlcGxveW1lbnRJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZGVwbG95bWVudHMvJHsgZGVwbG95bWVudElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEZXBsb3ltZW50cyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RlcGxveW1lbnQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RTdGFnZU5hbWUoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRlcGxveW1lbnRzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZGVwbG95bWVudHNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEb2N1bWVudGF0aW9uUGFydCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvY3VtZW50YXRpb25QYXJ0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSBkb2N1bWVudGF0aW9uUGFydElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvY3VtZW50YXRpb25QYXJ0SWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb2N1bWVudGF0aW9uUGFydChyZXN0QXBpSWQ6IHN0cmluZywgZG9jdW1lbnRhdGlvblBhcnRJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZG9jdW1lbnRhdGlvbi9wYXJ0cy8keyBkb2N1bWVudGF0aW9uUGFydElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEb2N1bWVudGF0aW9uUGFydHMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Eb2N1bWVudGF0aW9uUGFydC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9jdW1lbnRhdGlvblBhcnRzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZG9jdW1lbnRhdGlvbi9wYXJ0c2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvY3VtZW50YXRpb25WZXJzaW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9jdW1lbnRhdGlvblZlcnNpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIGRvY3VtZW50YXRpb25WZXJzaW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9jdW1lbnRhdGlvblZlcnNpb25JZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvY3VtZW50YXRpb25WZXJzaW9uKHJlc3RBcGlJZDogc3RyaW5nLCBkb2N1bWVudGF0aW9uVmVyc2lvbklkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kb2N1bWVudGF0aW9uL3ZlcnNpb25zLyR7IGRvY3VtZW50YXRpb25WZXJzaW9uSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvY3VtZW50YXRpb25WZXJzaW9ucyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvY3VtZW50YXRpb25WZXJzaW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Eb2N1bWVudGF0aW9uVmVyc2lvbnMocmVzdEFwaUlkOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9kb2N1bWVudGF0aW9uL3ZlcnNpb25zYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRG9tYWluTmFtZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvbWFpbk5hbWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5OYW1lLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEVuZHBvaW50VHlwZSgpXG4gICAqIC0gLmlmUmVxdWVzdE10bHNUcnVzdFN0b3JlVXJpKClcbiAgICogLSAuaWZSZXF1ZXN0TXRsc1RydXN0U3RvcmVWZXJzaW9uKClcbiAgICogLSAuaWZSZXF1ZXN0U2VjdXJpdHlQb2xpY3koKVxuICAgKiAtIC5pZlJlc291cmNlRW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXNvdXJjZU10bHNUcnVzdFN0b3JlVXJpKClcbiAgICogLSAuaWZSZXNvdXJjZU10bHNUcnVzdFN0b3JlVmVyc2lvbigpXG4gICAqIC0gLmlmUmVzb3VyY2VTZWN1cml0eVBvbGljeSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9tYWluTmFtZShkb21haW5OYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEb21haW5OYW1lcyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvbWFpbk5hbWUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEVuZHBvaW50VHlwZSgpXG4gICAqIC0gLmlmUmVxdWVzdE10bHNUcnVzdFN0b3JlVXJpKClcbiAgICogLSAuaWZSZXF1ZXN0TXRsc1RydXN0U3RvcmVWZXJzaW9uKClcbiAgICogLSAuaWZSZXF1ZXN0U2VjdXJpdHlQb2xpY3koKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvbWFpbk5hbWVzKHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9kb21haW5uYW1lc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIERvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0RvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBzb3VyY2VUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIHNvdXJjZVR5cGUuXG4gICAqIEBwYXJhbSBzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25Eb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb24oZG9tYWluTmFtZTogc3RyaW5nLCBzb3VyY2VUeXBlOiBzdHJpbmcsIHNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06L2RvbWFpbm5hbWVhY2Nlc3Nhc3NvY2lhdGlvbnMvZG9tYWlubmFtZS8keyBkb21haW5OYW1lIH0vJHsgc291cmNlVHlwZSB9LyR7IHNvdXJjZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBEb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb25zIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFjY2Vzc0Fzc29jaWF0aW9uU291cmNlKClcbiAgICogLSAuaWZSZXF1ZXN0RG9tYWluTmFtZUFybigpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9ucyhhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9kb21haW5uYW1lYWNjZXNzYXNzb2NpYXRpb25zYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgR2F0ZXdheVJlc3BvbnNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfR2F0ZXdheVJlc3BvbnNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZXNwb25zZVR5cGUgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzcG9uc2VUeXBlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uR2F0ZXdheVJlc3BvbnNlKHJlc3RBcGlJZDogc3RyaW5nLCByZXNwb25zZVR5cGU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L2dhdGV3YXlyZXNwb25zZXMvJHsgcmVzcG9uc2VUeXBlIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBHYXRld2F5UmVzcG9uc2VzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfR2F0ZXdheVJlc3BvbnNlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25HYXRld2F5UmVzcG9uc2VzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vZ2F0ZXdheXJlc3BvbnNlc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEludGVncmF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfSW50ZWdyYXRpb24uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGh0dHBNZXRob2RUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIGh0dHBNZXRob2RUeXBlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW50ZWdyYXRpb24ocmVzdEFwaUlkOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgaHR0cE1ldGhvZFR5cGU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3Jlc291cmNlcy8keyByZXNvdXJjZUlkIH0vbWV0aG9kcy8keyBodHRwTWV0aG9kVHlwZSB9L2ludGVncmF0aW9uYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgSW50ZWdyYXRpb25SZXNwb25zZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0ludGVncmF0aW9uUmVzcG9uc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGh0dHBNZXRob2RUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIGh0dHBNZXRob2RUeXBlLlxuICAgKiBAcGFyYW0gc3RhdHVzQ29kZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzdGF0dXNDb2RlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uSW50ZWdyYXRpb25SZXNwb25zZShyZXN0QXBpSWQ6IHN0cmluZywgcmVzb3VyY2VJZDogc3RyaW5nLCBodHRwTWV0aG9kVHlwZTogc3RyaW5nLCBzdGF0dXNDb2RlOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9yZXNvdXJjZXMvJHsgcmVzb3VyY2VJZCB9L21ldGhvZHMvJHsgaHR0cE1ldGhvZFR5cGUgfS9pbnRlZ3JhdGlvbi9yZXNwb25zZXMvJHsgc3RhdHVzQ29kZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTWV0aG9kIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfTWV0aG9kLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZXNvdXJjZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc291cmNlSWQuXG4gICAqIEBwYXJhbSBodHRwTWV0aG9kVHlwZSAtIElkZW50aWZpZXIgZm9yIHRoZSBodHRwTWV0aG9kVHlwZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RBcGlLZXlSZXF1aXJlZCgpXG4gICAqIC0gLmlmUmVxdWVzdFJvdXRlQXV0aG9yaXphdGlvblR5cGUoKVxuICAgKiAtIC5pZlJlc291cmNlQXBpS2V5UmVxdWlyZWQoKVxuICAgKiAtIC5pZlJlc291cmNlUm91dGVBdXRob3JpemF0aW9uVHlwZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTWV0aG9kKHJlc3RBcGlJZDogc3RyaW5nLCByZXNvdXJjZUlkOiBzdHJpbmcsIGh0dHBNZXRob2RUeXBlOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9yZXNvdXJjZXMvJHsgcmVzb3VyY2VJZCB9L21ldGhvZHMvJHsgaHR0cE1ldGhvZFR5cGUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIE1ldGhvZFJlc3BvbnNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfTWV0aG9kUmVzcG9uc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGh0dHBNZXRob2RUeXBlIC0gSWRlbnRpZmllciBmb3IgdGhlIGh0dHBNZXRob2RUeXBlLlxuICAgKiBAcGFyYW0gc3RhdHVzQ29kZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzdGF0dXNDb2RlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTWV0aG9kUmVzcG9uc2UocmVzdEFwaUlkOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgaHR0cE1ldGhvZFR5cGU6IHN0cmluZywgc3RhdHVzQ29kZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVzb3VyY2VzLyR7IHJlc291cmNlSWQgfS9tZXRob2RzLyR7IGh0dHBNZXRob2RUeXBlIH0vcmVzcG9uc2VzLyR7IHN0YXR1c0NvZGUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIE1vZGVsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfTW9kZWwuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIG1vZGVsTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBtb2RlbE5hbWUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Nb2RlbChyZXN0QXBpSWQ6IHN0cmluZywgbW9kZWxOYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9tb2RlbHMvJHsgbW9kZWxOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBNb2RlbHMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Nb2RlbC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTW9kZWxzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vbW9kZWxzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUHJpdmF0ZUJhc2VQYXRoTWFwcGluZyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0Jhc2VQYXRoTWFwcGluZy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbk5hbWUuXG4gICAqIEBwYXJhbSBkb21haW5JZGVudGlmaWVyIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbklkZW50aWZpZXIuXG4gICAqIEBwYXJhbSBiYXNlUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSBiYXNlUGF0aC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByaXZhdGVCYXNlUGF0aE1hcHBpbmcoZG9tYWluTmFtZTogc3RyaW5nLCBkb21haW5JZGVudGlmaWVyOiBzdHJpbmcsIGJhc2VQYXRoOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH0rJHsgZG9tYWluSWRlbnRpZmllciB9L2Jhc2VwYXRobWFwcGluZ3MvJHsgYmFzZVBhdGggfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFByaXZhdGVCYXNlUGF0aE1hcHBpbmdzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfQmFzZVBhdGhNYXBwaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIGRvbWFpbklkZW50aWZpZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWRlbnRpZmllci5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByaXZhdGVCYXNlUGF0aE1hcHBpbmdzKGRvbWFpbk5hbWU6IHN0cmluZywgZG9tYWluSWRlbnRpZmllcjogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovZG9tYWlubmFtZXMvJHsgZG9tYWluTmFtZSB9KyR7IGRvbWFpbklkZW50aWZpZXIgfS9iYXNlcGF0aG1hcHBpbmdzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUHJpdmF0ZURvbWFpbk5hbWUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9Eb21haW5OYW1lLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluTmFtZS5cbiAgICogQHBhcmFtIGRvbWFpbklkZW50aWZpZXIgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWRlbnRpZmllci5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEVuZHBvaW50VHlwZSgpXG4gICAqIC0gLmlmUmVzb3VyY2VFbmRwb2ludFR5cGUoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByaXZhdGVEb21haW5OYW1lKGRvbWFpbk5hbWU6IHN0cmluZywgZG9tYWluSWRlbnRpZmllcjogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9kb21haW5uYW1lcy8keyBkb21haW5OYW1lIH0rJHsgZG9tYWluSWRlbnRpZmllciB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUmVxdWVzdFZhbGlkYXRvciB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1JlcXVlc3RWYWxpZGF0b3IuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlcXVlc3RWYWxpZGF0b3JJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXF1ZXN0VmFsaWRhdG9ySWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXF1ZXN0VmFsaWRhdG9yKHJlc3RBcGlJZDogc3RyaW5nLCByZXF1ZXN0VmFsaWRhdG9ySWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3JlcXVlc3R2YWxpZGF0b3JzLyR7IHJlcXVlc3RWYWxpZGF0b3JJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUmVxdWVzdFZhbGlkYXRvcnMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9SZXF1ZXN0VmFsaWRhdG9yLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZXF1ZXN0VmFsaWRhdG9ycyhyZXN0QXBpSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3JlcXVlc3R2YWxpZGF0b3JzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUmVzb3VyY2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9SZXNvdXJjZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVzb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXNvdXJjZUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVzb3VyY2UocmVzdEFwaUlkOiBzdHJpbmcsIHJlc291cmNlSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3Jlc291cmNlcy8keyByZXNvdXJjZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBSZXNvdXJjZXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9SZXNvdXJjZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVzb3VyY2VzKHJlc3RBcGlJZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vcmVzb3VyY2VzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUmVzdEFwaSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1Jlc3RBcGkuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gcmVzdEFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIHJlc3RBcGlJZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RBcGlLZXlSZXF1aXJlZCgpXG4gICAqIC0gLmlmUmVxdWVzdEFwaU5hbWUoKVxuICAgKiAtIC5pZlJlcXVlc3RBdXRob3JpemVyVHlwZSgpXG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJVcmkoKVxuICAgKiAtIC5pZlJlcXVlc3REaXNhYmxlRXhlY3V0ZUFwaUVuZHBvaW50KClcbiAgICogLSAuaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0Um91dGVBdXRob3JpemF0aW9uVHlwZSgpXG4gICAqIC0gLmlmUmVzb3VyY2VBcGlLZXlSZXF1aXJlZCgpXG4gICAqIC0gLmlmUmVzb3VyY2VBcGlOYW1lKClcbiAgICogLSAuaWZSZXNvdXJjZUF1dGhvcml6ZXJUeXBlKClcbiAgICogLSAuaWZSZXNvdXJjZUF1dGhvcml6ZXJVcmkoKVxuICAgKiAtIC5pZlJlc291cmNlRGlzYWJsZUV4ZWN1dGVBcGlFbmRwb2ludCgpXG4gICAqIC0gLmlmUmVzb3VyY2VFbmRwb2ludFR5cGUoKVxuICAgKiAtIC5pZlJlc291cmNlUm91dGVBdXRob3JpemF0aW9uVHlwZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVzdEFwaShyZXN0QXBpSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUmVzdEFwaXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9SZXN0QXBpLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RBcGlLZXlSZXF1aXJlZCgpXG4gICAqIC0gLmlmUmVxdWVzdEFwaU5hbWUoKVxuICAgKiAtIC5pZlJlcXVlc3RBdXRob3JpemVyVHlwZSgpXG4gICAqIC0gLmlmUmVxdWVzdEF1dGhvcml6ZXJVcmkoKVxuICAgKiAtIC5pZlJlcXVlc3REaXNhYmxlRXhlY3V0ZUFwaUVuZHBvaW50KClcbiAgICogLSAuaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKClcbiAgICogLSAuaWZSZXF1ZXN0Um91dGVBdXRob3JpemF0aW9uVHlwZSgpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVzdEFwaXMocmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU2RrIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc3RBcGlJZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZXN0QXBpSWQuXG4gICAqIEBwYXJhbSBzdGFnZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc3RhZ2VOYW1lLlxuICAgKiBAcGFyYW0gc2RrVHlwZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzZGtUeXBlLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2RrKHJlc3RBcGlJZDogc3RyaW5nLCBzdGFnZU5hbWU6IHN0cmluZywgc2RrVHlwZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovcmVzdGFwaXMvJHsgcmVzdEFwaUlkIH0vc3RhZ2VzLyR7IHN0YWdlTmFtZSB9L3Nka3MvJHsgc2RrVHlwZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU3RhZ2UgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9TdGFnZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gc3RhZ2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YWdlTmFtZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlJlcXVlc3RBY2Nlc3NMb2dnaW5nRGVzdGluYXRpb24oKVxuICAgKiAtIC5pZlJlcXVlc3RBY2Nlc3NMb2dnaW5nRm9ybWF0KClcbiAgICogLSAuaWZSZXNvdXJjZUFjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbigpXG4gICAqIC0gLmlmUmVzb3VyY2VBY2Nlc3NMb2dnaW5nRm9ybWF0KClcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TdGFnZShyZXN0QXBpSWQ6IHN0cmluZywgc3RhZ2VOYW1lOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi9yZXN0YXBpcy8keyByZXN0QXBpSWQgfS9zdGFnZXMvJHsgc3RhZ2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBTdGFnZXMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9TdGFnZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZXN0QXBpSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzdEFwaUlkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbigpXG4gICAqIC0gLmlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdGb3JtYXQoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblN0YWdlcyhyZXN0QXBpSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzLyR7IHJlc3RBcGlJZCB9L3N0YWdlc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFRlbXBsYXRlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG1vZGVsTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBtb2RlbE5hbWUuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25UZW1wbGF0ZShtb2RlbE5hbWU6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3Jlc3RhcGlzL21vZGVscy8keyBtb2RlbE5hbWUgfS90ZW1wbGF0ZWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFVzYWdlUGxhbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1VzYWdlUGxhbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB1c2FnZVBsYW5JZCAtIElkZW50aWZpZXIgZm9yIHRoZSB1c2FnZVBsYW5JZC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblVzYWdlUGxhbih1c2FnZVBsYW5JZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovdXNhZ2VwbGFucy8keyB1c2FnZVBsYW5JZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVXNhZ2VQbGFucyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1VzYWdlUGxhbi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Vc2FnZVBsYW5zKHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi91c2FnZXBsYW5zYCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVXNhZ2VQbGFuS2V5IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfVXNhZ2VQbGFuS2V5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHVzYWdlUGxhbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHVzYWdlUGxhbklkLlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Vc2FnZVBsYW5LZXkodXNhZ2VQbGFuSWQ6IHN0cmluZywgaWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3VzYWdlcGxhbnMvJHsgdXNhZ2VQbGFuSWQgfS9rZXlzLyR7IGlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBVc2FnZVBsYW5LZXlzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfVXNhZ2VQbGFuS2V5Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHVzYWdlUGxhbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIHVzYWdlUGxhbklkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVXNhZ2VQbGFuS2V5cyh1c2FnZVBsYW5JZDogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjovdXNhZ2VwbGFucy8keyB1c2FnZVBsYW5JZCB9L2tleXNgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBWcGNMaW5rIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2FwaS9BUElfVnBjTGluay5odG1sXG4gICAqXG4gICAqIEBwYXJhbSB2cGNMaW5rSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgdnBjTGlua0lkLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uVnBjTGluayh2cGNMaW5rSWQ6IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcGlnYXRld2F5OiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTo6L3ZwY2xpbmtzLyR7IHZwY0xpbmtJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgVnBjTGlua3MgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9WcGNMaW5rLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblZwY0xpbmtzKHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi92cGNsaW5rc2ApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFRhZ3MgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYXBpZ2F0ZXdheS10YWdnaW5nLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHVybEVuY29kZWRSZXNvdXJjZUFSTiAtIElkZW50aWZpZXIgZm9yIHRoZSB1cmxFbmNvZGVkUmVzb3VyY2VBUk4uXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblRhZ3ModXJsRW5jb2RlZFJlc291cmNlQVJOOiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBpZ2F0ZXdheTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06Oi90YWdzLyR7IHVybEVuY29kZWRSZXNvdXJjZUFSTiB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYWNjZXNzIGFzc29jaWF0aW9uIHNvdXJjZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uIG9wZXJhdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVBY2Nlc3NBc3NvY2lhdGlvbnNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0QWNjZXNzQXNzb2NpYXRpb25Tb3VyY2UodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvQWNjZXNzQXNzb2NpYXRpb25Tb3VyY2VgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhY2Nlc3MgbG9nIGRlc3RpbmF0aW9uLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVTdGFnZSBhbmQgVXBkYXRlU3RhZ2Ugb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFN0YWdlXG4gICAqIC0gU3RhZ2VzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdEFjY2Vzc0xvZ2dpbmdEZXN0aW5hdGlvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9BY2Nlc3NMb2dnaW5nRGVzdGluYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhY2Nlc3MgbG9nIGZvcm1hdC4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlU3RhZ2UgYW5kIFVwZGF0ZVN0YWdlIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBTdGFnZVxuICAgKiAtIFN0YWdlc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBY2Nlc3NMb2dnaW5nRm9ybWF0KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0FjY2Vzc0xvZ2dpbmdGb3JtYXRgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB3aGV0aGVyIGFuIEFQSSBrZXkgaXMgcmVxdWlyZWQgb3Igbm90LiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVNZXRob2QgYW5kIFB1dE1ldGhvZCBvcGVyYXRpb25zLiBBbHNvIGF2YWlsYWJsZSBhcyBhIGNvbGxlY3Rpb24gZHVyaW5nIGltcG9ydCBhbmQgcmVpbXBvcnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBNZXRob2RcbiAgICogLSBSZXN0QXBpXG4gICAqIC0gUmVzdEFwaXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0QXBpS2V5UmVxdWlyZWQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvQXBpS2V5UmVxdWlyZWRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgQVBJIG5hbWUuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZVJlc3RBcGkgYW5kIFVwZGF0ZVJlc3RBcGkgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBcGlOYW1lKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0FwaU5hbWVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0eXBlIG9mIGF1dGhvcml6ZXIgaW4gdGhlIHJlcXVlc3QsIGZvciBleGFtcGxlIFRPS0VOLCBSRVFVRVNULCBKV1QuIEF2YWlsYWJsZSBkdXJpbmcgQ3JlYXRlQXV0aG9yaXplciBhbmQgVXBkYXRlQXV0aG9yaXplci4gQWxzbyBhdmFpbGFibGUgZHVyaW5nIGltcG9ydCBhbmQgcmVpbXBvcnQgYXMgYW4gQXJyYXlPZlN0cmluZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEF1dGhvcml6ZXJcbiAgICogLSBBdXRob3JpemVyc1xuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RBdXRob3JpemVyVHlwZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9BdXRob3JpemVyVHlwZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IFVSSSBvZiBhIExhbWJkYSBhdXRob3JpemVyIGZ1bmN0aW9uLiBBdmFpbGFibGUgZHVyaW5nIENyZWF0ZUF1dGhvcml6ZXIgYW5kIFVwZGF0ZUF1dGhvcml6ZXIuIEFsc28gYXZhaWxhYmxlIGR1cmluZyBpbXBvcnQgYW5kIHJlaW1wb3J0IGFzIGFuIEFycmF5T2ZTdHJpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBBdXRob3JpemVyXG4gICAqIC0gQXV0aG9yaXplcnNcbiAgICogLSBSZXN0QXBpXG4gICAqIC0gUmVzdEFwaXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0QXV0aG9yaXplclVyaSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVxdWVzdC9BdXRob3JpemVyVXJpYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgQ29nbml0byB1c2VyIHBvb2wgcHJvdmlkZXIgQVJOLiBBdmFpbGFibGUgZHVyaW5nIENyZWF0ZUF1dGhvcml6ZXIgYW5kIFVwZGF0ZUF1dGhvcml6ZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBBdXRob3JpemVyXG4gICAqIC0gQXV0aG9yaXplcnNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0Q29nbml0b1VzZXJQb29sUHJvdmlkZXJBcm4odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvQ29nbml0b1VzZXJQb29sUHJvdmlkZXJBcm5gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBzdGF0dXMgb2YgdGhlIGRlZmF1bHQgZXhlY3V0ZS1hcGkgZW5kcG9pbnQuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZVJlc3RBcGkgYW5kIERlbGV0ZVJlc3RBcGkgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3REaXNhYmxlRXhlY3V0ZUFwaUVuZHBvaW50KHZhbHVlPzogYm9vbGVhbikge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0Rpc2FibGVFeGVjdXRlQXBpRW5kcG9pbnRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgZG9tYWluIG5hbWUgQVJOLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVEb21haW5OYW1lQWNjZXNzQXNzb2NpYXRpb24gb3BlcmF0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3REb21haW5OYW1lQXJuKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0RvbWFpbk5hbWVBcm5gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBlbmRwb2ludCB0eXBlLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVEb21haW5OYW1lLCBVcGRhdGVEb21haW5OYW1lLCBDcmVhdGVSZXN0QXBpLCBhbmQgVXBkYXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKiAtIERvbWFpbk5hbWVzXG4gICAqIC0gUHJpdmF0ZURvbWFpbk5hbWVcbiAgICogLSBSZXN0QXBpXG4gICAqIC0gUmVzdEFwaXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0RW5kcG9pbnRUeXBlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L0VuZHBvaW50VHlwZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IFVSSSBvZiB0aGUgdHJ1c3RzdG9yZSB1c2VkIGZvciBtdXR1YWwgVExTIGF1dGhlbnRpY2F0aW9uLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVEb21haW5OYW1lIGFuZCBVcGRhdGVEb21haW5OYW1lIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEb21haW5OYW1lXG4gICAqIC0gRG9tYWluTmFtZXNcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXF1ZXN0TXRsc1RydXN0U3RvcmVVcmkodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvTXRsc1RydXN0U3RvcmVVcmlgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB2ZXJzaW9uIG9mIHRoZSB0cnVzdHN0b3JlIHVzZWQgZm9yIG11dHVhbCBUTFMgYXV0aGVudGljYXRpb24uIEF2YWlsYWJsZSBkdXJpbmcgdGhlIENyZWF0ZURvbWFpbk5hbWUgYW5kIFVwZGF0ZURvbWFpbk5hbWUgb3BlcmF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIERvbWFpbk5hbWVcbiAgICogLSBEb21haW5OYW1lc1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RNdGxzVHJ1c3RTdG9yZVZlcnNpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvTXRsc1RydXN0U3RvcmVWZXJzaW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYXV0aG9yaXphdGlvbiB0eXBlLCBmb3IgZXhhbXBsZSBOT05FLCBBV1NfSUFNLCBDVVNUT00sIEpXVCwgQ09HTklUT19VU0VSX1BPT0xTLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVNZXRob2QgYW5kIFB1dE1ldGhvZCBvcGVyYXRpb25zIEFsc28gYXZhaWxhYmxlIGFzIGEgY29sbGVjdGlvbiBkdXJpbmcgaW1wb3J0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gTWV0aG9kXG4gICAqIC0gUmVzdEFwaVxuICAgKiAtIFJlc3RBcGlzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdFJvdXRlQXV0aG9yaXphdGlvblR5cGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvUm91dGVBdXRob3JpemF0aW9uVHlwZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IFRMUyB2ZXJzaW9uLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBDcmVhdGVEb21haW4gYW5kIFVwZGF0ZURvbWFpbiBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKiAtIERvbWFpbk5hbWVzXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVxdWVzdFNlY3VyaXR5UG9saWN5KHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXF1ZXN0L1NlY3VyaXR5UG9saWN5YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgc3RhZ2UgbmFtZSBvZiB0aGUgZGVwbG95bWVudCB0aGF0IHlvdSBhdHRlbXB0IHRvIGNyZWF0ZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgQ3JlYXRlRGVwbG95bWVudCBvcGVyYXRpb25cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEZXBsb3ltZW50c1xuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlcXVlc3RTdGFnZU5hbWUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlcXVlc3QvU3RhZ2VOYW1lYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYWNjZXNzIGxvZyBkZXN0aW5hdGlvbiBvZiB0aGUgY3VycmVudCBTdGFnZSByZXNvdXJjZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgVXBkYXRlU3RhZ2UgYW5kIERlbGV0ZVN0YWdlIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBTdGFnZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQWNjZXNzTG9nZ2luZ0Rlc3RpbmF0aW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9BY2Nlc3NMb2dnaW5nRGVzdGluYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhY2Nlc3MgbG9nIGZvcm1hdCBvZiB0aGUgY3VycmVudCBTdGFnZSByZXNvdXJjZS4gQXZhaWxhYmxlIGR1cmluZyB0aGUgVXBkYXRlU3RhZ2UgYW5kIERlbGV0ZVN0YWdlIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBTdGFnZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQWNjZXNzTG9nZ2luZ0Zvcm1hdCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQWNjZXNzTG9nZ2luZ0Zvcm1hdGAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHdoZXRoZXIgYW4gQVBJIGtleSBpcyByZXF1aXJlZCBvciBub3QgZm9yIHRoZSBleGlzdGluZyBNZXRob2QgcmVzb3VyY2UuIEF2YWlsYWJsZSBkdXJpbmcgdGhlIFB1dE1ldGhvZCBhbmQgRGVsZXRlTWV0aG9kIG9wZXJhdGlvbnMuIEFsc28gYXZhaWxhYmxlIGFzIGEgY29sbGVjdGlvbiBkdXJpbmcgcmVpbXBvcnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBNZXRob2RcbiAgICogLSBSZXN0QXBpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBgdHJ1ZWAgb3IgYGZhbHNlYC4gKipEZWZhdWx0OioqIGB0cnVlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VBcGlLZXlSZXF1aXJlZCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQXBpS2V5UmVxdWlyZWRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgQVBJIG5hbWUgb2YgdGhlIGV4aXN0aW5nIFJlc3RBcGkgcmVzb3VyY2UuIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlUmVzdEFwaSBhbmQgRGVsZXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gUmVzdEFwaVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQXBpTmFtZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQXBpTmFtZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBjdXJyZW50IHR5cGUgb2YgYXV0aG9yaXplciwgZm9yIGV4YW1wbGUgVE9LRU4sIFJFUVVFU1QsIEpXVC4gQXZhaWxhYmxlIGR1cmluZyBVcGRhdGVBdXRob3JpemVyIGFuZCBEZWxldGVBdXRob3JpemVyIG9wZXJhdGlvbnMuIEFsc28gYXZhaWxhYmxlIGR1cmluZyByZWltcG9ydCBhcyBhbiBBcnJheU9mU3RyaW5nXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQXV0aG9yaXplclxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZUF1dGhvcml6ZXJUeXBlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9BdXRob3JpemVyVHlwZWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IFVSSSBvZiBhIExhbWJkYSBhdXRob3JpemVyIGZ1bmN0aW9uLiBBdmFpbGFibGUgZHVyaW5nIFVwZGF0ZUF1dGhvcml6ZXIgYW5kIERlbGV0ZUF1dGhvcml6ZXIgb3BlcmF0aW9ucy4gQWxzbyBhdmFpbGFibGUgZHVyaW5nIHJlaW1wb3J0IGFzIGFuIEFycmF5T2ZTdHJpbmdcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBBdXRob3JpemVyXG4gICAqIC0gUmVzdEFwaVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQXV0aG9yaXplclVyaSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvQXV0aG9yaXplclVyaWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IENvZ25pdG8gdXNlciBwb29sIHByb3ZpZGVyIEFSTi4gQXZhaWxhYmxlIGR1cmluZyBDcmVhdGVBdXRob3JpemVyIGFuZCBVcGRhdGVBdXRob3JpemVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gQXV0aG9yaXplclxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlQ29nbml0b1VzZXJQb29sUHJvdmlkZXJBcm4odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL0NvZ25pdG9Vc2VyUG9vbFByb3ZpZGVyQXJuYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdBcm5MaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgc3RhdHVzIG9mIHRoZSBkZWZhdWx0IGV4ZWN1dGUtYXBpIGVuZHBvaW50IG9mIHRoZSBjdXJyZW50IFJlc3RBcGkgcmVzb3VyY2UuIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlUmVzdEFwaSBhbmQgRGVsZXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gUmVzdEFwaVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlRGlzYWJsZUV4ZWN1dGVBcGlFbmRwb2ludCh2YWx1ZT86IGJvb2xlYW4pIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvRGlzYWJsZUV4ZWN1dGVBcGlFbmRwb2ludGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBlbmRwb2ludCB0eXBlLiBBdmFpbGFibGUgZHVyaW5nIHRoZSBVcGRhdGVEb21haW5OYW1lLCBEZWxldGVEb21haW5OYW1lLCBVcGRhdGVSZXN0QXBpLCBhbmQgRGVsZXRlUmVzdEFwaSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKiAtIFByaXZhdGVEb21haW5OYW1lXG4gICAqIC0gUmVzdEFwaVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlRW5kcG9pbnRUeXBlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9FbmRwb2ludFR5cGVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBVUkkgb2YgdGhlIHRydXN0c3RvcmUgdXNlZCBmb3IgbXV0dWFsIFRMUyBhdXRoZW50aWNhdGlvbi4gQXZhaWxhYmxlIGR1cmluZyBVcGRhdGVEb21haW5OYW1lIGFuZCBEZWxldGVEb21haW5OYW1lIG9wZXJhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBEb21haW5OYW1lXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmUmVzb3VyY2VNdGxzVHJ1c3RTdG9yZVVyaSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgUmVzb3VyY2UvTXRsc1RydXN0U3RvcmVVcmlgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB2ZXJzaW9uIG9mIHRoZSB0cnVzdHN0b3JlIHVzZWQgZm9yIG11dHVhbCBUTFMgYXV0aGVudGljYXRpb24uIEF2YWlsYWJsZSBkdXJpbmcgVXBkYXRlRG9tYWluTmFtZSBhbmQgRGVsZXRlRG9tYWluTmFtZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlTXRsc1RydXN0U3RvcmVWZXJzaW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBSZXNvdXJjZS9NdGxzVHJ1c3RTdG9yZVZlcnNpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhdXRob3JpemF0aW9uIHR5cGUgb2YgdGhlIGV4aXN0aW5nIE1ldGhvZCByZXNvdXJjZSwgZm9yIGV4YW1wbGUgTk9ORSwgQVdTX0lBTSwgQ1VTVE9NLCBKV1QsIENPR05JVE9fVVNFUl9QT09MUy4gQXZhaWxhYmxlIGR1cmluZyB0aGUgUHV0TWV0aG9kIGFuZCBEZWxldGVNZXRob2Qgb3BlcmF0aW9ucy4gQWxzbyBhdmFpbGFibGUgYXMgYSBjb2xsZWN0aW9uIGR1cmluZyByZWltcG9ydFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIE1ldGhvZFxuICAgKiAtIFJlc3RBcGlcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZSZXNvdXJjZVJvdXRlQXV0aG9yaXphdGlvblR5cGUodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL1JvdXRlQXV0aG9yaXphdGlvblR5cGVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBUTFMgdmVyc2lvbi4gQXZhaWxhYmxlIGR1cmluZyBVcGRhdGVEb21haW4gYW5kIERlbGV0ZURvbWFpbiBvcGVyYXRpb25zXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9zZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gRG9tYWluTmFtZVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZlJlc291cmNlU2VjdXJpdHlQb2xpY3kodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYFJlc291cmNlL1NlY3VyaXR5UG9saWN5YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaWdhdGV3YXktdGFnZ2luZy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9ERUxFVEUoKVxuICAgKiAtIC50b1BBVENIKClcbiAgICogLSAudG9QT1NUKClcbiAgICogLSAudG9QVVQoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZ3MgYXR0YWNoZWQgdG8gdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGlnYXRld2F5LXRhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIEFwaUtleVxuICAgKiAtIEFwaUtleXNcbiAgICogLSBBdXRob3JpemVyXG4gICAqIC0gQXV0aG9yaXplcnNcbiAgICogLSBCYXNlUGF0aE1hcHBpbmdcbiAgICogLSBCYXNlUGF0aE1hcHBpbmdzXG4gICAqIC0gQ2xpZW50Q2VydGlmaWNhdGVcbiAgICogLSBDbGllbnRDZXJ0aWZpY2F0ZXNcbiAgICogLSBEZXBsb3ltZW50XG4gICAqIC0gRGVwbG95bWVudHNcbiAgICogLSBEb2N1bWVudGF0aW9uUGFydFxuICAgKiAtIERvY3VtZW50YXRpb25QYXJ0c1xuICAgKiAtIERvY3VtZW50YXRpb25WZXJzaW9uXG4gICAqIC0gRG9jdW1lbnRhdGlvblZlcnNpb25zXG4gICAqIC0gRG9tYWluTmFtZVxuICAgKiAtIERvbWFpbk5hbWVzXG4gICAqIC0gRG9tYWluTmFtZUFjY2Vzc0Fzc29jaWF0aW9uc1xuICAgKiAtIEdhdGV3YXlSZXNwb25zZVxuICAgKiAtIEdhdGV3YXlSZXNwb25zZXNcbiAgICogLSBJbnRlZ3JhdGlvblxuICAgKiAtIEludGVncmF0aW9uUmVzcG9uc2VcbiAgICogLSBNZXRob2RcbiAgICogLSBNZXRob2RSZXNwb25zZVxuICAgKiAtIE1vZGVsXG4gICAqIC0gTW9kZWxzXG4gICAqIC0gUHJpdmF0ZUJhc2VQYXRoTWFwcGluZ1xuICAgKiAtIFByaXZhdGVCYXNlUGF0aE1hcHBpbmdzXG4gICAqIC0gUHJpdmF0ZURvbWFpbk5hbWVcbiAgICogLSBSZXF1ZXN0VmFsaWRhdG9yXG4gICAqIC0gUmVxdWVzdFZhbGlkYXRvcnNcbiAgICogLSBSZXNvdXJjZVxuICAgKiAtIFJlc291cmNlc1xuICAgKiAtIFJlc3RBcGlcbiAgICogLSBSZXN0QXBpc1xuICAgKiAtIFNka1xuICAgKiAtIFN0YWdlXG4gICAqIC0gU3RhZ2VzXG4gICAqIC0gVGVtcGxhdGVcbiAgICogLSBVc2FnZVBsYW5cbiAgICogLSBVc2FnZVBsYW5zXG4gICAqIC0gVXNhZ2VQbGFuS2V5XG4gICAqIC0gVXNhZ2VQbGFuS2V5c1xuICAgKiAtIFZwY0xpbmtcbiAgICogLSBWcGNMaW5rc1xuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGtleXMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaWdhdGV3YXktdGFnZ2luZy5odG1sXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9ERUxFVEUoKVxuICAgKiAtIC50b1BBVENIKClcbiAgICogLSAudG9QT1NUKClcbiAgICogLSAudG9QVVQoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthcGlnYXRld2F5XShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hbWF6b25hcGlnYXRld2F5bWFuYWdlbWVudC5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=