"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Apprunner = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [apprunner](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsapprunner.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Apprunner extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate your own domain name with the AWS App Runner subdomain URL of your App Runner service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_AssociateCustomDomain.html
     */
    toAssociateCustomDomain() {
        return this.to('AssociateCustomDomain');
    }
    /**
     * Grants permission to associate the service with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/dg/waf-manage.html
     */
    toAssociateWebAcl() {
        return this.to('AssociateWebAcl');
    }
    /**
     * Grants permission to create an AWS App Runner automatic scaling configuration resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateAutoScalingConfiguration.html
     */
    toCreateAutoScalingConfiguration() {
        return this.to('CreateAutoScalingConfiguration');
    }
    /**
     * Grants permission to create an AWS App Runner connection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create an AWS App Runner observability configuration resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateObservabilityConfiguration.html
     */
    toCreateObservabilityConfiguration() {
        return this.to('CreateObservabilityConfiguration');
    }
    /**
     * Grants permission to create an AWS App Runner service resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifConnectionArn()
     * - .ifAutoScalingConfigurationArn()
     * - .ifObservabilityConfigurationArn()
     * - .ifVpcConnectorArn()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateService.html
     */
    toCreateService() {
        return this.to('CreateService');
    }
    /**
     * Grants permission to create an AWS App Runner VPC connector resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateVpcConnector.html
     */
    toCreateVpcConnector() {
        return this.to('CreateVpcConnector');
    }
    /**
     * Grants permission to create an AWS App Runner VpcIngressConnection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifServiceArn()
     * - .ifVpcId()
     * - .ifVpcEndpointId()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_CreateVpcIngressConnection.html
     */
    toCreateVpcIngressConnection() {
        return this.to('CreateVpcIngressConnection');
    }
    /**
     * Grants permission to delete an AWS App Runner automatic scaling configuration resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteAutoScalingConfiguration.html
     */
    toDeleteAutoScalingConfiguration() {
        return this.to('DeleteAutoScalingConfiguration');
    }
    /**
     * Grants permission to delete an AWS App Runner connection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete an AWS App Runner observability configuration resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteObservabilityConfiguration.html
     */
    toDeleteObservabilityConfiguration() {
        return this.to('DeleteObservabilityConfiguration');
    }
    /**
     * Grants permission to delete an AWS App Runner service resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteService.html
     */
    toDeleteService() {
        return this.to('DeleteService');
    }
    /**
     * Grants permission to delete an AWS App Runner VPC connector resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteVpcConnector.html
     */
    toDeleteVpcConnector() {
        return this.to('DeleteVpcConnector');
    }
    /**
     * Grants permission to delete an AWS App Runner VpcIngressConnection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DeleteVpcIngressConnection.html
     */
    toDeleteVpcIngressConnection() {
        return this.to('DeleteVpcIngressConnection');
    }
    /**
     * Grants permission to retrieve the description of an AWS App Runner automatic scaling configuration resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeAutoScalingConfiguration.html
     */
    toDescribeAutoScalingConfiguration() {
        return this.to('DescribeAutoScalingConfiguration');
    }
    /**
     * Grants permission to retrieve descriptions of custom domain names associated with an AWS App Runner service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeCustomDomains.html
     */
    toDescribeCustomDomains() {
        return this.to('DescribeCustomDomains');
    }
    /**
     * Grants permission to retrieve the description of an AWS App Runner observability configuration resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeObservabilityConfiguration.html
     */
    toDescribeObservabilityConfiguration() {
        return this.to('DescribeObservabilityConfiguration');
    }
    /**
     * Grants permission to retrieve the description of an operation that occurred on an AWS App Runner service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeOperation.html
     */
    toDescribeOperation() {
        return this.to('DescribeOperation');
    }
    /**
     * Grants permission to retrieve the description of an AWS App Runner service resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeService.html
     */
    toDescribeService() {
        return this.to('DescribeService');
    }
    /**
     * Grants permission to retrieve the description of an AWS App Runner VPC connector resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeVpcConnector.html
     */
    toDescribeVpcConnector() {
        return this.to('DescribeVpcConnector');
    }
    /**
     * Grants permission to retrieve the description of an AWS App Runner VpcIngressConnection resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DescribeVpcIngressConnection.html
     */
    toDescribeVpcIngressConnection() {
        return this.to('DescribeVpcIngressConnection');
    }
    /**
     * Grants permission to get the AWS WAF web ACL that is associated with an AWS App Runner service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/dg/waf-manage.html
     */
    toDescribeWebAclForService() {
        return this.to('DescribeWebAclForService');
    }
    /**
     * Grants permission to disassociate a custom domain name from an AWS App Runner service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_DisassociateCustomDomain.html
     */
    toDisassociateCustomDomain() {
        return this.to('DisassociateCustomDomain');
    }
    /**
     * Grants permission to disassociate the service with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/dg/waf-manage.html
     */
    toDisassociateWebAcl() {
        return this.to('DisassociateWebAcl');
    }
    /**
     * Grants permission to list the services that are associated with an AWS WAF web ACL
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/dg/waf-manage.html
     */
    toListAssociatedServicesForWebAcl() {
        return this.to('ListAssociatedServicesForWebAcl');
    }
    /**
     * Grants permission to retrieve a list of AWS App Runner automatic scaling configurations in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListAutoScalingConfigurations.html
     */
    toListAutoScalingConfigurations() {
        return this.to('ListAutoScalingConfigurations');
    }
    /**
     * Grants permission to retrieve a list of AWS App Runner connections in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListConnections.html
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to retrieve a list of AWS App Runner observability configurations in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListObservabilityConfigurations.html
     */
    toListObservabilityConfigurations() {
        return this.to('ListObservabilityConfigurations');
    }
    /**
     * Grants permission to retrieve a list of operations that occurred on an AWS App Runner service resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListOperations.html
     */
    toListOperations() {
        return this.to('ListOperations');
    }
    /**
     * Grants permission to retrieve a list of running AWS App Runner services in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListServices.html
     */
    toListServices() {
        return this.to('ListServices');
    }
    /**
     * Grants permission to retrieve a list of associated AppRunner services of an AWS App Runner automatic scaling configuration in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListServicesForAutoScalingConfiguration.html
     */
    toListServicesForAutoScalingConfiguration() {
        return this.to('ListServicesForAutoScalingConfiguration');
    }
    /**
     * Grants permission to list tags associated with an AWS App Runner resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of AWS App Runner VPC connectors in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListVpcConnectors.html
     */
    toListVpcConnectors() {
        return this.to('ListVpcConnectors');
    }
    /**
     * Grants permission to retrieve a list of AWS App Runner VpcIngressConnections in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ListVpcConnections.html
     */
    toListVpcIngressConnections() {
        return this.to('ListVpcIngressConnections');
    }
    /**
     * Grants permission to pause an active AWS App Runner service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_PauseService.html
     */
    toPauseService() {
        return this.to('PauseService');
    }
    /**
     * Grants permission to resume an active AWS App Runner service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_ResumeService.html
     */
    toResumeService() {
        return this.to('ResumeService');
    }
    /**
     * Grants permission to initiate a manual deployemnt to an AWS App Runner service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_StartDeployment.html
     */
    toStartDeployment() {
        return this.to('StartDeployment');
    }
    /**
     * Grants permission to add tags to, or update tag values of, an AWS App Runner resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an AWS App Runner resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an AWS App Runner automatic scaling configuration to be the default in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_UpdateDefaultAutoScalingConfiguration.html
     */
    toUpdateDefaultAutoScalingConfiguration() {
        return this.to('UpdateDefaultAutoScalingConfiguration');
    }
    /**
     * Grants permission to update an AWS App Runner service resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifConnectionArn()
     * - .ifAutoScalingConfigurationArn()
     * - .ifObservabilityConfigurationArn()
     * - .ifVpcConnectorArn()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_UpdateService.html
     */
    toUpdateService() {
        return this.to('UpdateService');
    }
    /**
     * Grants permission to update an AWS App Runner VpcIngressConnection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcId()
     * - .ifVpcEndpointId()
     *
     * https://docs.aws.amazon.com/apprunner/latest/api/API_UpdateVpcIngressConnection.html
     */
    toUpdateVpcIngressConnection() {
        return this.to('UpdateVpcIngressConnection');
    }
    /**
     * Adds a resource of type service to the statement
     *
     * @param serviceName - Identifier for the serviceName.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onService(serviceName, serviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceName}/${serviceId}`);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * @param connectionName - Identifier for the connectionName.
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnection(connectionName, connectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connection/${connectionName}/${connectionId}`);
    }
    /**
     * Adds a resource of type autoscalingconfiguration to the statement
     *
     * @param autoscalingConfigurationName - Identifier for the autoscalingConfigurationName.
     * @param autoscalingConfigurationVersion - Identifier for the autoscalingConfigurationVersion.
     * @param autoscalingConfigurationId - Identifier for the autoscalingConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAutoscalingconfiguration(autoscalingConfigurationName, autoscalingConfigurationVersion, autoscalingConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:autoscalingconfiguration/${autoscalingConfigurationName}/${autoscalingConfigurationVersion}/${autoscalingConfigurationId}`);
    }
    /**
     * Adds a resource of type observabilityconfiguration to the statement
     *
     * @param observabilityConfigurationName - Identifier for the observabilityConfigurationName.
     * @param observabilityConfigurationVersion - Identifier for the observabilityConfigurationVersion.
     * @param observabilityConfigurationId - Identifier for the observabilityConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onObservabilityconfiguration(observabilityConfigurationName, observabilityConfigurationVersion, observabilityConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:observabilityconfiguration/${observabilityConfigurationName}/${observabilityConfigurationVersion}/${observabilityConfigurationId}`);
    }
    /**
     * Adds a resource of type vpcconnector to the statement
     *
     * @param vpcConnectorName - Identifier for the vpcConnectorName.
     * @param vpcConnectorVersion - Identifier for the vpcConnectorVersion.
     * @param vpcConnectorId - Identifier for the vpcConnectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcconnector(vpcConnectorName, vpcConnectorVersion, vpcConnectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vpcconnector/${vpcConnectorName}/${vpcConnectorVersion}/${vpcConnectorId}`);
    }
    /**
     * Adds a resource of type vpcingressconnection to the statement
     *
     * @param vpcIngressConnectionName - Identifier for the vpcIngressConnectionName.
     * @param vpcIngressConnectionId - Identifier for the vpcIngressConnectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcingressconnection(vpcIngressConnectionName, vpcIngressConnectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vpcingressconnection/${vpcIngressConnectionName}/${vpcIngressConnectionId}`);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWebacl(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/webacl/${name}/${id}`);
    }
    /**
     * Filters access by the CreateService and UpdateService actions based on the ARN of an associated AutoScalingConfiguration resource
     *
     * Applies to actions:
     * - .toCreateService()
     * - .toUpdateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifAutoScalingConfigurationArn(value, operator) {
        return this.if(`AutoScalingConfigurationArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the CreateService and UpdateService actions based on the ARN of an associated Connection resource
     *
     * Applies to actions:
     * - .toCreateService()
     * - .toUpdateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifConnectionArn(value, operator) {
        return this.if(`ConnectionArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the CreateService and UpdateService actions based on the ARN of an associated ObservabilityConfiguration resource
     *
     * Applies to actions:
     * - .toCreateService()
     * - .toUpdateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifObservabilityConfigurationArn(value, operator) {
        return this.if(`ObservabilityConfigurationArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the CreateVpcIngressConnection action based on the ARN of an associated Service resource
     *
     * Applies to actions:
     * - .toCreateVpcIngressConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifServiceArn(value, operator) {
        return this.if(`ServiceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the CreateService and UpdateService actions based on the ARN of an associated VpcConnector resource
     *
     * Applies to actions:
     * - .toCreateService()
     * - .toUpdateService()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifVpcConnectorArn(value, operator) {
        return this.if(`VpcConnectorArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the CreateVpcIngressConnection and UpdateVpcIngressConnection actions based on the VPC Endpoint in the request
     *
     * Applies to actions:
     * - .toCreateVpcIngressConnection()
     * - .toUpdateVpcIngressConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcEndpointId(value, operator) {
        return this.if(`VpcEndpointId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the CreateVpcIngressConnection and UpdateVpcIngressConnection actions based on the VPC in the request
     *
     * Applies to actions:
     * - .toCreateVpcIngressConnection()
     * - .toUpdateVpcIngressConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcId(value, operator) {
        return this.if(`VpcId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * Applies to actions:
     * - .toCreateAutoScalingConfiguration()
     * - .toCreateConnection()
     * - .toCreateObservabilityConfiguration()
     * - .toCreateService()
     * - .toCreateVpcConnector()
     * - .toCreateVpcIngressConnection()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * Applies to resource types:
     * - service
     * - connection
     * - autoscalingconfiguration
     * - observabilityconfiguration
     * - vpcconnector
     * - vpcingressconnection
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * Applies to actions:
     * - .toCreateAutoScalingConfiguration()
     * - .toCreateConnection()
     * - .toCreateObservabilityConfiguration()
     * - .toCreateService()
     * - .toCreateVpcConnector()
     * - .toCreateVpcIngressConnection()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [apprunner](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsapprunner.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'apprunner';
        this.accessLevelList = {
            Write: [
                'AssociateCustomDomain',
                'AssociateWebAcl',
                'CreateAutoScalingConfiguration',
                'CreateConnection',
                'CreateObservabilityConfiguration',
                'CreateService',
                'CreateVpcConnector',
                'CreateVpcIngressConnection',
                'DeleteAutoScalingConfiguration',
                'DeleteConnection',
                'DeleteObservabilityConfiguration',
                'DeleteService',
                'DeleteVpcConnector',
                'DeleteVpcIngressConnection',
                'DisassociateCustomDomain',
                'DisassociateWebAcl',
                'PauseService',
                'ResumeService',
                'StartDeployment',
                'UpdateDefaultAutoScalingConfiguration',
                'UpdateService',
                'UpdateVpcIngressConnection'
            ],
            Read: [
                'DescribeAutoScalingConfiguration',
                'DescribeCustomDomains',
                'DescribeObservabilityConfiguration',
                'DescribeOperation',
                'DescribeService',
                'DescribeVpcConnector',
                'DescribeVpcIngressConnection',
                'DescribeWebAclForService',
                'ListTagsForResource'
            ],
            List: [
                'ListAssociatedServicesForWebAcl',
                'ListAutoScalingConfigurations',
                'ListConnections',
                'ListObservabilityConfigurations',
                'ListOperations',
                'ListServices',
                'ListServicesForAutoScalingConfiguration',
                'ListVpcConnectors',
                'ListVpcIngressConnections'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Apprunner = Apprunner;
//# sourceMappingURL=data:application/json;base64,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