"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BcmPricingCalculator = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [bcm-pricing-calculator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbillingandcostmanagementpricingcalculator.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class BcmPricingCalculator extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new bill estimate. Charge is incurred for successful bill estimates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_CreateBillEstimate.html
     */
    toCreateBillEstimate() {
        return this.to('CreateBillEstimate');
    }
    /**
     * Grants permission to create a new bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_CreateBillScenario.html
     */
    toCreateBillScenario() {
        return this.to('CreateBillScenario');
    }
    /**
     * Grants permission to create new commitments or remove existing commitment from a specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchCreateBillScenarioCommitmentModification.html
     */
    toCreateBillScenarioCommitmentModification() {
        return this.to('CreateBillScenarioCommitmentModification');
    }
    /**
     * Grants permission to create usage in the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchCreateBillScenarioUsageModification.html
     */
    toCreateBillScenarioUsageModification() {
        return this.to('CreateBillScenarioUsageModification');
    }
    /**
     * Grants permission to create a new Workload estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_CreateWorkloadEstimate.html
     */
    toCreateWorkloadEstimate() {
        return this.to('CreateWorkloadEstimate');
    }
    /**
     * Grants permission to create usage in the specified workload estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchCreateWorkloadEstimateUsage.html
     */
    toCreateWorkloadEstimateUsage() {
        return this.to('CreateWorkloadEstimateUsage');
    }
    /**
     * Grants permission to delete bill estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_DeleteBillEstimate.html
     */
    toDeleteBillEstimate() {
        return this.to('DeleteBillEstimate');
    }
    /**
     * Grants permission to delete a bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_DeleteBillScenario.html
     */
    toDeleteBillScenario() {
        return this.to('DeleteBillScenario');
    }
    /**
     * Grants permission to delete newly added commitments from the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchDeleteBillScenarioCommitmentModification.html
     */
    toDeleteBillScenarioCommitmentModification() {
        return this.to('DeleteBillScenarioCommitmentModification');
    }
    /**
     * Grants permission to delete newly added usage from the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchDeleteBillScenarioUsageModification.html
     */
    toDeleteBillScenarioUsageModification() {
        return this.to('DeleteBillScenarioUsageModification');
    }
    /**
     * Grants permission to delete the specified workload estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_DeleteWorkloadEstimate.html
     */
    toDeleteWorkloadEstimate() {
        return this.to('DeleteWorkloadEstimate');
    }
    /**
     * Grants permission to delete newly added usage from the specified workload estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchDeleteWorkloadEstimateUsage.html
     */
    toDeleteWorkloadEstimateUsage() {
        return this.to('DeleteWorkloadEstimateUsage');
    }
    /**
     * Grants permission to retrieve details of a bill estimate including estimated cost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_GetBillEstimate.html
     */
    toGetBillEstimate() {
        return this.to('GetBillEstimate');
    }
    /**
     * Grants permission to retrieve information associated with a bill scenario
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_GetBillScenario.html
     */
    toGetBillScenario() {
        return this.to('GetBillScenario');
    }
    /**
     * Grants permission to retrieve applicable rate type preferences for the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_GetPreferences.html
     */
    toGetPreferences() {
        return this.to('GetPreferences');
    }
    /**
     * Grants permission to retrieve information associated with a workload estimate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_GetWorkloadEstimate.html
     */
    toGetWorkloadEstimate() {
        return this.to('GetWorkloadEstimate');
    }
    /**
     * Grants permission to list commitments associated with the specified bill estimate
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillEstimateCommitments.html
     */
    toListBillEstimateCommitments() {
        return this.to('ListBillEstimateCommitments');
    }
    /**
     * Grants permission to list added or removed commitments for a specified bill estimate
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillEstimateInputCommitmentModifications.html
     */
    toListBillEstimateInputCommitmentModifications() {
        return this.to('ListBillEstimateInputCommitmentModifications');
    }
    /**
     * Grants permission to list added or modified usage for a specified bill estimate
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillEstimateInputUsageModifications.html
     */
    toListBillEstimateInputUsageModifications() {
        return this.to('ListBillEstimateInputUsageModifications');
    }
    /**
     * Grants permission to list result line items for a specified bill estimate
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillEstimateLineItems.html
     */
    toListBillEstimateLineItems() {
        return this.to('ListBillEstimateLineItems');
    }
    /**
     * Grants permission to list bill estimates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillEstimates.html
     */
    toListBillEstimates() {
        return this.to('ListBillEstimates');
    }
    /**
     * Grants permission to list commitments included in a bill scenario
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillScenarioCommitmentModifications.html
     */
    toListBillScenarioCommitmentModifications() {
        return this.to('ListBillScenarioCommitmentModifications');
    }
    /**
     * Grants permission to list usage lines of a specified bill scenario
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillScenarioUsageModifications.html
     */
    toListBillScenarioUsageModifications() {
        return this.to('ListBillScenarioUsageModifications');
    }
    /**
     * Grants permission to list bill scenarios
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListBillScenarios.html
     */
    toListBillScenarios() {
        return this.to('ListBillScenarios');
    }
    /**
     * Grants permission to return a list of tags for a resource
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list usage lines for the specified workload estimate
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListWorkloadEstimateUsage.html
     */
    toListWorkloadEstimateUsage() {
        return this.to('ListWorkloadEstimateUsage');
    }
    /**
     * Grants permission to list workload estimates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_ListWorkloadEstimates.html
     */
    toListWorkloadEstimates() {
        return this.to('ListWorkloadEstimates');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update bill estimate name and expiration date time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_UpdateBillEstimate.html
     */
    toUpdateBillEstimate() {
        return this.to('UpdateBillEstimate');
    }
    /**
     * Grants permission to update name and expiration date time of the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_UpdateBillScenario.html
     */
    toUpdateBillScenario() {
        return this.to('UpdateBillScenario');
    }
    /**
     * Grants permission to update commitment group of commitments in the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchUpdateBillScenarioCommitmentModification.html
     */
    toUpdateBillScenarioCommitmentModification() {
        return this.to('UpdateBillScenarioCommitmentModification');
    }
    /**
     * Grants permission to update usage amount, usage hour, and usage group in the specified bill scenario
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchUpdateBillScenarioUsageModification.html
     */
    toUpdateBillScenarioUsageModification() {
        return this.to('UpdateBillScenarioUsageModification');
    }
    /**
     * Grants permission to update rate type preferences for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_UpdatePreferences.html
     */
    toUpdatePreferences() {
        return this.to('UpdatePreferences');
    }
    /**
     * Grants permission to update name and expiration date time of the specified workload estimate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_UpdateWorkloadEstimate.html
     */
    toUpdateWorkloadEstimate() {
        return this.to('UpdateWorkloadEstimate');
    }
    /**
     * Grants permission to update usage amount and usage group in the specified workload estimate based on the usage id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AWSBCMPricingCalculator_BatchUpdateWorkloadEstimateUsage.html
     */
    toUpdateWorkloadEstimateUsage() {
        return this.to('UpdateWorkloadEstimateUsage');
    }
    /**
     * Adds a resource of type bill-estimate to the statement
     *
     * https://docs.aws.amazon.com/cost-management/latest/userguide/pc-bill-estimate.html
     *
     * @param billEstimateId - Identifier for the billEstimateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBillEstimate(billEstimateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bcm-pricing-calculator:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bill-estimate/${billEstimateId}`);
    }
    /**
     * Adds a resource of type bill-scenario to the statement
     *
     * https://docs.aws.amazon.com/cost-management/latest/userguide/pc-bill-scenario.html
     *
     * @param billScenarioId - Identifier for the billScenarioId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBillScenario(billScenarioId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bcm-pricing-calculator:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bill-scenario/${billScenarioId}`);
    }
    /**
     * Adds a resource of type workload-estimate to the statement
     *
     * https://docs.aws.amazon.com/cost-management/latest/userguide/pc-workload-estimate.html
     *
     * @param workloadEstimateId - Identifier for the workloadEstimateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWorkloadEstimate(workloadEstimateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:bcm-pricing-calculator:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workload-estimate/${workloadEstimateId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [bcm-pricing-calculator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsbillingandcostmanagementpricingcalculator.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'bcm-pricing-calculator';
        this.accessLevelList = {
            Write: [
                'CreateBillEstimate',
                'CreateBillScenario',
                'CreateBillScenarioCommitmentModification',
                'CreateBillScenarioUsageModification',
                'CreateWorkloadEstimate',
                'CreateWorkloadEstimateUsage',
                'DeleteBillEstimate',
                'DeleteBillScenario',
                'DeleteBillScenarioCommitmentModification',
                'DeleteBillScenarioUsageModification',
                'DeleteWorkloadEstimate',
                'DeleteWorkloadEstimateUsage',
                'UpdateBillEstimate',
                'UpdateBillScenario',
                'UpdateBillScenarioCommitmentModification',
                'UpdateBillScenarioUsageModification',
                'UpdatePreferences',
                'UpdateWorkloadEstimate',
                'UpdateWorkloadEstimateUsage'
            ],
            Read: [
                'GetBillEstimate',
                'GetBillScenario',
                'GetPreferences',
                'GetWorkloadEstimate'
            ],
            List: [
                'ListBillEstimateCommitments',
                'ListBillEstimateInputCommitmentModifications',
                'ListBillEstimateInputUsageModifications',
                'ListBillEstimateLineItems',
                'ListBillEstimates',
                'ListBillScenarioCommitmentModifications',
                'ListBillScenarioUsageModifications',
                'ListBillScenarios',
                'ListWorkloadEstimateUsage',
                'ListWorkloadEstimates'
            ],
            Tagging: [
                'ListTagsForResource',
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.BcmPricingCalculator = BcmPricingCalculator;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmlsbGluZ2FuZGNvc3RtYW5hZ2VtZW50cHJpY2luZ2NhbGN1bGF0b3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJiaWxsaW5nYW5kY29zdG1hbmFnZW1lbnRwcmljaW5nY2FsY3VsYXRvci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsb0JBQXFCLFNBQVEsd0JBQWU7SUFHdkQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBDQUEwQztRQUMvQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBDQUEwQztRQUMvQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhDQUE4QztRQUNuRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOENBQThDLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBDQUEwQztRQUMvQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUNBQXFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQ0FBcUMsQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFpREQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYyxDQUFDLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsMkJBQTRCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQkFBbUIsY0FBZSxFQUFFLENBQUMsQ0FBQztJQUMvTCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYyxDQUFDLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsMkJBQTRCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxrQkFBbUIsY0FBZSxFQUFFLENBQUMsQ0FBQztJQUMvTCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksa0JBQWtCLENBQUMsa0JBQTBCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDekcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsMkJBQTRCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxzQkFBdUIsa0JBQW1CLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZNLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFoaUJSLGtCQUFhLEdBQUcsd0JBQXdCLENBQUM7UUFxWnRDLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxvQkFBb0I7Z0JBQ3BCLG9CQUFvQjtnQkFDcEIsMENBQTBDO2dCQUMxQyxxQ0FBcUM7Z0JBQ3JDLHdCQUF3QjtnQkFDeEIsNkJBQTZCO2dCQUM3QixvQkFBb0I7Z0JBQ3BCLG9CQUFvQjtnQkFDcEIsMENBQTBDO2dCQUMxQyxxQ0FBcUM7Z0JBQ3JDLHdCQUF3QjtnQkFDeEIsNkJBQTZCO2dCQUM3QixvQkFBb0I7Z0JBQ3BCLG9CQUFvQjtnQkFDcEIsMENBQTBDO2dCQUMxQyxxQ0FBcUM7Z0JBQ3JDLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2dCQUN4Qiw2QkFBNkI7YUFDOUI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osaUJBQWlCO2dCQUNqQixpQkFBaUI7Z0JBQ2pCLGdCQUFnQjtnQkFDaEIscUJBQXFCO2FBQ3RCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDZCQUE2QjtnQkFDN0IsOENBQThDO2dCQUM5Qyx5Q0FBeUM7Z0JBQ3pDLDJCQUEyQjtnQkFDM0IsbUJBQW1CO2dCQUNuQix5Q0FBeUM7Z0JBQ3pDLG9DQUFvQztnQkFDcEMsbUJBQW1CO2dCQUNuQiwyQkFBMkI7Z0JBQzNCLHVCQUF1QjthQUN4QjtZQUNELE9BQU8sRUFBRTtnQkFDUCxxQkFBcUI7Z0JBQ3JCLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUErRkYsQ0FBQztDQUNGO0FBbmlCRCxvREFtaUJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtiY20tcHJpY2luZy1jYWxjdWxhdG9yXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiaWxsaW5nYW5kY29zdG1hbmFnZW1lbnRwcmljaW5nY2FsY3VsYXRvci5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEJjbVByaWNpbmdDYWxjdWxhdG9yIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnYmNtLXByaWNpbmctY2FsY3VsYXRvcic7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5ldyBiaWxsIGVzdGltYXRlLiBDaGFyZ2UgaXMgaW5jdXJyZWQgZm9yIHN1Y2Nlc3NmdWwgYmlsbCBlc3RpbWF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9DcmVhdGVCaWxsRXN0aW1hdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQmlsbEVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVCaWxsRXN0aW1hdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgYmlsbCBzY2VuYXJpb1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0NyZWF0ZUJpbGxTY2VuYXJpby5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCaWxsU2NlbmFyaW8oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUJpbGxTY2VuYXJpbycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBuZXcgY29tbWl0bWVudHMgb3IgcmVtb3ZlIGV4aXN0aW5nIGNvbW1pdG1lbnQgZnJvbSBhIHNwZWNpZmllZCBiaWxsIHNjZW5hcmlvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfQmF0Y2hDcmVhdGVCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUJpbGxTY2VuYXJpb0NvbW1pdG1lbnRNb2RpZmljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUJpbGxTY2VuYXJpb0NvbW1pdG1lbnRNb2RpZmljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgdXNhZ2UgaW4gdGhlIHNwZWNpZmllZCBiaWxsIHNjZW5hcmlvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfQmF0Y2hDcmVhdGVCaWxsU2NlbmFyaW9Vc2FnZU1vZGlmaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVCaWxsU2NlbmFyaW9Vc2FnZU1vZGlmaWNhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgV29ya2xvYWQgZXN0aW1hdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9DcmVhdGVXb3JrbG9hZEVzdGltYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVdvcmtsb2FkRXN0aW1hdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVdvcmtsb2FkRXN0aW1hdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgdXNhZ2UgaW4gdGhlIHNwZWNpZmllZCB3b3JrbG9hZCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0JhdGNoQ3JlYXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVdvcmtsb2FkRXN0aW1hdGVVc2FnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGJpbGwgZXN0aW1hdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9EZWxldGVCaWxsRXN0aW1hdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQmlsbEVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVCaWxsRXN0aW1hdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBiaWxsIHNjZW5hcmlvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfRGVsZXRlQmlsbFNjZW5hcmlvLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUJpbGxTY2VuYXJpbygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQmlsbFNjZW5hcmlvJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIG5ld2x5IGFkZGVkIGNvbW1pdG1lbnRzIGZyb20gdGhlIHNwZWNpZmllZCBiaWxsIHNjZW5hcmlvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfQmF0Y2hEZWxldGVCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUJpbGxTY2VuYXJpb0NvbW1pdG1lbnRNb2RpZmljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUJpbGxTY2VuYXJpb0NvbW1pdG1lbnRNb2RpZmljYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgbmV3bHkgYWRkZWQgdXNhZ2UgZnJvbSB0aGUgc3BlY2lmaWVkIGJpbGwgc2NlbmFyaW9cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9CYXRjaERlbGV0ZUJpbGxTY2VuYXJpb1VzYWdlTW9kaWZpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUJpbGxTY2VuYXJpb1VzYWdlTW9kaWZpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVCaWxsU2NlbmFyaW9Vc2FnZU1vZGlmaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgc3BlY2lmaWVkIHdvcmtsb2FkIGVzdGltYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfRGVsZXRlV29ya2xvYWRFc3RpbWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVXb3JrbG9hZEVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXb3JrbG9hZEVzdGltYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIG5ld2x5IGFkZGVkIHVzYWdlIGZyb20gdGhlIHNwZWNpZmllZCB3b3JrbG9hZCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0JhdGNoRGVsZXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVdvcmtsb2FkRXN0aW1hdGVVc2FnZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBvZiBhIGJpbGwgZXN0aW1hdGUgaW5jbHVkaW5nIGVzdGltYXRlZCBjb3N0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9HZXRCaWxsRXN0aW1hdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QmlsbEVzdGltYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCaWxsRXN0aW1hdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhc3NvY2lhdGVkIHdpdGggYSBiaWxsIHNjZW5hcmlvXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9HZXRCaWxsU2NlbmFyaW8uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0QmlsbFNjZW5hcmlvKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRCaWxsU2NlbmFyaW8nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhcHBsaWNhYmxlIHJhdGUgdHlwZSBwcmVmZXJlbmNlcyBmb3IgdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0dldFByZWZlcmVuY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFByZWZlcmVuY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQcmVmZXJlbmNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFzc29jaWF0ZWQgd2l0aCBhIHdvcmtsb2FkIGVzdGltYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9HZXRXb3JrbG9hZEVzdGltYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdvcmtsb2FkRXN0aW1hdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFdvcmtsb2FkRXN0aW1hdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGNvbW1pdG1lbnRzIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3BlY2lmaWVkIGJpbGwgZXN0aW1hdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0xpc3RCaWxsRXN0aW1hdGVDb21taXRtZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QmlsbEVzdGltYXRlQ29tbWl0bWVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCaWxsRXN0aW1hdGVDb21taXRtZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWRkZWQgb3IgcmVtb3ZlZCBjb21taXRtZW50cyBmb3IgYSBzcGVjaWZpZWQgYmlsbCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxFc3RpbWF0ZUlucHV0Q29tbWl0bWVudE1vZGlmaWNhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJpbGxFc3RpbWF0ZUlucHV0Q29tbWl0bWVudE1vZGlmaWNhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCaWxsRXN0aW1hdGVJbnB1dENvbW1pdG1lbnRNb2RpZmljYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhZGRlZCBvciBtb2RpZmllZCB1c2FnZSBmb3IgYSBzcGVjaWZpZWQgYmlsbCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxFc3RpbWF0ZUlucHV0VXNhZ2VNb2RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RCaWxsRXN0aW1hdGVJbnB1dFVzYWdlTW9kaWZpY2F0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEJpbGxFc3RpbWF0ZUlucHV0VXNhZ2VNb2RpZmljYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCByZXN1bHQgbGluZSBpdGVtcyBmb3IgYSBzcGVjaWZpZWQgYmlsbCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxFc3RpbWF0ZUxpbmVJdGVtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QmlsbEVzdGltYXRlTGluZUl0ZW1zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0QmlsbEVzdGltYXRlTGluZUl0ZW1zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBiaWxsIGVzdGltYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxFc3RpbWF0ZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJpbGxFc3RpbWF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCaWxsRXN0aW1hdGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBjb21taXRtZW50cyBpbmNsdWRlZCBpbiBhIGJpbGwgc2NlbmFyaW9cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0xpc3RCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QmlsbFNjZW5hcmlvQ29tbWl0bWVudE1vZGlmaWNhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdXNhZ2UgbGluZXMgb2YgYSBzcGVjaWZpZWQgYmlsbCBzY2VuYXJpb1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxTY2VuYXJpb1VzYWdlTW9kaWZpY2F0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0QmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBiaWxsIHNjZW5hcmlvc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdEJpbGxTY2VuYXJpb3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEJpbGxTY2VuYXJpb3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RCaWxsU2NlbmFyaW9zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0dXJuIGEgbGlzdCBvZiB0YWdzIGZvciBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHVzYWdlIGxpbmVzIGZvciB0aGUgc3BlY2lmaWVkIHdvcmtsb2FkIGVzdGltYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9MaXN0V29ya2xvYWRFc3RpbWF0ZVVzYWdlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JrbG9hZEVzdGltYXRlVXNhZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RXb3JrbG9hZEVzdGltYXRlVXNhZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHdvcmtsb2FkIGVzdGltYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfTGlzdFdvcmtsb2FkRXN0aW1hdGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RXb3JrbG9hZEVzdGltYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFdvcmtsb2FkRXN0aW1hdGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGEgdGFnIHRvIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSBhIHRhZyBmcm9tIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBiaWxsIGVzdGltYXRlIG5hbWUgYW5kIGV4cGlyYXRpb24gZGF0ZSB0aW1lXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfVXBkYXRlQmlsbEVzdGltYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUJpbGxFc3RpbWF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQmlsbEVzdGltYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIG5hbWUgYW5kIGV4cGlyYXRpb24gZGF0ZSB0aW1lIG9mIHRoZSBzcGVjaWZpZWQgYmlsbCBzY2VuYXJpb1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX1VwZGF0ZUJpbGxTY2VuYXJpby5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVCaWxsU2NlbmFyaW8oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUJpbGxTY2VuYXJpbycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBjb21taXRtZW50IGdyb3VwIG9mIGNvbW1pdG1lbnRzIGluIHRoZSBzcGVjaWZpZWQgYmlsbCBzY2VuYXJpb1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0JhdGNoVXBkYXRlQmlsbFNjZW5hcmlvQ29tbWl0bWVudE1vZGlmaWNhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHVzYWdlIGFtb3VudCwgdXNhZ2UgaG91ciwgYW5kIHVzYWdlIGdyb3VwIGluIHRoZSBzcGVjaWZpZWQgYmlsbCBzY2VuYXJpb1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX0JhdGNoVXBkYXRlQmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUJpbGxTY2VuYXJpb1VzYWdlTW9kaWZpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHJhdGUgdHlwZSBwcmVmZXJlbmNlcyBmb3IgdGhlIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXdzLWNvc3QtbWFuYWdlbWVudC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9BV1NCQ01QcmljaW5nQ2FsY3VsYXRvcl9VcGRhdGVQcmVmZXJlbmNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVQcmVmZXJlbmNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUHJlZmVyZW5jZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgbmFtZSBhbmQgZXhwaXJhdGlvbiBkYXRlIHRpbWUgb2YgdGhlIHNwZWNpZmllZCB3b3JrbG9hZCBlc3RpbWF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hd3MtY29zdC1tYW5hZ2VtZW50L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0FXU0JDTVByaWNpbmdDYWxjdWxhdG9yX1VwZGF0ZVdvcmtsb2FkRXN0aW1hdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlV29ya2xvYWRFc3RpbWF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlV29ya2xvYWRFc3RpbWF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB1c2FnZSBhbW91bnQgYW5kIHVzYWdlIGdyb3VwIGluIHRoZSBzcGVjaWZpZWQgd29ya2xvYWQgZXN0aW1hdGUgYmFzZWQgb24gdGhlIHVzYWdlIGlkXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F3cy1jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQVdTQkNNUHJpY2luZ0NhbGN1bGF0b3JfQmF0Y2hVcGRhdGVXb3JrbG9hZEVzdGltYXRlVXNhZ2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVXb3JrbG9hZEVzdGltYXRlVXNhZ2UnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0NyZWF0ZUJpbGxFc3RpbWF0ZScsXG4gICAgICAnQ3JlYXRlQmlsbFNjZW5hcmlvJyxcbiAgICAgICdDcmVhdGVCaWxsU2NlbmFyaW9Db21taXRtZW50TW9kaWZpY2F0aW9uJyxcbiAgICAgICdDcmVhdGVCaWxsU2NlbmFyaW9Vc2FnZU1vZGlmaWNhdGlvbicsXG4gICAgICAnQ3JlYXRlV29ya2xvYWRFc3RpbWF0ZScsXG4gICAgICAnQ3JlYXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlJyxcbiAgICAgICdEZWxldGVCaWxsRXN0aW1hdGUnLFxuICAgICAgJ0RlbGV0ZUJpbGxTY2VuYXJpbycsXG4gICAgICAnRGVsZXRlQmlsbFNjZW5hcmlvQ29tbWl0bWVudE1vZGlmaWNhdGlvbicsXG4gICAgICAnRGVsZXRlQmlsbFNjZW5hcmlvVXNhZ2VNb2RpZmljYXRpb24nLFxuICAgICAgJ0RlbGV0ZVdvcmtsb2FkRXN0aW1hdGUnLFxuICAgICAgJ0RlbGV0ZVdvcmtsb2FkRXN0aW1hdGVVc2FnZScsXG4gICAgICAnVXBkYXRlQmlsbEVzdGltYXRlJyxcbiAgICAgICdVcGRhdGVCaWxsU2NlbmFyaW8nLFxuICAgICAgJ1VwZGF0ZUJpbGxTY2VuYXJpb0NvbW1pdG1lbnRNb2RpZmljYXRpb24nLFxuICAgICAgJ1VwZGF0ZUJpbGxTY2VuYXJpb1VzYWdlTW9kaWZpY2F0aW9uJyxcbiAgICAgICdVcGRhdGVQcmVmZXJlbmNlcycsXG4gICAgICAnVXBkYXRlV29ya2xvYWRFc3RpbWF0ZScsXG4gICAgICAnVXBkYXRlV29ya2xvYWRFc3RpbWF0ZVVzYWdlJ1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0dldEJpbGxFc3RpbWF0ZScsXG4gICAgICAnR2V0QmlsbFNjZW5hcmlvJyxcbiAgICAgICdHZXRQcmVmZXJlbmNlcycsXG4gICAgICAnR2V0V29ya2xvYWRFc3RpbWF0ZSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0QmlsbEVzdGltYXRlQ29tbWl0bWVudHMnLFxuICAgICAgJ0xpc3RCaWxsRXN0aW1hdGVJbnB1dENvbW1pdG1lbnRNb2RpZmljYXRpb25zJyxcbiAgICAgICdMaXN0QmlsbEVzdGltYXRlSW5wdXRVc2FnZU1vZGlmaWNhdGlvbnMnLFxuICAgICAgJ0xpc3RCaWxsRXN0aW1hdGVMaW5lSXRlbXMnLFxuICAgICAgJ0xpc3RCaWxsRXN0aW1hdGVzJyxcbiAgICAgICdMaXN0QmlsbFNjZW5hcmlvQ29tbWl0bWVudE1vZGlmaWNhdGlvbnMnLFxuICAgICAgJ0xpc3RCaWxsU2NlbmFyaW9Vc2FnZU1vZGlmaWNhdGlvbnMnLFxuICAgICAgJ0xpc3RCaWxsU2NlbmFyaW9zJyxcbiAgICAgICdMaXN0V29ya2xvYWRFc3RpbWF0ZVVzYWdlJyxcbiAgICAgICdMaXN0V29ya2xvYWRFc3RpbWF0ZXMnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZScsXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBiaWxsLWVzdGltYXRlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvc3QtbWFuYWdlbWVudC9sYXRlc3QvdXNlcmd1aWRlL3BjLWJpbGwtZXN0aW1hdGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYmlsbEVzdGltYXRlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYmlsbEVzdGltYXRlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQmlsbEVzdGltYXRlKGJpbGxFc3RpbWF0ZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YmNtLXByaWNpbmctY2FsY3VsYXRvcjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06YmlsbC1lc3RpbWF0ZS8keyBiaWxsRXN0aW1hdGVJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYmlsbC1zY2VuYXJpbyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb3N0LW1hbmFnZW1lbnQvbGF0ZXN0L3VzZXJndWlkZS9wYy1iaWxsLXNjZW5hcmlvLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGJpbGxTY2VuYXJpb0lkIC0gSWRlbnRpZmllciBmb3IgdGhlIGJpbGxTY2VuYXJpb0lkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkJpbGxTY2VuYXJpbyhiaWxsU2NlbmFyaW9JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmJjbS1wcmljaW5nLWNhbGN1bGF0b3I6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmJpbGwtc2NlbmFyaW8vJHsgYmlsbFNjZW5hcmlvSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHdvcmtsb2FkLWVzdGltYXRlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvc3QtbWFuYWdlbWVudC9sYXRlc3QvdXNlcmd1aWRlL3BjLXdvcmtsb2FkLWVzdGltYXRlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHdvcmtsb2FkRXN0aW1hdGVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JrbG9hZEVzdGltYXRlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uV29ya2xvYWRFc3RpbWF0ZSh3b3JrbG9hZEVzdGltYXRlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpiY20tcHJpY2luZy1jYWxjdWxhdG9yOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp3b3JrbG9hZC1lc3RpbWF0ZS8keyB3b3JrbG9hZEVzdGltYXRlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVxdWVzdFRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXF1ZXN0VGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZ3MgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtdGFna2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtiY20tcHJpY2luZy1jYWxjdWxhdG9yXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NiaWxsaW5nYW5kY29zdG1hbmFnZW1lbnRwcmljaW5nY2FsY3VsYXRvci5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=