"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Chatbot = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Chatbot extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a resource with a configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_AssociateToConfiguration.html
     */
    toAssociateToConfiguration() {
        return this.to('AssociateToConfiguration');
    }
    /**
     * Grants permission to create an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateChimeWebhookConfiguration.html
     */
    toCreateChimeWebhookConfiguration() {
        return this.to('CreateChimeWebhookConfiguration');
    }
    /**
     * Grants permission to create a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateCustomAction.html
     */
    toCreateCustomAction() {
        return this.to('CreateCustomAction');
    }
    /**
     * Grants permission to create an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateMicrosoftTeamsChannelConfiguration.html
     */
    toCreateMicrosoftTeamsChannelConfiguration() {
        return this.to('CreateMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to create an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_CreateSlackChannelConfiguration.html
     */
    toCreateSlackChannelConfiguration() {
        return this.to('CreateSlackChannelConfiguration');
    }
    /**
     * Grants permission to delete an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteChimeWebhookConfiguration.html
     */
    toDeleteChimeWebhookConfiguration() {
        return this.to('DeleteChimeWebhookConfiguration');
    }
    /**
     * Grants permission to delete a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteCustomAction.html
     */
    toDeleteCustomAction() {
        return this.to('DeleteCustomAction');
    }
    /**
     * Grants permission to delete an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsChannelConfiguration.html
     */
    toDeleteMicrosoftTeamsChannelConfiguration() {
        return this.to('DeleteMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to delete the Microsoft Teams configured with AWS Chatbot in an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsConfiguredTeam.html
     */
    toDeleteMicrosoftTeamsConfiguredTeam() {
        return this.to('DeleteMicrosoftTeamsConfiguredTeam');
    }
    /**
     * Grants permission to delete an AWS Chatbot Microsoft Teams User Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteMicrosoftTeamsUserIdentity.html
     */
    toDeleteMicrosoftTeamsUserIdentity() {
        return this.to('DeleteMicrosoftTeamsUserIdentity');
    }
    /**
     * Grants permission to delete an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackChannelConfiguration.html
     */
    toDeleteSlackChannelConfiguration() {
        return this.to('DeleteSlackChannelConfiguration');
    }
    /**
     * Grants permission to delete an AWS Chatbot Slack User Identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackUserIdentity.html
     */
    toDeleteSlackUserIdentity() {
        return this.to('DeleteSlackUserIdentity');
    }
    /**
     * Grants permission to delete the Slack workspace authorization with AWS Chatbot, associated with an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DeleteSlackWorkspaceAuthorization.html
     */
    toDeleteSlackWorkspaceAuthorization() {
        return this.to('DeleteSlackWorkspaceAuthorization');
    }
    /**
     * Grants permission to list all AWS Chatbot Chime Webhook Configurations in an AWS Account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeChimeWebhookConfigurations.html
     */
    toDescribeChimeWebhookConfigurations() {
        return this.to('DescribeChimeWebhookConfigurations');
    }
    /**
     * Grants permission to list all AWS Chatbot Slack Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackChannelConfigurations.html
     */
    toDescribeSlackChannelConfigurations() {
        return this.to('DescribeSlackChannelConfigurations');
    }
    /**
     * Grants permission to list all public Slack channels in the Slack workspace connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toDescribeSlackChannels() {
        return this.to('DescribeSlackChannels');
    }
    /**
     * Grants permission to describe AWS Chatbot Slack User Identities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackUserIdentities.html
     */
    toDescribeSlackUserIdentities() {
        return this.to('DescribeSlackUserIdentities');
    }
    /**
     * Grants permission to list all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DescribeSlackWorkspaces.html
     */
    toDescribeSlackWorkspaces() {
        return this.to('DescribeSlackWorkspaces');
    }
    /**
     * Grants permission to disassociate a resource from a configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_DisassociateFromConfiguration.html
     */
    toDisassociateFromConfiguration() {
        return this.to('DisassociateFromConfiguration');
    }
    /**
     * Grants permission to retrieve AWS Chatbot account preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetAccountPreferences.html
     */
    toGetAccountPreferences() {
        return this.to('GetAccountPreferences');
    }
    /**
     * Grants permission to get a custom action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetCustomAction.html
     */
    toGetCustomAction() {
        return this.to('GetCustomAction');
    }
    /**
     * Grants permission to get a single AWS Chatbot Microsoft Teams Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_GetMicrosoftTeamsChannelConfiguration.html
     */
    toGetMicrosoftTeamsChannelConfiguration() {
        return this.to('GetMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to generate OAuth parameters to request Microsoft Teams OAuth code to be used by the AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toGetMicrosoftTeamsOauthParameters() {
        return this.to('GetMicrosoftTeamsOauthParameters');
    }
    /**
     * Grants permission to generate OAuth parameters to request Slack OAuth code to be used by the AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toGetSlackOauthParameters() {
        return this.to('GetSlackOauthParameters');
    }
    /**
     * Grants permission to list resources associated with a configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListAssociations.html
     */
    toListAssociations() {
        return this.to('ListAssociations');
    }
    /**
     * Grants permission to list custom actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListCustomActions.html
     */
    toListCustomActions() {
        return this.to('ListCustomActions');
    }
    /**
     * Grants permission to list all AWS Chatbot Microsoft Teams Channel Configurations in an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsChannelConfigurations.html
     */
    toListMicrosoftTeamsChannelConfigurations() {
        return this.to('ListMicrosoftTeamsChannelConfigurations');
    }
    /**
     * Grants permission to list all Microsoft Teams connected to the AWS Account onboarded with AWS Chatbot service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsConfiguredTeams.html
     */
    toListMicrosoftTeamsConfiguredTeams() {
        return this.to('ListMicrosoftTeamsConfiguredTeams');
    }
    /**
     * Grants permission to describe AWS Chatbot Microsoft Teams User Identities
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListMicrosoftTeamsUserIdentities.html
     */
    toListMicrosoftTeamsUserIdentities() {
        return this.to('ListMicrosoftTeamsUserIdentities');
    }
    /**
     * Grants permission to List all tags associated with the AWS Chatbot Channel Configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to redeem previously generated parameters with Microsoft APIs, to acquire OAuth tokens to be used by the AWS Chatbot service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toRedeemMicrosoftTeamsOauthCode() {
        return this.to('RedeemMicrosoftTeamsOauthCode');
    }
    /**
     * Grants permission to redeem previously generated parameters with Slack API, to acquire OAuth tokens to be used by the AWS Chatbot service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_Operations.html
     */
    toRedeemSlackOauthCode() {
        return this.to('RedeemSlackOauthCode');
    }
    /**
     * Grants permission to create tags on AWS Chatbot Channel Configuration
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags on AWS Chatbot Channel Configuration
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update AWS Chatbot account preferences
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateAccountPreferences.html
     */
    toUpdateAccountPreferences() {
        return this.to('UpdateAccountPreferences');
    }
    /**
     * Grants permission to update an AWS Chatbot Chime Webhook Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateChimeWebhookConfiguration.html
     */
    toUpdateChimeWebhookConfiguration() {
        return this.to('UpdateChimeWebhookConfiguration');
    }
    /**
     * Grants permission to update a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateCustomAction.html
     */
    toUpdateCustomAction() {
        return this.to('UpdateCustomAction');
    }
    /**
     * Grants permission to update an AWS Chatbot Microsoft Teams Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateMicrosoftTeamsChannelConfiguration.html
     */
    toUpdateMicrosoftTeamsChannelConfiguration() {
        return this.to('UpdateMicrosoftTeamsChannelConfiguration');
    }
    /**
     * Grants permission to update an AWS Chatbot Slack Channel Configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/chatbot/latest/APIReference/API_UpdateSlackChannelConfiguration.html
     */
    toUpdateSlackChannelConfiguration() {
        return this.to('UpdateSlackChannelConfiguration');
    }
    /**
     * Adds a resource of type ChatbotConfiguration to the statement
     *
     * https://docs.aws.amazon.com/chatbot/latest/adminguide/what-is.html
     *
     * @param configurationType - Identifier for the configurationType.
     * @param chatbotConfigurationName - Identifier for the chatbotConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onChatbotConfiguration(configurationType, chatbotConfigurationName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:chatbot::${account ?? this.defaultAccount}:chat-configuration/${configurationType}/${chatbotConfigurationName}`);
    }
    /**
     * Adds a resource of type custom-action to the statement
     *
     * https://docs.aws.amazon.com/chatbot/latest/adminguide/what-is.html
     *
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCustomAction(actionName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:chatbot::${account ?? this.defaultAccount}:custom-action/${actionName}`);
    }
    /**
     * Statement provider for service [chatbot](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awschatbot.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'chatbot';
        this.accessLevelList = {
            Write: [
                'AssociateToConfiguration',
                'CreateChimeWebhookConfiguration',
                'CreateCustomAction',
                'CreateMicrosoftTeamsChannelConfiguration',
                'CreateSlackChannelConfiguration',
                'DeleteChimeWebhookConfiguration',
                'DeleteCustomAction',
                'DeleteMicrosoftTeamsChannelConfiguration',
                'DeleteMicrosoftTeamsConfiguredTeam',
                'DeleteMicrosoftTeamsUserIdentity',
                'DeleteSlackChannelConfiguration',
                'DeleteSlackUserIdentity',
                'DeleteSlackWorkspaceAuthorization',
                'DisassociateFromConfiguration',
                'RedeemMicrosoftTeamsOauthCode',
                'RedeemSlackOauthCode',
                'UpdateAccountPreferences',
                'UpdateChimeWebhookConfiguration',
                'UpdateCustomAction',
                'UpdateMicrosoftTeamsChannelConfiguration',
                'UpdateSlackChannelConfiguration'
            ],
            Read: [
                'DescribeChimeWebhookConfigurations',
                'DescribeSlackChannelConfigurations',
                'DescribeSlackChannels',
                'DescribeSlackUserIdentities',
                'DescribeSlackWorkspaces',
                'GetAccountPreferences',
                'GetCustomAction',
                'GetMicrosoftTeamsChannelConfiguration',
                'GetMicrosoftTeamsOauthParameters',
                'GetSlackOauthParameters',
                'ListAssociations',
                'ListMicrosoftTeamsChannelConfigurations',
                'ListMicrosoftTeamsConfiguredTeams',
                'ListMicrosoftTeamsUserIdentities',
                'ListTagsForResource'
            ],
            List: [
                'ListCustomActions'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Chatbot = Chatbot;
//# sourceMappingURL=data:application/json;base64,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