"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloud9 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloud9](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloud9.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloud9 extends shared_1.PolicyStatement {
    /**
     * Grants permission to start the Amazon EC2 instance that your AWS Cloud9 IDE connects to
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toActivateEC2Remote() {
        return this.to('ActivateEC2Remote');
    }
    /**
     * Grants permission to create an AWS Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and then hosts the environment on the instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnvironmentName()
     * - .ifInstanceType()
     * - .ifSubnetId()
     * - .ifUserArn()
     * - .ifOwnerArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentEC2.html
     */
    toCreateEnvironmentEC2() {
        return this.to('CreateEnvironmentEC2');
    }
    /**
     * Grants permission to add an environment member to an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     * - .ifPermissions()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_CreateEnvironmentMembership.html
     */
    toCreateEnvironmentMembership() {
        return this.to('CreateEnvironmentMembership');
    }
    /**
     * Grants permission to create an AWS Cloud9 SSH development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEnvironmentName()
     * - .ifOwnerArn()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toCreateEnvironmentSSH() {
        return this.to('CreateEnvironmentSSH');
    }
    /**
     * Grants permission to create an authentication token that allows a connection between the AWS Cloud9 IDE and the user's environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toCreateEnvironmentToken() {
        return this.to('CreateEnvironmentToken');
    }
    /**
     * Grants permission to delete an AWS Cloud9 development environment. If the environment is hosted on an Amazon Elastic Compute Cloud (Amazon EC2) instance, also terminates the instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment() {
        return this.to('DeleteEnvironment');
    }
    /**
     * Grants permission to delete an environment member from an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DeleteEnvironmentMembership.html
     */
    toDeleteEnvironmentMembership() {
        return this.to('DeleteEnvironmentMembership');
    }
    /**
     * Grants permission to get details about the connection to the EC2 development environment, including host, user, and port
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toDescribeEC2Remote() {
        return this.to('DescribeEC2Remote');
    }
    /**
     * Grants permission to get information about environment members for an AWS Cloud9 development environment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentMemberships.html
     */
    toDescribeEnvironmentMemberships() {
        return this.to('DescribeEnvironmentMemberships');
    }
    /**
     * Grants permission to get status information for an AWS Cloud9 development environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironmentStatus.html
     */
    toDescribeEnvironmentStatus() {
        return this.to('DescribeEnvironmentStatus');
    }
    /**
     * Grants permission to get information about AWS Cloud9 development environments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_DescribeEnvironments.html
     */
    toDescribeEnvironments() {
        return this.to('DescribeEnvironments');
    }
    /**
     * Grants permission to get details about the connection to the SSH development environment, including host, user, and port
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toDescribeSSHRemote() {
        return this.to('DescribeSSHRemote');
    }
    /**
     * Grants permission to get configuration information that's used to initialize the AWS Cloud9 IDE
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetEnvironmentConfig() {
        return this.to('GetEnvironmentConfig');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified development environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetEnvironmentSettings() {
        return this.to('GetEnvironmentSettings');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified environment member
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetMembershipSettings() {
        return this.to('GetMembershipSettings');
    }
    /**
     * Grants permission to get the migration experience for a cloud9 user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetMigrationExperiences() {
        return this.to('GetMigrationExperiences');
    }
    /**
     * Grants permission to get the user's public SSH key, which is used by AWS Cloud9 to connect to SSH development environments
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetUserPublicKey() {
        return this.to('GetUserPublicKey');
    }
    /**
     * Grants permission to get the AWS Cloud9 IDE settings for a specified user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toGetUserSettings() {
        return this.to('GetUserSettings');
    }
    /**
     * Grants permission to get a list of AWS Cloud9 development environment identifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListEnvironments.html
     */
    toListEnvironments() {
        return this.to('ListEnvironments');
    }
    /**
     * Grants permission to list tags for a cloud9 environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to set AWS managed temporary credentials on the Amazon EC2 instance that's used by the AWS Cloud9 integrated development environment (IDE)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toModifyTemporaryCredentialsOnEnvironmentEC2() {
        return this.to('ModifyTemporaryCredentialsOnEnvironmentEC2');
    }
    /**
     * Grants permission to add tags to a cloud9 environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a cloud9 environment
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to change the settings of an existing AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        return this.to('UpdateEnvironment');
    }
    /**
     * Grants permission to change the settings of an existing environment member for an AWS Cloud9 development environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUserArn()
     * - .ifEnvironmentId()
     * - .ifPermissions()
     *
     * https://docs.aws.amazon.com/cloud9/latest/APIReference/API_UpdateEnvironmentMembership.html
     */
    toUpdateEnvironmentMembership() {
        return this.to('UpdateEnvironmentMembership');
    }
    /**
     * Grants permission to update the AWS Cloud9 IDE settings for a specified development environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateEnvironmentSettings() {
        return this.to('UpdateEnvironmentSettings');
    }
    /**
     * Grants permission to update the AWS Cloud9 IDE settings for a specified environment member
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateMembershipSettings() {
        return this.to('UpdateMembershipSettings');
    }
    /**
     * Grants permission to update details about the connection to the SSH development environment, including host, user, and port
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateSSHRemote() {
        return this.to('UpdateSSHRemote');
    }
    /**
     * Grants permission to update IDE-specific settings of an AWS Cloud9 user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toUpdateUserSettings() {
        return this.to('UpdateUserSettings');
    }
    /**
     * Grants permission to validate the environment name during the process of creating an AWS Cloud9 development environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloud9/latest/user-guide/security-iam.html#auth-and-access-control-ref-matrix
     */
    toValidateEnvironmentName() {
        return this.to('ValidateEnvironmentName');
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-environment
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloud9:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment:${resourceId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - environment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment ID
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_EnvironmentId
     *
     * Applies to actions:
     * - .toCreateEnvironmentMembership()
     * - .toDescribeEnvironmentMemberships()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentId(value, operator) {
        return this.if(`EnvironmentId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS Cloud9 environment name
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_EnvironmentName
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnvironmentName(value, operator) {
        return this.if(`EnvironmentName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the instance type of the AWS Cloud9 environment's Amazon EC2 instance
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_InstanceType
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceType(value, operator) {
        return this.if(`InstanceType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the owner ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_OwnerArn
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentSSH()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifOwnerArn(value, operator) {
        return this.if(`OwnerArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the type of AWS Cloud9 permissions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_Permissions
     *
     * Applies to actions:
     * - .toCreateEnvironmentMembership()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPermissions(value, operator) {
        return this.if(`Permissions`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the subnet ID that the AWS Cloud9 environment will be created in
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_SubnetId
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnetId(value, operator) {
        return this.if(`SubnetId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the user ARN specified
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloud9.html##awscloud9-cloud9_UserArn
     *
     * Applies to actions:
     * - .toCreateEnvironmentEC2()
     * - .toCreateEnvironmentMembership()
     * - .toDescribeEnvironmentMemberships()
     * - .toGetUserPublicKey()
     * - .toUpdateEnvironmentMembership()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUserArn(value, operator) {
        return this.if(`UserArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [cloud9](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloud9.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloud9';
        this.accessLevelList = {
            Write: [
                'ActivateEC2Remote',
                'CreateEnvironmentEC2',
                'CreateEnvironmentMembership',
                'CreateEnvironmentSSH',
                'DeleteEnvironment',
                'DeleteEnvironmentMembership',
                'ModifyTemporaryCredentialsOnEnvironmentEC2',
                'UpdateEnvironment',
                'UpdateEnvironmentMembership',
                'UpdateEnvironmentSettings',
                'UpdateMembershipSettings',
                'UpdateSSHRemote',
                'UpdateUserSettings'
            ],
            Read: [
                'CreateEnvironmentToken',
                'DescribeEC2Remote',
                'DescribeEnvironmentMemberships',
                'DescribeEnvironmentStatus',
                'DescribeEnvironments',
                'DescribeSSHRemote',
                'GetEnvironmentConfig',
                'GetEnvironmentSettings',
                'GetMembershipSettings',
                'GetMigrationExperiences',
                'GetUserPublicKey',
                'GetUserSettings',
                'ListEnvironments',
                'ListTagsForResource',
                'ValidateEnvironmentName'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloud9 = Cloud9;
//# sourceMappingURL=data:application/json;base64,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