"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudformation = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudformation](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudformation.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudformation extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate trusted access between StackSets and Organizations. With trusted access between StackSets and Organizations activated, the management account has permissions to create and manage StackSets for your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ActivateOrganizationsAccess.html
     */
    toActivateOrganizationsAccess() {
        return this.to('ActivateOrganizationsAccess');
    }
    /**
     * Grants permission to activate a public third-party extension, making it available for use in stack templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ActivateType.html
     */
    toActivateType() {
        return this.to('ActivateType');
    }
    /**
     * Grants permission to return configuration data for the specified CloudFormation extensions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_BatchDescribeTypeConfigurations.html
     */
    toBatchDescribeTypeConfigurations() {
        return this.to('BatchDescribeTypeConfigurations');
    }
    /**
     * Grants permission to cancel an update on the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html
     */
    toCancelUpdateStack() {
        return this.to('CancelUpdateStack');
    }
    /**
     * Grants permission to continue rolling back a stack that is in the UPDATE_ROLLBACK_FAILED state to the UPDATE_ROLLBACK_COMPLETE state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ContinueUpdateRollback.html
     */
    toContinueUpdateRollback() {
        return this.to('ContinueUpdateRollback');
    }
    /**
     * Grants permission to create a list of changes for a stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     * - .ifResourceTypes()
     * - .ifImportResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateChangeSet.html
     */
    toCreateChangeSet() {
        return this.to('CreateChangeSet');
    }
    /**
     * Grants permission to create a template from existing resources that are not already managed with CloudFormation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateGeneratedTemplate.html
     */
    toCreateGeneratedTemplate() {
        return this.to('CreateGeneratedTemplate');
    }
    /**
     * Grants permission to create a stack as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        return this.to('CreateStack');
    }
    /**
     * Grants permission to create stack instances for the specified accounts, within the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackInstances.html
     */
    toCreateStackInstances() {
        return this.to('CreateStackInstances');
    }
    /**
     * Grants permission to create a stack refactor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackRefactor.html
     */
    toCreateStackRefactor() {
        return this.to('CreateStackRefactor');
    }
    /**
     * Grants permission to create a stackset as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackSet.html
     */
    toCreateStackSet() {
        return this.to('CreateStackSet');
    }
    /**
     * Grants permission to upload templates to Amazon S3 buckets. Used only by the AWS CloudFormation console and is not documented in the API reference
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     */
    toCreateUploadBucket() {
        return this.to('CreateUploadBucket');
    }
    /**
     * Grants permission to deactivate trusted access between StackSets and Organizations. If trusted access is deactivated, the management account does not have permissions to create and manage service-managed StackSets for your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeactivateOrganizationsAccess.html
     */
    toDeactivateOrganizationsAccess() {
        return this.to('DeactivateOrganizationsAccess');
    }
    /**
     * Grants permission to deactivate a public extension that was previously activated in this account and region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeactivateType.html
     */
    toDeactivateType() {
        return this.to('DeactivateType');
    }
    /**
     * Grants permission to delete the specified change set. Deleting change sets ensures that no one executes the wrong change set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteChangeSet.html
     */
    toDeleteChangeSet() {
        return this.to('DeleteChangeSet');
    }
    /**
     * Grants permission to delete a generated template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteGeneratedTemplate.html
     */
    toDeleteGeneratedTemplate() {
        return this.to('DeleteGeneratedTemplate');
    }
    /**
     * Grants permission to delete a specified stack
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        return this.to('DeleteStack');
    }
    /**
     * Grants permission to delete stack instances for the specified accounts, in the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackInstances.html
     */
    toDeleteStackInstances() {
        return this.to('DeleteStackInstances');
    }
    /**
     * Grants permission to delete a specified stackset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackSet.html
     */
    toDeleteStackSet() {
        return this.to('DeleteStackSet');
    }
    /**
     * Grants permission to deregister an existing CloudFormation type or type version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeregisterType.html
     */
    toDeregisterType() {
        return this.to('DeregisterType');
    }
    /**
     * Grants permission to retrieve your account's AWS CloudFormation limits
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        return this.to('DescribeAccountLimits');
    }
    /**
     * Grants permission to return the description for the specified change set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSet.html
     */
    toDescribeChangeSet() {
        return this.to('DescribeChangeSet');
    }
    /**
     * Grants permission to return the Hook invocation information for the specified change set
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSetHooks.html
     */
    toDescribeChangeSetHooks() {
        return this.to('DescribeChangeSetHooks');
    }
    /**
     * Grants permission to describe a generated template. The output includes details about the progress of the creation of a generated template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeGeneratedTemplate.html
     */
    toDescribeGeneratedTemplate() {
        return this.to('DescribeGeneratedTemplate');
    }
    /**
     * Grants permission to return information about the account's OrganizationAccess status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeOrganizationsAccess.html
     */
    toDescribeOrganizationsAccess() {
        return this.to('DescribeOrganizationsAccess');
    }
    /**
     * Grants permission to return information about a CloudFormation extension publisher
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribePublisher.html
     */
    toDescribePublisher() {
        return this.to('DescribePublisher');
    }
    /**
     * Grants permission to describe details of a resource scan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeResourceScan.html
     */
    toDescribeResourceScan() {
        return this.to('DescribeResourceScan');
    }
    /**
     * Grants permission to return information about a stack drift detection operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackDriftDetectionStatus.html
     */
    toDescribeStackDriftDetectionStatus() {
        return this.to('DescribeStackDriftDetectionStatus');
    }
    /**
     * Grants permission to return all stack related events for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackEvents.html
     */
    toDescribeStackEvents() {
        return this.to('DescribeStackEvents');
    }
    /**
     * Grants permission to return the stack instance that's associated with the specified stack set, AWS account, and region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackInstance.html
     */
    toDescribeStackInstance() {
        return this.to('DescribeStackInstance');
    }
    /**
     * Grants permission to return the description for the specified stack refactor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackRefactor.html
     */
    toDescribeStackRefactor() {
        return this.to('DescribeStackRefactor');
    }
    /**
     * Grants permission to return a description of the specified resource in the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResource.html
     */
    toDescribeStackResource() {
        return this.to('DescribeStackResource');
    }
    /**
     * Grants permission to return drift information for the resources that have been checked for drift in the specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResourceDrifts.html
     */
    toDescribeStackResourceDrifts() {
        return this.to('DescribeStackResourceDrifts');
    }
    /**
     * Grants permission to return AWS resource descriptions for running and deleted stacks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResources.html
     */
    toDescribeStackResources() {
        return this.to('DescribeStackResources');
    }
    /**
     * Grants permission to return the description of the specified stack set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSet.html
     */
    toDescribeStackSet() {
        return this.to('DescribeStackSet');
    }
    /**
     * Grants permission to return the description of the specified stack set operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSetOperation.html
     */
    toDescribeStackSetOperation() {
        return this.to('DescribeStackSetOperation');
    }
    /**
     * Grants permission to return the description for the specified stack, and to all stacks when used in combination with the ListStacks action
     *
     * Access Level: List
     *
     * Dependent actions:
     * - cloudformation:ListStacks
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        return this.to('DescribeStacks');
    }
    /**
     * Grants permission to return information about the CloudFormation type requested
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeType.html
     */
    toDescribeType() {
        return this.to('DescribeType');
    }
    /**
     * Grants permission to return information about the registration process for a CloudFormation type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeTypeRegistration.html
     */
    toDescribeTypeRegistration() {
        return this.to('DescribeTypeRegistration');
    }
    /**
     * Grants permission to detects whether a stack's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackDrift.html
     */
    toDetectStackDrift() {
        return this.to('DetectStackDrift');
    }
    /**
     * Grants permission to return information about whether a resource's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackResourceDrift.html
     */
    toDetectStackResourceDrift() {
        return this.to('DetectStackResourceDrift');
    }
    /**
     * Grants permission to enable users to detect drift on a stack set and the stack instances that belong to that stack set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackSetDrift.html
     */
    toDetectStackSetDrift() {
        return this.to('DetectStackSetDrift');
    }
    /**
     * Grants permission to return the estimated monthly cost of a template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_EstimateTemplateCost.html
     */
    toEstimateTemplateCost() {
        return this.to('EstimateTemplateCost');
    }
    /**
     * Grants permission to update a stack using the input information that was provided when the specified change set was created
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteChangeSet.html
     */
    toExecuteChangeSet() {
        return this.to('ExecuteChangeSet');
    }
    /**
     * Grants permission to execute a stack refactor using the input information that was provided when the specified stack refactor was created
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteStackRefactor.html
     */
    toExecuteStackRefactor() {
        return this.to('ExecuteStackRefactor');
    }
    /**
     * Grants permission to retrieve a generated template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetGeneratedTemplate.html
     */
    toGetGeneratedTemplate() {
        return this.to('GetGeneratedTemplate');
    }
    /**
     * Grants permission to return the stack policy for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetStackPolicy.html
     */
    toGetStackPolicy() {
        return this.to('GetStackPolicy');
    }
    /**
     * Grants permission to return the template body for a specified stack
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        return this.to('GetTemplate');
    }
    /**
     * Grants permission to return information about a new or existing template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplateSummary.html
     */
    toGetTemplateSummary() {
        return this.to('GetTemplateSummary');
    }
    /**
     * Grants permission to enable users to import existing stacks to a new or existing stackset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ImportStacksToStackSet.html
     */
    toImportStacksToStackSet() {
        return this.to('ImportStacksToStackSet');
    }
    /**
     * Grants permission to return the ID and status of each active change set for a stack. For example, AWS CloudFormation lists change sets that are in the CREATE_IN_PROGRESS or CREATE_PENDING state
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListChangeSets.html
     */
    toListChangeSets() {
        return this.to('ListChangeSets');
    }
    /**
     * Grants permission to list all exported output values in the account and region in which you call this action
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list your generated templates in this Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListGeneratedTemplates.html
     */
    toListGeneratedTemplates() {
        return this.to('ListGeneratedTemplates');
    }
    /**
     * Grants permission to return Hook invocations result information for the specified target
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListHookResults.html
     */
    toListHookResults() {
        return this.to('ListHookResults');
    }
    /**
     * Grants permission to list all stacks that are importing an exported output value
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list the related resources for a list of resources from a resource scan. The response indicates whether each returned resource is already managed by CloudFormation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScanRelatedResources.html
     */
    toListResourceScanRelatedResources() {
        return this.to('ListResourceScanRelatedResources');
    }
    /**
     * Grants permission to list the resources from a resource scan. The results can be filtered by resource identifier, resource type prefix, tag key, and tag value
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScanResources.html
     */
    toListResourceScanResources() {
        return this.to('ListResourceScanResources');
    }
    /**
     * Grants permission to list the resource scans from newest to oldest. By default it will return up to 10 resource scans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListResourceScans.html
     */
    toListResourceScans() {
        return this.to('ListResourceScans');
    }
    /**
     * Grants permission to return drift information for the resources that have been checked for drift in the specified stack instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackInstanceResourceDrifts.html
     */
    toListStackInstanceResourceDrifts() {
        return this.to('ListStackInstanceResourceDrifts');
    }
    /**
     * Grants permission to return summary information about stack instances that are associated with the specified stack set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackInstances() {
        return this.to('ListStackInstances');
    }
    /**
     * Grants permission to return the list of actions of the specified stack refactor
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackRefactorActions.html
     */
    toListStackRefactorActions() {
        return this.to('ListStackRefactorActions');
    }
    /**
     * Grants permission to return the ID and status of each active stack refactor
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackRefactors.html
     */
    toListStackRefactors() {
        return this.to('ListStackRefactors');
    }
    /**
     * Grants permission to return descriptions of all resources of the specified stack
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackResources.html
     */
    toListStackResources() {
        return this.to('ListStackResources');
    }
    /**
     * Grants permission to return summary information about StackSet Auto Deployment Targets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetAutoDeploymentTargets.html
     */
    toListStackSetAutoDeploymentTargets() {
        return this.to('ListStackSetAutoDeploymentTargets');
    }
    /**
     * Grants permission to return summary information about the results of a stack set operation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperationResults.html
     */
    toListStackSetOperationResults() {
        return this.to('ListStackSetOperationResults');
    }
    /**
     * Grants permission to return summary information about operations performed on a stack set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperations.html
     */
    toListStackSetOperations() {
        return this.to('ListStackSetOperations');
    }
    /**
     * Grants permission to return summary information about stack sets that are associated with the user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackSets() {
        return this.to('ListStackSets');
    }
    /**
     * Grants permission to return the summary information for stacks whose status matches the specified StackStatusFilter. In combination with the DescribeStacks action, grants permission to list descriptions for stacks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStacks.html
     */
    toListStacks() {
        return this.to('ListStacks');
    }
    /**
     * Grants permission to list CloudFormation type registration attempts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeRegistrations.html
     */
    toListTypeRegistrations() {
        return this.to('ListTypeRegistrations');
    }
    /**
     * Grants permission to list versions of a particular CloudFormation type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeVersions.html
     */
    toListTypeVersions() {
        return this.to('ListTypeVersions');
    }
    /**
     * Grants permission to list available CloudFormation types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        return this.to('ListTypes');
    }
    /**
     * Grants permission to publish the specified extension to the CloudFormation registry as a public extension in this region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_PublishType.html
     */
    toPublishType() {
        return this.to('PublishType');
    }
    /**
     * Grants permission to record the handler progress
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RecordHandlerProgress.html
     */
    toRecordHandlerProgress() {
        return this.to('RecordHandlerProgress');
    }
    /**
     * Grants permission to register account as a publisher of public extensions in the CloudFormation registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterPublisher.html
     */
    toRegisterPublisher() {
        return this.to('RegisterPublisher');
    }
    /**
     * Grants permission to register a new CloudFormation type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterType.html
     */
    toRegisterType() {
        return this.to('RegisterType');
    }
    /**
     * Grants permission to rollback the stack to the last stable state
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RollbackStack.html
     */
    toRollbackStack() {
        return this.to('RollbackStack');
    }
    /**
     * Grants permission to set a stack policy for a specified stack
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifStackPolicyUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetStackPolicy.html
     */
    toSetStackPolicy() {
        return this.to('SetStackPolicy');
    }
    /**
     * Grants permission to set the configuration data for a registered CloudFormation extension, in the given account and region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeConfiguration.html
     */
    toSetTypeConfiguration() {
        return this.to('SetTypeConfiguration');
    }
    /**
     * Grants permission to set which version of a CloudFormation type applies to CloudFormation operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeDefaultVersion.html
     */
    toSetTypeDefaultVersion() {
        return this.to('SetTypeDefaultVersion');
    }
    /**
     * Grants permission to send a signal to the specified resource with a success or failure status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SignalResource.html
     */
    toSignalResource() {
        return this.to('SignalResource');
    }
    /**
     * Grants permission to start a scan of the resources in this account in this Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StartResourceScan.html
     */
    toStartResourceScan() {
        return this.to('StartResourceScan');
    }
    /**
     * Grants permission to stop an in-progress operation on a stack set and its associated stack instances
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StopStackSetOperation.html
     */
    toStopStackSetOperation() {
        return this.to('StopStackSetOperation');
    }
    /**
     * Grants permission to tag cloudformation resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a registered extension to make sure it meets all necessary requirements for being published in the CloudFormation registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_TestType.html
     */
    toTestType() {
        return this.to('TestType');
    }
    /**
     * Grants permission to untag cloudformation resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a generated template. This can be used to change the name, add and remove resources, refresh resources, and change the DeletionPolicy and UpdateReplacePolicy settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateGeneratedTemplate.html
     */
    toUpdateGeneratedTemplate() {
        return this.to('UpdateGeneratedTemplate');
    }
    /**
     * Grants permission to update a stack as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        return this.to('UpdateStack');
    }
    /**
     * Grants permission to update the parameter values for stack instances for the specified accounts, within the specified regions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetRegion()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackInstances.html
     */
    toUpdateStackInstances() {
        return this.to('UpdateStackInstances');
    }
    /**
     * Grants permission to update a stackset as specified in the template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifTargetRegion()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html
     */
    toUpdateStackSet() {
        return this.to('UpdateStackSet');
    }
    /**
     * Grants permission to update termination protection for the specified stack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateTerminationProtection.html
     */
    toUpdateTerminationProtection() {
        return this.to('UpdateTerminationProtection');
    }
    /**
     * Grants permission to validate a specified template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifTemplateUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ValidateTemplate.html
     */
    toValidateTemplate() {
        return this.to('ValidateTemplate');
    }
    /**
     * Adds a resource of type changeset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15c11
     *
     * @param changeSetName - Identifier for the changeSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChangeset(changeSetName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:changeSet/${changeSetName}/${id}`);
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15b9
     *
     * @param stackName - Identifier for the stackName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStack(stackName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stack/${stackName}/${id}`);
    }
    /**
     * Adds a resource of type stackset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset
     *
     * @param stackSetName - Identifier for the stackSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStackset(stackSetName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stackset/${stackSetName}:${id}`);
    }
    /**
     * Adds a resource of type stackset-target to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     *
     * @param stackSetTarget - Identifier for the stackSetTarget.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStacksetTarget(stackSetTarget, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:stackset-target/${stackSetTarget}`);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html
     *
     * @param type - Identifier for the type.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onType(type, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:type/resource/${type}`);
    }
    /**
     * Adds a resource of type generatedtemplate to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/generate-IaC.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGeneratedtemplate(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:generatedTemplate/${id}`);
    }
    /**
     * Adds a resource of type resourcescan to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/generate-IaC.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResourcescan(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudformation:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:resourceScan/${id}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toTagResource()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to resource types:
     * - changeset
     * - stack
     * - stackset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackInstances()
     * - .toCreateStackSet()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an AWS CloudFormation change set name. Use to control which change sets IAM users can execute or delete
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toDeleteChangeSet()
     * - .toDescribeChangeSet()
     * - .toDescribeChangeSetHooks()
     * - .toExecuteChangeSet()
     * - .toListHookResults()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeSetName(value, operator) {
        return this.if(`ChangeSetName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of a resource-mutating API action. Use to control which APIs IAM users can use to add or remove tags on a stack or stack set
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`CreateAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the template resource types, such as AWS::EC2::Instance. Use to control which resource types IAM users can work with when they want to import a resource into a stack
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImportResourceTypes(value, operator) {
        return this.if(`ImportResourceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the template resource types, such as AWS::EC2::Instance. Use to control which resource types IAM users can work with when they create or update a stack
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTypes(value, operator) {
        return this.if(`ResourceTypes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of an IAM service role. Use to control which service role IAM users can use to work with stacks or change sets
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toContinueUpdateRollback()
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toDeleteStack()
     * - .toRollbackStack()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRoleArn(value, operator) {
        return this.if(`RoleArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by an Amazon S3 stack policy URL. Use to control which stack policies IAM users can associate with a stack during a create or update stack action
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toSetStackPolicy()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStackPolicyUrl(value, operator) {
        return this.if(`StackPolicyUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by stack set target region. Use to control which regions IAM users can use when they create or update stack sets
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateStackInstances()
     * - .toDeleteStackInstances()
     * - .toUpdateStackInstances()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargetRegion(value, operator) {
        return this.if(`TargetRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an Amazon S3 template URL. Use to control which templates IAM users can use when they create or update stacks
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#using-iam-template-conditions
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toEstimateTemplateCost()
     * - .toGetTemplateSummary()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     * - .toValidateTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateUrl(value, operator) {
        return this.if(`TemplateUrl`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cloudformation](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscloudformation.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudformation';
        this.accessLevelList = {
            Write: [
                'ActivateOrganizationsAccess',
                'ActivateType',
                'CancelUpdateStack',
                'ContinueUpdateRollback',
                'CreateChangeSet',
                'CreateGeneratedTemplate',
                'CreateStack',
                'CreateStackInstances',
                'CreateStackRefactor',
                'CreateStackSet',
                'CreateUploadBucket',
                'DeactivateOrganizationsAccess',
                'DeactivateType',
                'DeleteChangeSet',
                'DeleteGeneratedTemplate',
                'DeleteStack',
                'DeleteStackInstances',
                'DeleteStackSet',
                'DeregisterType',
                'ExecuteChangeSet',
                'ExecuteStackRefactor',
                'ImportStacksToStackSet',
                'PublishType',
                'RecordHandlerProgress',
                'RegisterPublisher',
                'RegisterType',
                'RollbackStack',
                'SetTypeConfiguration',
                'SetTypeDefaultVersion',
                'SignalResource',
                'StartResourceScan',
                'StopStackSetOperation',
                'TestType',
                'UpdateGeneratedTemplate',
                'UpdateStack',
                'UpdateStackInstances',
                'UpdateStackSet',
                'UpdateTerminationProtection'
            ],
            Read: [
                'BatchDescribeTypeConfigurations',
                'DescribeAccountLimits',
                'DescribeChangeSet',
                'DescribeChangeSetHooks',
                'DescribeGeneratedTemplate',
                'DescribeOrganizationsAccess',
                'DescribePublisher',
                'DescribeResourceScan',
                'DescribeStackDriftDetectionStatus',
                'DescribeStackEvents',
                'DescribeStackInstance',
                'DescribeStackRefactor',
                'DescribeStackResource',
                'DescribeStackResourceDrifts',
                'DescribeStackResources',
                'DescribeStackSet',
                'DescribeStackSetOperation',
                'DescribeType',
                'DescribeTypeRegistration',
                'DetectStackDrift',
                'DetectStackResourceDrift',
                'DetectStackSetDrift',
                'EstimateTemplateCost',
                'GetGeneratedTemplate',
                'GetStackPolicy',
                'GetTemplate',
                'GetTemplateSummary',
                'ValidateTemplate'
            ],
            List: [
                'DescribeStacks',
                'ListChangeSets',
                'ListExports',
                'ListGeneratedTemplates',
                'ListHookResults',
                'ListImports',
                'ListResourceScanRelatedResources',
                'ListResourceScanResources',
                'ListResourceScans',
                'ListStackInstanceResourceDrifts',
                'ListStackInstances',
                'ListStackRefactorActions',
                'ListStackRefactors',
                'ListStackResources',
                'ListStackSetAutoDeploymentTargets',
                'ListStackSetOperationResults',
                'ListStackSetOperations',
                'ListStackSets',
                'ListStacks',
                'ListTypeRegistrations',
                'ListTypeVersions',
                'ListTypes'
            ],
            'Permissions management': [
                'SetStackPolicy'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloudformation = Cloudformation;
//# sourceMappingURL=data:application/json;base64,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