"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudfront = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudfront extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended log delivery for a distribution
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html#AWS-logs-infrastructure-V2-service-specific
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to associate an alias to a CloudFront distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateAlias.html
     */
    toAssociateAlias() {
        return this.to('AssociateAlias');
    }
    /**
     * Grants permission to associate a distribution tenant with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateDistributionTenantWebACL.html
     */
    toAssociateDistributionTenantWebACL() {
        return this.to('AssociateDistributionTenantWebACL');
    }
    /**
     * Grants permission to associate a distribution with an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_AssociateDistributionWebACL.html
     */
    toAssociateDistributionWebACL() {
        return this.to('AssociateDistributionWebACL');
    }
    /**
     * Grants permission to copy an existing distribution and create a new web distribution
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudfront:CopyDistribution
     * - cloudfront:CreateDistribution
     * - cloudfront:GetDistribution
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CopyDistribution.html
     */
    toCopyDistribution() {
        return this.to('CopyDistribution');
    }
    /**
     * Grants permission to create an Anycast static IP list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateAnycastIpList.html
     */
    toCreateAnycastIpList() {
        return this.to('CreateAnycastIpList');
    }
    /**
     * Grants permission to add a new cache policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCachePolicy.html
     */
    toCreateCachePolicy() {
        return this.to('CreateCachePolicy');
    }
    /**
     * Grants permission to create a new CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateCloudFrontOriginAccessIdentity.html
     */
    toCreateCloudFrontOriginAccessIdentity() {
        return this.to('CreateCloudFrontOriginAccessIdentity');
    }
    /**
     * Grants permission to create a connection group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateConnectionGroup.html
     */
    toCreateConnectionGroup() {
        return this.to('CreateConnectionGroup');
    }
    /**
     * Grants permission to add a new continuous-deployment policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateContinuousDeploymentPolicy.html
     */
    toCreateContinuousDeploymentPolicy() {
        return this.to('CreateContinuousDeploymentPolicy');
    }
    /**
     * Grants permission to create a new web distribution
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - cloudfront:CreateConnectionGroup
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistribution.html
     */
    toCreateDistribution() {
        return this.to('CreateDistribution');
    }
    /**
     * Grants permission to create a distribution tenant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateDistributionTenant.html
     */
    toCreateDistributionTenant() {
        return this.to('CreateDistributionTenant');
    }
    /**
     * Grants permission to create a new field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionConfig.html
     */
    toCreateFieldLevelEncryptionConfig() {
        return this.to('CreateFieldLevelEncryptionConfig');
    }
    /**
     * Grants permission to create a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFieldLevelEncryptionProfile.html
     */
    toCreateFieldLevelEncryptionProfile() {
        return this.to('CreateFieldLevelEncryptionProfile');
    }
    /**
     * Grants permission to create a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction() {
        return this.to('CreateFunction');
    }
    /**
     * Grants permission to create a new invalidation batch request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidation.html
     */
    toCreateInvalidation() {
        return this.to('CreateInvalidation');
    }
    /**
     * Grants permission to create an invalidation for a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateInvalidationForDistributionTenant.html
     */
    toCreateInvalidationForDistributionTenant() {
        return this.to('CreateInvalidationForDistributionTenant');
    }
    /**
     * Grants permission to add a new key group to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateKeyGroup.html
     */
    toCreateKeyGroup() {
        return this.to('CreateKeyGroup');
    }
    /**
     * Grants permission to create a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateKeyValueStore.html
     */
    toCreateKeyValueStore() {
        return this.to('CreateKeyValueStore');
    }
    /**
     * Grants permission to enable additional CloudWatch metrics for the specified CloudFront distribution. The additional metrics incur an additional cost
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateMonitoringSubscription.html
     */
    toCreateMonitoringSubscription() {
        return this.to('CreateMonitoringSubscription');
    }
    /**
     * Grants permission to create a new origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginAccessControl.html
     */
    toCreateOriginAccessControl() {
        return this.to('CreateOriginAccessControl');
    }
    /**
     * Grants permission to add a new origin request policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateOriginRequestPolicy.html
     */
    toCreateOriginRequestPolicy() {
        return this.to('CreateOriginRequestPolicy');
    }
    /**
     * Grants permission to add a new public key to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreatePublicKey.html
     */
    toCreatePublicKey() {
        return this.to('CreatePublicKey');
    }
    /**
     * Grants permission to create a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateRealtimeLogConfig.html
     */
    toCreateRealtimeLogConfig() {
        return this.to('CreateRealtimeLogConfig');
    }
    /**
     * Grants permission to add a new response headers policy to CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateResponseHeadersPolicy.html
     */
    toCreateResponseHeadersPolicy() {
        return this.to('CreateResponseHeadersPolicy');
    }
    /**
     * Grants permission to create a new savings plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toCreateSavingsPlan() {
        return this.to('CreateSavingsPlan');
    }
    /**
     * Grants permission to create a new RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistribution.html
     */
    toCreateStreamingDistribution() {
        return this.to('CreateStreamingDistribution');
    }
    /**
     * Grants permission to create a new RTMP distribution with tags
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateStreamingDistributionWithTags.html
     */
    toCreateStreamingDistributionWithTags() {
        return this.to('CreateStreamingDistributionWithTags');
    }
    /**
     * Grants permission to create a VPC origin
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_CreateVpcOrigin.html
     */
    toCreateVpcOrigin() {
        return this.to('CreateVpcOrigin');
    }
    /**
     * Grants permission to delete an Anycast static IP list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteAnycastIpList.html
     */
    toDeleteAnycastIpList() {
        return this.to('DeleteAnycastIpList');
    }
    /**
     * Grants permission to delete a cache policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCachePolicy.html
     */
    toDeleteCachePolicy() {
        return this.to('DeleteCachePolicy');
    }
    /**
     * Grants permission to delete a CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteCloudFrontOriginAccessIdentity.html
     */
    toDeleteCloudFrontOriginAccessIdentity() {
        return this.to('DeleteCloudFrontOriginAccessIdentity');
    }
    /**
     * Grants permission to delete a connection group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteConnectionGroup.html
     */
    toDeleteConnectionGroup() {
        return this.to('DeleteConnectionGroup');
    }
    /**
     * Grants permission to delete a continuous-deployment policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteContinuousDeploymentPolicy.html
     */
    toDeleteContinuousDeploymentPolicy() {
        return this.to('DeleteContinuousDeploymentPolicy');
    }
    /**
     * Grants permission to delete a web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistribution.html
     */
    toDeleteDistribution() {
        return this.to('DeleteDistribution');
    }
    /**
     * Grants permission to delete a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteDistributionTenant.html
     */
    toDeleteDistributionTenant() {
        return this.to('DeleteDistributionTenant');
    }
    /**
     * Grants permission to delete a field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionConfig.html
     */
    toDeleteFieldLevelEncryptionConfig() {
        return this.to('DeleteFieldLevelEncryptionConfig');
    }
    /**
     * Grants permission to delete a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFieldLevelEncryptionProfile.html
     */
    toDeleteFieldLevelEncryptionProfile() {
        return this.to('DeleteFieldLevelEncryptionProfile');
    }
    /**
     * Grants permission to delete a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction() {
        return this.to('DeleteFunction');
    }
    /**
     * Grants permission to delete a key group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteKeyGroup.html
     */
    toDeleteKeyGroup() {
        return this.to('DeleteKeyGroup');
    }
    /**
     * Grants permission to delete a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteKeyValueStore.html
     */
    toDeleteKeyValueStore() {
        return this.to('DeleteKeyValueStore');
    }
    /**
     * Grants permission to disable additional CloudWatch metrics for the specified CloudFront distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteMonitoringSubscription.html
     */
    toDeleteMonitoringSubscription() {
        return this.to('DeleteMonitoringSubscription');
    }
    /**
     * Grants permission to delete an origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginAccessControl.html
     */
    toDeleteOriginAccessControl() {
        return this.to('DeleteOriginAccessControl');
    }
    /**
     * Grants permission to delete an origin request policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteOriginRequestPolicy.html
     */
    toDeleteOriginRequestPolicy() {
        return this.to('DeleteOriginRequestPolicy');
    }
    /**
     * Grants permission to delete a public key from CloudFront
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeletePublicKey.html
     */
    toDeletePublicKey() {
        return this.to('DeletePublicKey');
    }
    /**
     * Grants permission to delete a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteRealtimeLogConfig.html
     */
    toDeleteRealtimeLogConfig() {
        return this.to('DeleteRealtimeLogConfig');
    }
    /**
     * Grants permission to delete a response headers policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteResponseHeadersPolicy.html
     */
    toDeleteResponseHeadersPolicy() {
        return this.to('DeleteResponseHeadersPolicy');
    }
    /**
     * Grants permission to delete an RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteStreamingDistribution.html
     */
    toDeleteStreamingDistribution() {
        return this.to('DeleteStreamingDistribution');
    }
    /**
     * Grants permission to delete a VPC origin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DeleteVpcOrigin.html
     */
    toDeleteVpcOrigin() {
        return this.to('DeleteVpcOrigin');
    }
    /**
     * Grants permission to get a CloudFront function summary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DescribeFunction.html
     */
    toDescribeFunction() {
        return this.to('DescribeFunction');
    }
    /**
     * Grants permission to get a CloudFront KeyValueStore summary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DescribeKeyValueStore.html
     */
    toDescribeKeyValueStore() {
        return this.to('DescribeKeyValueStore');
    }
    /**
     * Grants permission to disassociate a distribution tenant from an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DisassociateDistributionTenantWebACL.html
     */
    toDisassociateDistributionTenantWebACL() {
        return this.to('DisassociateDistributionTenantWebACL');
    }
    /**
     * Grants permission to disassociate a distribution from an AWS WAF web ACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_DisassociateDistributionWebACL.html
     */
    toDisassociateDistributionWebACL() {
        return this.to('DisassociateDistributionWebACL');
    }
    /**
     * Grants permission to get an Anycast static IP list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetAnycastIpList.html
     */
    toGetAnycastIpList() {
        return this.to('GetAnycastIpList');
    }
    /**
     * Grants permission to get the cache policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicy.html
     */
    toGetCachePolicy() {
        return this.to('GetCachePolicy');
    }
    /**
     * Grants permission to get the cache policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCachePolicyConfig.html
     */
    toGetCachePolicyConfig() {
        return this.to('GetCachePolicyConfig');
    }
    /**
     * Grants permission to get the information about a CloudFront origin access identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentity.html
     */
    toGetCloudFrontOriginAccessIdentity() {
        return this.to('GetCloudFrontOriginAccessIdentity');
    }
    /**
     * Grants permission to get the configuration information about a Cloudfront origin access identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetCloudFrontOriginAccessIdentityConfig.html
     */
    toGetCloudFrontOriginAccessIdentityConfig() {
        return this.to('GetCloudFrontOriginAccessIdentityConfig');
    }
    /**
     * Grants permission to get information about a connection group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetConnectionGroup.html
     */
    toGetConnectionGroup() {
        return this.to('GetConnectionGroup');
    }
    /**
     * Grants permission to get information about a connection group by the specified routing endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetConnectionGroupByRoutingEndpoint.html
     */
    toGetConnectionGroupByRoutingEndpoint() {
        return this.to('GetConnectionGroupByRoutingEndpoint');
    }
    /**
     * Grants permission to get the continuous-deployment policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetContinuousDeploymentPolicy.html
     */
    toGetContinuousDeploymentPolicy() {
        return this.to('GetContinuousDeploymentPolicy');
    }
    /**
     * Grants permission to get the continuous-deployment policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetContinuousDeploymentPolicyConfig.html
     */
    toGetContinuousDeploymentPolicyConfig() {
        return this.to('GetContinuousDeploymentPolicyConfig');
    }
    /**
     * Grants permission to get the information about a web distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistribution.html
     */
    toGetDistribution() {
        return this.to('GetDistribution');
    }
    /**
     * Grants permission to get the configuration information about a distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionConfig.html
     */
    toGetDistributionConfig() {
        return this.to('GetDistributionConfig');
    }
    /**
     * Grants permission to get information about a distribution tenant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionTenant.html
     */
    toGetDistributionTenant() {
        return this.to('GetDistributionTenant');
    }
    /**
     * Grants permission to get information about a distribution tenant by the associated domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetDistributionTenantByDomain.html
     */
    toGetDistributionTenantByDomain() {
        return this.to('GetDistributionTenantByDomain');
    }
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryption.html
     */
    toGetFieldLevelEncryption() {
        return this.to('GetFieldLevelEncryption');
    }
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionConfig.html
     */
    toGetFieldLevelEncryptionConfig() {
        return this.to('GetFieldLevelEncryptionConfig');
    }
    /**
     * Grants permission to get the field-level encryption configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfile.html
     */
    toGetFieldLevelEncryptionProfile() {
        return this.to('GetFieldLevelEncryptionProfile');
    }
    /**
     * Grants permission to get the field-level encryption profile configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFieldLevelEncryptionProfileConfig.html
     */
    toGetFieldLevelEncryptionProfileConfig() {
        return this.to('GetFieldLevelEncryptionProfileConfig');
    }
    /**
     * Grants permission to get a CloudFront function's code
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetFunction.html
     */
    toGetFunction() {
        return this.to('GetFunction');
    }
    /**
     * Grants permission to get the information about an invalidation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidation.html
     */
    toGetInvalidation() {
        return this.to('GetInvalidation');
    }
    /**
     * Grants permission to get information about an invalidation for a distribution tenant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetInvalidationForDistributionTenant.html
     */
    toGetInvalidationForDistributionTenant() {
        return this.to('GetInvalidationForDistributionTenant');
    }
    /**
     * Grants permission to get a key group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetKeyGroup.html
     */
    toGetKeyGroup() {
        return this.to('GetKeyGroup');
    }
    /**
     * Grants permission to get a key group configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetKeyGroupConfig.html
     */
    toGetKeyGroupConfig() {
        return this.to('GetKeyGroupConfig');
    }
    /**
     * Grants permission to get details about a CloudFront managed certificate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetManagedCertificateDetails.html
     */
    toGetManagedCertificateDetails() {
        return this.to('GetManagedCertificateDetails');
    }
    /**
     * Grants permission to get information about whether additional CloudWatch metrics are enabled for the specified CloudFront distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetMonitoringSubscription.html
     */
    toGetMonitoringSubscription() {
        return this.to('GetMonitoringSubscription');
    }
    /**
     * Grants permission to get the origin access control
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginAccessControl.html
     */
    toGetOriginAccessControl() {
        return this.to('GetOriginAccessControl');
    }
    /**
     * Grants permission to get the origin access control configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginAccessControlConfig.html
     */
    toGetOriginAccessControlConfig() {
        return this.to('GetOriginAccessControlConfig');
    }
    /**
     * Grants permission to get the origin request policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicy.html
     */
    toGetOriginRequestPolicy() {
        return this.to('GetOriginRequestPolicy');
    }
    /**
     * Grants permission to get the origin request policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetOriginRequestPolicyConfig.html
     */
    toGetOriginRequestPolicyConfig() {
        return this.to('GetOriginRequestPolicyConfig');
    }
    /**
     * Grants permission to get the public key information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        return this.to('GetPublicKey');
    }
    /**
     * Grants permission to get the public key configuration information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetPublicKeyConfig.html
     */
    toGetPublicKeyConfig() {
        return this.to('GetPublicKeyConfig');
    }
    /**
     * Grants permission to get a real-time log configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetRealtimeLogConfig.html
     */
    toGetRealtimeLogConfig() {
        return this.to('GetRealtimeLogConfig');
    }
    /**
     * Grants permission to get the response headers policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetResponseHeadersPolicy.html
     */
    toGetResponseHeadersPolicy() {
        return this.to('GetResponseHeadersPolicy');
    }
    /**
     * Grants permission to get the response headers policy configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetResponseHeadersPolicyConfig.html
     */
    toGetResponseHeadersPolicyConfig() {
        return this.to('GetResponseHeadersPolicyConfig');
    }
    /**
     * Grants permission to get a savings plan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toGetSavingsPlan() {
        return this.to('GetSavingsPlan');
    }
    /**
     * Grants permission to get the information about an RTMP distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistribution.html
     */
    toGetStreamingDistribution() {
        return this.to('GetStreamingDistribution');
    }
    /**
     * Grants permission to get the configuration information about a streaming distribution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetStreamingDistributionConfig.html
     */
    toGetStreamingDistributionConfig() {
        return this.to('GetStreamingDistributionConfig');
    }
    /**
     * Grants permission to get the information about a VPC origin
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_GetVpcOrigin.html
     */
    toGetVpcOrigin() {
        return this.to('GetVpcOrigin');
    }
    /**
     * Grants permission to list your Anycast static IP lists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListAnycastIpLists.html
     */
    toListAnycastIpLists() {
        return this.to('ListAnycastIpLists');
    }
    /**
     * Grants permission to list all cache policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCachePolicies.html
     */
    toListCachePolicies() {
        return this.to('ListCachePolicies');
    }
    /**
     * Grants permission to list your CloudFront origin access identities
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListCloudFrontOriginAccessIdentities.html
     */
    toListCloudFrontOriginAccessIdentities() {
        return this.to('ListCloudFrontOriginAccessIdentities');
    }
    /**
     * Grants permission to list all aliases that conflict with the given alias in CloudFront
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListConflictingAliases.html
     */
    toListConflictingAliases() {
        return this.to('ListConflictingAliases');
    }
    /**
     * Grants permission to list the connection groups in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListConnectionGroups.html
     */
    toListConnectionGroups() {
        return this.to('ListConnectionGroups');
    }
    /**
     * Grants permission to list all continuous-deployment policies in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListContinuousDeploymentPolicies.html
     */
    toListContinuousDeploymentPolicies() {
        return this.to('ListContinuousDeploymentPolicies');
    }
    /**
     * Grants permission to list the distribution tenants in your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionTenants.html
     */
    toListDistributionTenants() {
        return this.to('ListDistributionTenants');
    }
    /**
     * Grants permission to list the distribution tenants by the customization that you specify
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionTenantsByCustomization.html
     */
    toListDistributionTenantsByCustomization() {
        return this.to('ListDistributionTenantsByCustomization');
    }
    /**
     * Grants permission to list the distributions associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributions.html
     */
    toListDistributions() {
        return this.to('ListDistributions');
    }
    /**
     * Grants permission to list the distributions in your account that are associated with the specified AnycastIpListId
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByAnycastIpListId.html
     */
    toListDistributionsByAnycastIpListId() {
        return this.to('ListDistributionsByAnycastIpListId');
    }
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified cache policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByCachePolicyId.html
     */
    toListDistributionsByCachePolicyId() {
        return this.to('ListDistributionsByCachePolicyId');
    }
    /**
     * Grants permission to list the distributions by the specified connection mode
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByConnectionMode.html
     */
    toListDistributionsByConnectionMode() {
        return this.to('ListDistributionsByConnectionMode');
    }
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified key group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByKeyGroup.html
     */
    toListDistributionsByKeyGroup() {
        return this.to('ListDistributionsByKeyGroup');
    }
    /**
     * Grants permission to list the distributions associated a Lambda function
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListDistributionsByLambdaFunction() {
        return this.to('ListDistributionsByLambdaFunction');
    }
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified origin request policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByOriginRequestPolicyId.html
     */
    toListDistributionsByOriginRequestPolicyId() {
        return this.to('ListDistributionsByOriginRequestPolicyId');
    }
    /**
     * Grants permission to get a list of distributions that have a cache behavior that's associated with the specified real-time log configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByRealtimeLogConfig.html
     */
    toListDistributionsByRealtimeLogConfig() {
        return this.to('ListDistributionsByRealtimeLogConfig');
    }
    /**
     * Grants permission to list distribution IDs for distributions that have a cache behavior that's associated with the specified response headers policy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByResponseHeadersPolicyId.html
     */
    toListDistributionsByResponseHeadersPolicyId() {
        return this.to('ListDistributionsByResponseHeadersPolicyId');
    }
    /**
     * Grants permission to list IDs for distributions associated with the specified VPC origin
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByVpcOriginId.html
     */
    toListDistributionsByVpcOriginId() {
        return this.to('ListDistributionsByVpcOriginId');
    }
    /**
     * Grants permission to list the distributions associated with your AWS account with given AWS WAF web ACL
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDistributionsByWebACLId.html
     */
    toListDistributionsByWebACLId() {
        return this.to('ListDistributionsByWebACLId');
    }
    /**
     * Grants permission to list domain conflicts for a specified domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListDomainConflicts.html
     */
    toListDomainConflicts() {
        return this.to('ListDomainConflicts');
    }
    /**
     * Grants permission to list all field-level encryption configurations that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionConfigs.html
     */
    toListFieldLevelEncryptionConfigs() {
        return this.to('ListFieldLevelEncryptionConfigs');
    }
    /**
     * Grants permission to list all field-level encryption profiles that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFieldLevelEncryptionProfiles.html
     */
    toListFieldLevelEncryptionProfiles() {
        return this.to('ListFieldLevelEncryptionProfiles');
    }
    /**
     * Grants permission to get a list of CloudFront functions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions() {
        return this.to('ListFunctions');
    }
    /**
     * Grants permission to list your invalidation batches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidations.html
     */
    toListInvalidations() {
        return this.to('ListInvalidations');
    }
    /**
     * Grants permission to list the invalidations for a distribution tenant
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListInvalidationsForDistributionTenant.html
     */
    toListInvalidationsForDistributionTenant() {
        return this.to('ListInvalidationsForDistributionTenant');
    }
    /**
     * Grants permission to list all key groups that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListKeyGroups.html
     */
    toListKeyGroups() {
        return this.to('ListKeyGroups');
    }
    /**
     * Grants permission to get a list of CloudFront KeyValueStores
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListKeyValueStores.html
     */
    toListKeyValueStores() {
        return this.to('ListKeyValueStores');
    }
    /**
     * Grants permission to list all origin access controls in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginAccessControls.html
     */
    toListOriginAccessControls() {
        return this.to('ListOriginAccessControls');
    }
    /**
     * Grants permission to list all origin request policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListOriginRequestPolicies.html
     */
    toListOriginRequestPolicies() {
        return this.to('ListOriginRequestPolicies');
    }
    /**
     * Grants permission to list all public keys that have been added to CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListPublicKeys.html
     */
    toListPublicKeys() {
        return this.to('ListPublicKeys');
    }
    /**
     * Grants permission to list CloudFront rate cards for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListRateCards() {
        return this.to('ListRateCards');
    }
    /**
     * Grants permission to get a list of real-time log configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListRealtimeLogConfigs.html
     */
    toListRealtimeLogConfigs() {
        return this.to('ListRealtimeLogConfigs');
    }
    /**
     * Grants permission to list all response headers policies that have been created in CloudFront for this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListResponseHeadersPolicies.html
     */
    toListResponseHeadersPolicies() {
        return this.to('ListResponseHeadersPolicies');
    }
    /**
     * Grants permission to list savings plans in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListSavingsPlans() {
        return this.to('ListSavingsPlans');
    }
    /**
     * Grants permission to list your RTMP distributions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListStreamingDistributions.html
     */
    toListStreamingDistributions() {
        return this.to('ListStreamingDistributions');
    }
    /**
     * Grants permission to list tags for a CloudFront resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list CloudFront usage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toListUsages() {
        return this.to('ListUsages');
    }
    /**
     * Grants permission to list VPC origins
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ListVpcOrigins.html
     */
    toListVpcOrigins() {
        return this.to('ListVpcOrigins');
    }
    /**
     * Grants permission to publish a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_PublishFunction.html
     */
    toPublishFunction() {
        return this.to('PublishFunction');
    }
    /**
     * Grants permission to add tags to a CloudFront resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_TestFunction.html
     */
    toTestFunction() {
        return this.to('TestFunction');
    }
    /**
     * Grants permission to remove tags from a CloudFront resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a cache policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCachePolicy.html
     */
    toUpdateCachePolicy() {
        return this.to('UpdateCachePolicy');
    }
    /**
     * Grants permission to set the configuration for a CloudFront origin access identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateCloudFrontOriginAccessIdentity.html
     */
    toUpdateCloudFrontOriginAccessIdentity() {
        return this.to('UpdateCloudFrontOriginAccessIdentity');
    }
    /**
     * Grants permission to update a connection group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateConnectionGroup.html
     */
    toUpdateConnectionGroup() {
        return this.to('UpdateConnectionGroup');
    }
    /**
     * Grants permission to update a continuous-deployment policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateContinuousDeploymentPolicy.html
     */
    toUpdateContinuousDeploymentPolicy() {
        return this.to('UpdateContinuousDeploymentPolicy');
    }
    /**
     * Grants permission to update the configuration for a web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistribution.html
     */
    toUpdateDistribution() {
        return this.to('UpdateDistribution');
    }
    /**
     * Grants permission to update a distribution tenant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistributionTenant.html
     */
    toUpdateDistributionTenant() {
        return this.to('UpdateDistributionTenant');
    }
    /**
     * Grants permission to copy the configuration from a staging web distribution to its corresponding primary web distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDistributionWithStagingConfig.html
     */
    toUpdateDistributionWithStagingConfig() {
        return this.to('UpdateDistributionWithStagingConfig');
    }
    /**
     * Grants permission to update a domain association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateDomainAssociation.html
     */
    toUpdateDomainAssociation() {
        return this.to('UpdateDomainAssociation');
    }
    /**
     * Grants permission to update a field-level encryption configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionConfig.html
     */
    toUpdateFieldLevelEncryptionConfig() {
        return this.to('UpdateFieldLevelEncryptionConfig');
    }
    /**
     * Grants permission to update a field-level encryption profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFieldLevelEncryptionProfile.html
     */
    toUpdateFieldLevelEncryptionProfile() {
        return this.to('UpdateFieldLevelEncryptionProfile');
    }
    /**
     * Grants permission to update a CloudFront function
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction() {
        return this.to('UpdateFunction');
    }
    /**
     * Grants permission to update a key group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateKeyGroup.html
     */
    toUpdateKeyGroup() {
        return this.to('UpdateKeyGroup');
    }
    /**
     * Grants permission to update a CloudFront KeyValueStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateKeyValueStore.html
     */
    toUpdateKeyValueStore() {
        return this.to('UpdateKeyValueStore');
    }
    /**
     * Grants permission to update an origin access control
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginAccessControl.html
     */
    toUpdateOriginAccessControl() {
        return this.to('UpdateOriginAccessControl');
    }
    /**
     * Grants permission to update an origin request policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateOriginRequestPolicy.html
     */
    toUpdateOriginRequestPolicy() {
        return this.to('UpdateOriginRequestPolicy');
    }
    /**
     * Grants permission to update public key information
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdatePublicKey.html
     */
    toUpdatePublicKey() {
        return this.to('UpdatePublicKey');
    }
    /**
     * Grants permission to update a real-time log configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateRealtimeLogConfig.html
     */
    toUpdateRealtimeLogConfig() {
        return this.to('UpdateRealtimeLogConfig');
    }
    /**
     * Grants permission to update a response headers policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateResponseHeadersPolicy.html
     */
    toUpdateResponseHeadersPolicy() {
        return this.to('UpdateResponseHeadersPolicy');
    }
    /**
     * Grants permission to update a savings plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cf-api-permissions-ref.html
     */
    toUpdateSavingsPlan() {
        return this.to('UpdateSavingsPlan');
    }
    /**
     * Grants permission to update the configuration for an RTMP distribution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateStreamingDistribution.html
     */
    toUpdateStreamingDistribution() {
        return this.to('UpdateStreamingDistribution');
    }
    /**
     * Grants permission to update a VPC origin
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_UpdateVpcOrigin.html
     */
    toUpdateVpcOrigin() {
        return this.to('UpdateVpcOrigin');
    }
    /**
     * Grants permission to verify the DNS configuration for a specified domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_VerifyDnsConfiguration.html
     */
    toVerifyDnsConfiguration() {
        return this.to('VerifyDnsConfiguration');
    }
    /**
     * Adds a resource of type distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistribution(distributionId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:distribution/${distributionId}`);
    }
    /**
     * Adds a resource of type streaming-distribution to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-working-with.html
     *
     * @param distributionId - Identifier for the distributionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStreamingDistribution(distributionId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:streaming-distribution/${distributionId}`);
    }
    /**
     * Adds a resource of type origin-access-identity to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html#private-content-restricting-access-to-s3-overview
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginAccessIdentity(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:origin-access-identity/${id}`);
    }
    /**
     * Adds a resource of type field-level-encryption-config to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFieldLevelEncryptionConfig(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:field-level-encryption-config/${id}`);
    }
    /**
     * Adds a resource of type field-level-encryption-profile to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/field-level-encryption.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFieldLevelEncryptionProfile(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:field-level-encryption-profile/${id}`);
    }
    /**
     * Adds a resource of type cache-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCachePolicy(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:cache-policy/${id}`);
    }
    /**
     * Adds a resource of type origin-request-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginRequestPolicy(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:origin-request-policy/${id}`);
    }
    /**
     * Adds a resource of type realtime-log-config to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/real-time-logs.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRealtimeLogConfig(name, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:realtime-log-config/${name}`);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/cloudfront-functions.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFunction(name, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:function/${name}`);
    }
    /**
     * Adds a resource of type key-value-store to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/kvs-with-functions.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKeyValueStore(name, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:key-value-store/${name}`);
    }
    /**
     * Adds a resource of type response-headers-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onResponseHeadersPolicy(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:response-headers-policy/${id}`);
    }
    /**
     * Adds a resource of type origin-access-control to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOriginAccessControl(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:origin-access-control/${id}`);
    }
    /**
     * Adds a resource of type continuous-deployment-policy to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/working-with-policies.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onContinuousDeploymentPolicy(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:continuous-deployment-policy/${id}`);
    }
    /**
     * Adds a resource of type anycast-ip-list to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/request-static-ips.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnycastIpList(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:anycast-ip-list/${id}`);
    }
    /**
     * Adds a resource of type vpcorigin to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-vpc-origins.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVpcorigin(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:vpcorigin/${id}`);
    }
    /**
     * Adds a resource of type distribution-tenant to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/API_DistributionTenant.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDistributionTenant(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:distribution-tenant/${id}`);
    }
    /**
     * Adds a resource of type connection-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/API_ConnectionGroup.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnectionGroup(id, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cloudfront::${account ?? this.defaultAccount}:connection-group/${id}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnycastIpList()
     * - .toCreateConnectionGroup()
     * - .toCreateDistributionTenant()
     * - .toCreateStreamingDistributionWithTags()
     * - .toCreateVpcOrigin()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - distribution
     * - streaming-distribution
     * - anycast-ip-list
     * - vpcorigin
     * - distribution-tenant
     * - connection-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnycastIpList()
     * - .toCreateConnectionGroup()
     * - .toCreateDistributionTenant()
     * - .toCreateStreamingDistributionWithTags()
     * - .toCreateVpcOrigin()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cloudfront](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudfront.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cloudfront';
        this.accessLevelList = {
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Write: [
                'AssociateAlias',
                'AssociateDistributionTenantWebACL',
                'AssociateDistributionWebACL',
                'CopyDistribution',
                'CreateAnycastIpList',
                'CreateCachePolicy',
                'CreateCloudFrontOriginAccessIdentity',
                'CreateConnectionGroup',
                'CreateContinuousDeploymentPolicy',
                'CreateDistribution',
                'CreateDistributionTenant',
                'CreateFieldLevelEncryptionConfig',
                'CreateFieldLevelEncryptionProfile',
                'CreateFunction',
                'CreateInvalidation',
                'CreateInvalidationForDistributionTenant',
                'CreateKeyGroup',
                'CreateKeyValueStore',
                'CreateMonitoringSubscription',
                'CreateOriginAccessControl',
                'CreateOriginRequestPolicy',
                'CreatePublicKey',
                'CreateRealtimeLogConfig',
                'CreateResponseHeadersPolicy',
                'CreateSavingsPlan',
                'CreateStreamingDistribution',
                'CreateStreamingDistributionWithTags',
                'CreateVpcOrigin',
                'DeleteAnycastIpList',
                'DeleteCachePolicy',
                'DeleteCloudFrontOriginAccessIdentity',
                'DeleteConnectionGroup',
                'DeleteContinuousDeploymentPolicy',
                'DeleteDistribution',
                'DeleteDistributionTenant',
                'DeleteFieldLevelEncryptionConfig',
                'DeleteFieldLevelEncryptionProfile',
                'DeleteFunction',
                'DeleteKeyGroup',
                'DeleteKeyValueStore',
                'DeleteMonitoringSubscription',
                'DeleteOriginAccessControl',
                'DeleteOriginRequestPolicy',
                'DeletePublicKey',
                'DeleteRealtimeLogConfig',
                'DeleteResponseHeadersPolicy',
                'DeleteStreamingDistribution',
                'DeleteVpcOrigin',
                'DisassociateDistributionTenantWebACL',
                'DisassociateDistributionWebACL',
                'PublishFunction',
                'TestFunction',
                'UpdateCachePolicy',
                'UpdateCloudFrontOriginAccessIdentity',
                'UpdateConnectionGroup',
                'UpdateContinuousDeploymentPolicy',
                'UpdateDistribution',
                'UpdateDistributionTenant',
                'UpdateDistributionWithStagingConfig',
                'UpdateDomainAssociation',
                'UpdateFieldLevelEncryptionConfig',
                'UpdateFieldLevelEncryptionProfile',
                'UpdateFunction',
                'UpdateKeyGroup',
                'UpdateKeyValueStore',
                'UpdateOriginAccessControl',
                'UpdateOriginRequestPolicy',
                'UpdatePublicKey',
                'UpdateRealtimeLogConfig',
                'UpdateResponseHeadersPolicy',
                'UpdateSavingsPlan',
                'UpdateStreamingDistribution',
                'UpdateVpcOrigin'
            ],
            Read: [
                'DescribeFunction',
                'DescribeKeyValueStore',
                'GetAnycastIpList',
                'GetCachePolicy',
                'GetCachePolicyConfig',
                'GetCloudFrontOriginAccessIdentity',
                'GetCloudFrontOriginAccessIdentityConfig',
                'GetConnectionGroup',
                'GetConnectionGroupByRoutingEndpoint',
                'GetContinuousDeploymentPolicy',
                'GetContinuousDeploymentPolicyConfig',
                'GetDistribution',
                'GetDistributionConfig',
                'GetDistributionTenant',
                'GetDistributionTenantByDomain',
                'GetFieldLevelEncryption',
                'GetFieldLevelEncryptionConfig',
                'GetFieldLevelEncryptionProfile',
                'GetFieldLevelEncryptionProfileConfig',
                'GetFunction',
                'GetInvalidation',
                'GetInvalidationForDistributionTenant',
                'GetKeyGroup',
                'GetKeyGroupConfig',
                'GetManagedCertificateDetails',
                'GetMonitoringSubscription',
                'GetOriginAccessControl',
                'GetOriginAccessControlConfig',
                'GetOriginRequestPolicy',
                'GetOriginRequestPolicyConfig',
                'GetPublicKey',
                'GetPublicKeyConfig',
                'GetRealtimeLogConfig',
                'GetResponseHeadersPolicy',
                'GetResponseHeadersPolicyConfig',
                'GetSavingsPlan',
                'GetStreamingDistribution',
                'GetStreamingDistributionConfig',
                'GetVpcOrigin',
                'ListTagsForResource',
                'VerifyDnsConfiguration'
            ],
            List: [
                'ListAnycastIpLists',
                'ListCachePolicies',
                'ListCloudFrontOriginAccessIdentities',
                'ListConflictingAliases',
                'ListConnectionGroups',
                'ListContinuousDeploymentPolicies',
                'ListDistributionTenants',
                'ListDistributionTenantsByCustomization',
                'ListDistributions',
                'ListDistributionsByAnycastIpListId',
                'ListDistributionsByCachePolicyId',
                'ListDistributionsByConnectionMode',
                'ListDistributionsByKeyGroup',
                'ListDistributionsByLambdaFunction',
                'ListDistributionsByOriginRequestPolicyId',
                'ListDistributionsByRealtimeLogConfig',
                'ListDistributionsByResponseHeadersPolicyId',
                'ListDistributionsByVpcOriginId',
                'ListDistributionsByWebACLId',
                'ListDomainConflicts',
                'ListFieldLevelEncryptionConfigs',
                'ListFieldLevelEncryptionProfiles',
                'ListFunctions',
                'ListInvalidations',
                'ListInvalidationsForDistributionTenant',
                'ListKeyGroups',
                'ListKeyValueStores',
                'ListOriginAccessControls',
                'ListOriginRequestPolicies',
                'ListPublicKeys',
                'ListRateCards',
                'ListRealtimeLogConfigs',
                'ListResponseHeadersPolicies',
                'ListSavingsPlans',
                'ListStreamingDistributions',
                'ListUsages',
                'ListVpcOrigins'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cloudfront = Cloudfront;
//# sourceMappingURL=data:application/json;base64,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