"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logs = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [logs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchlogs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Logs extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the specified AWS Key Management Service (AWS KMS) customer master key (CMK) with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AssociateKmsKey.html
     */
    toAssociateKmsKey() {
        return this.to('AssociateKmsKey');
    }
    /**
     * Grants permission to cancel an export task if it is in PENDING or RUNNING state
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CancelExportTask.html
     */
    toCancelExportTask() {
        return this.to('CancelExportTask');
    }
    /**
     * Grants permission to create a delivery connecting a delivery source to a delivery destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateDelivery.html
     */
    toCreateDelivery() {
        return this.to('CreateDelivery');
    }
    /**
     * Grants permission to create an ExportTask which allows you to efficiently export data from a Log Group to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateExportTask.html
     */
    toCreateExportTask() {
        return this.to('CreateExportTask');
    }
    /**
     * Grants permission to create a log anomaly detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogAnomalyDetector.html
     */
    toCreateLogAnomalyDetector() {
        return this.to('CreateLogAnomalyDetector');
    }
    /**
     * Grants permission to create the log delivery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html
     */
    toCreateLogDelivery() {
        return this.to('CreateLogDelivery');
    }
    /**
     * Grants permission to create a new log group with the specified name
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogGroup.html
     */
    toCreateLogGroup() {
        return this.to('CreateLogGroup');
    }
    /**
     * Grants permission to create a new log stream with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_CreateLogStream.html
     */
    toCreateLogStream() {
        return this.to('CreateLogStream');
    }
    /**
     * Grants permission to delete an account policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteAccountPolicy.html
     */
    toDeleteAccountPolicy() {
        return this.to('DeleteAccountPolicy');
    }
    /**
     * Grants permission to delete a data protection policy attached to a log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDataProtectionPolicy.html
     */
    toDeleteDataProtectionPolicy() {
        return this.to('DeleteDataProtectionPolicy');
    }
    /**
     * Grants permission to delete a delivery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDelivery.html
     */
    toDeleteDelivery() {
        return this.to('DeleteDelivery');
    }
    /**
     * Grants permission to delete a delivery destination after all associated deliveries are deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDeliveryDestination.html
     */
    toDeleteDeliveryDestination() {
        return this.to('DeleteDeliveryDestination');
    }
    /**
     * Grants permission to delete a delivery destination policy associated with a delivery destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDeliveryDestinationPolicy.html
     */
    toDeleteDeliveryDestinationPolicy() {
        return this.to('DeleteDeliveryDestinationPolicy');
    }
    /**
     * Grants permission to delete a delivery source after all associated deliveries are deleted
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDeliverySource.html
     */
    toDeleteDeliverySource() {
        return this.to('DeleteDeliverySource');
    }
    /**
     * Grants permission to delete the destination with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteDestination.html
     */
    toDeleteDestination() {
        return this.to('DeleteDestination');
    }
    /**
     * Grants permission to delete an index policy attached to a log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteIndexPolicy.html
     */
    toDeleteIndexPolicy() {
        return this.to('DeleteIndexPolicy');
    }
    /**
     * Grants permission to delete the integration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete a log anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogAnomalyDetector.html
     */
    toDeleteLogAnomalyDetector() {
        return this.to('DeleteLogAnomalyDetector');
    }
    /**
     * Grants permission to delete the log delivery information for specified log delivery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html
     */
    toDeleteLogDelivery() {
        return this.to('DeleteLogDelivery');
    }
    /**
     * Grants permission to delete the log group with the specified name
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogGroup.html
     */
    toDeleteLogGroup() {
        return this.to('DeleteLogGroup');
    }
    /**
     * Grants permission to delete a log stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteLogStream.html
     */
    toDeleteLogStream() {
        return this.to('DeleteLogStream');
    }
    /**
     * Grants permission to delete a metric filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteMetricFilter.html
     */
    toDeleteMetricFilter() {
        return this.to('DeleteMetricFilter');
    }
    /**
     * Grants permission to delete a saved CloudWatch Logs Insights query definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteQueryDefinition.html
     */
    toDeleteQueryDefinition() {
        return this.to('DeleteQueryDefinition');
    }
    /**
     * Grants permission to delete a resource policy from this account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete the retention policy of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteRetentionPolicy.html
     */
    toDeleteRetentionPolicy() {
        return this.to('DeleteRetentionPolicy');
    }
    /**
     * Grants permission to delete a subscription filter associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteSubscriptionFilter.html
     */
    toDeleteSubscriptionFilter() {
        return this.to('DeleteSubscriptionFilter');
    }
    /**
     * Grants permission to delete a transformer associated with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeleteTransformer.html
     */
    toDeleteTransformer() {
        return this.to('DeleteTransformer');
    }
    /**
     * Grants permission to retrieve account policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeAccountPolicies.html
     */
    toDescribeAccountPolicies() {
        return this.to('DescribeAccountPolicies');
    }
    /**
     * Grants permission to retrieve a list of configuration templates of available log types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeConfigurationTemplates.html
     */
    toDescribeConfigurationTemplates() {
        return this.to('DescribeConfigurationTemplates');
    }
    /**
     * Grants permission to retrieve a list of deliveries an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDeliveries.html
     */
    toDescribeDeliveries() {
        return this.to('DescribeDeliveries');
    }
    /**
     * Grants permission to retrieve a list of delivery destinations an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDeliveryDestinations.html
     */
    toDescribeDeliveryDestinations() {
        return this.to('DescribeDeliveryDestinations');
    }
    /**
     * Grants permission to retrieve a list of delivery sources in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDeliverySources.html
     */
    toDescribeDeliverySources() {
        return this.to('DescribeDeliverySources');
    }
    /**
     * Grants permission to return all the destinations that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeDestinations.html
     */
    toDescribeDestinations() {
        return this.to('DescribeDestinations');
    }
    /**
     * Grants permission to return all the export tasks that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeExportTasks.html
     */
    toDescribeExportTasks() {
        return this.to('DescribeExportTasks');
    }
    /**
     * Grants permission to return all the indexing attributes that are attached with the log groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeFieldIndexes.html
     */
    toDescribeFieldIndexes() {
        return this.to('DescribeFieldIndexes');
    }
    /**
     * Grants permission to return all the index policies that are attached with the log groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeIndexPolicies.html
     */
    toDescribeIndexPolicies() {
        return this.to('DescribeIndexPolicies');
    }
    /**
     * Grants permission to return all the log groups that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogGroups.html
     */
    toDescribeLogGroups() {
        return this.to('DescribeLogGroups');
    }
    /**
     * Grants permission to return all the log streams that are associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeLogStreams.html
     */
    toDescribeLogStreams() {
        return this.to('DescribeLogStreams');
    }
    /**
     * Grants permission to return all the metrics filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeMetricFilters.html
     */
    toDescribeMetricFilters() {
        return this.to('DescribeMetricFilters');
    }
    /**
     * Grants permission to return a list of CloudWatch Logs Insights queries that are scheduled, executing, or have been executed recently in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueries.html
     */
    toDescribeQueries() {
        return this.to('DescribeQueries');
    }
    /**
     * Grants permission to return a paginated list of your saved CloudWatch Logs Insights query definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeQueryDefinitions.html
     */
    toDescribeQueryDefinitions() {
        return this.to('DescribeQueryDefinitions');
    }
    /**
     * Grants permission to return all the resource policies in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeResourcePolicies.html
     */
    toDescribeResourcePolicies() {
        return this.to('DescribeResourcePolicies');
    }
    /**
     * Grants permission to return all the subscription filters associated with the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DescribeSubscriptionFilters.html
     */
    toDescribeSubscriptionFilters() {
        return this.to('DescribeSubscriptionFilters');
    }
    /**
     * Grants permission to disassociate the associated AWS Key Management Service (AWS KMS) customer master key (CMK) from the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DisassociateKmsKey.html
     */
    toDisassociateKmsKey() {
        return this.to('DisassociateKmsKey');
    }
    /**
     * Grants permission to retrieve log events, optionally filtered by a filter pattern from the specified log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_FilterLogEvents.html
     */
    toFilterLogEvents() {
        return this.to('FilterLogEvents');
    }
    /**
     * Grants permission to retrieve a data protection policy attached to a log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetDataProtectionPolicy.html
     */
    toGetDataProtectionPolicy() {
        return this.to('GetDataProtectionPolicy');
    }
    /**
     * Grants permission to retrieve a single delivery
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetDelivery.html
     */
    toGetDelivery() {
        return this.to('GetDelivery');
    }
    /**
     * Grants permission to retrieve a single delivery destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetDeliveryDestination.html
     */
    toGetDeliveryDestination() {
        return this.to('GetDeliveryDestination');
    }
    /**
     * Grants permission to retrieve a delivery destination policy attached to a delivery destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetDeliveryDestinationPolicy.html
     */
    toGetDeliveryDestinationPolicy() {
        return this.to('GetDeliveryDestinationPolicy');
    }
    /**
     * Grants permission to retrieve a single delivery source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetDeliverySource.html
     */
    toGetDeliverySource() {
        return this.to('GetDeliverySource');
    }
    /**
     * Grants permission to retrieve a single integration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetIntegration.html
     */
    toGetIntegration() {
        return this.to('GetIntegration');
    }
    /**
     * Grants permission to get a log anomaly detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogAnomalyDetector.html
     */
    toGetLogAnomalyDetector() {
        return this.to('GetLogAnomalyDetector');
    }
    /**
     * Grants permission to get the log delivery information for specified log delivery
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html
     */
    toGetLogDelivery() {
        return this.to('GetLogDelivery');
    }
    /**
     * Grants permission to retrieve log events from the specified log stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogEvents.html
     */
    toGetLogEvents() {
        return this.to('GetLogEvents');
    }
    /**
     * Grants permission to return a list of the fields that are included in log events in the specified log group, along with the percentage of log events that contain each field
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogGroupFields.html
     */
    toGetLogGroupFields() {
        return this.to('GetLogGroupFields');
    }
    /**
     * Grants permission to retrieve all the fields and values of a single log event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetLogRecord.html
     */
    toGetLogRecord() {
        return this.to('GetLogRecord');
    }
    /**
     * Grants permission to return the results from the specified query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetQueryResults.html
     */
    toGetQueryResults() {
        return this.to('GetQueryResults');
    }
    /**
     * Grants permission to return transformer associated with the specified log group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_GetTransformer.html
     */
    toGetTransformer() {
        return this.to('GetTransformer');
    }
    /**
     * Grants permission to share CloudWatch resources with a monitoring account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account-Setup.html#CloudWatch-Unified-Cross-Account-Setup-permissions
     */
    toLink() {
        return this.to('Link');
    }
    /**
     * Grants permission to list all anomalies detected in the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListAnomalies.html
     */
    toListAnomalies() {
        return this.to('ListAnomalies');
    }
    /**
     * Grants permission to retrieve all the entities that are associated with log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/permissions-reference-cwl.html
     */
    toListEntitiesForLogGroup() {
        return this.to('ListEntitiesForLogGroup');
    }
    /**
     * Grants permission to list all integrations associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListIntegrations.html
     */
    toListIntegrations() {
        return this.to('ListIntegrations');
    }
    /**
     * Grants permission to return all the anomaly detectors that are associated with the AWS account making the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListLogAnomalyDetectors.html
     */
    toListLogAnomalyDetectors() {
        return this.to('ListLogAnomalyDetectors');
    }
    /**
     * Grants permission to list all the log deliveries for specified account and/or log source
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html
     */
    toListLogDeliveries() {
        return this.to('ListLogDeliveries');
    }
    /**
     * Grants permission to retrieve all the log groups that are associated with entity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/permissions-reference-cwl.html
     */
    toListLogGroupsForEntity() {
        return this.to('ListLogGroupsForEntity');
    }
    /**
     * Grants permission to return all the log groups that are associated with the specified query
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListLogGroupsForQuery.html
     */
    toListLogGroupsForQuery() {
        return this.to('ListLogGroupsForQuery');
    }
    /**
     * Grants permission to list the tags for the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the tags for the specified log group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_ListTagsLogGroup.html
     */
    toListTagsLogGroup() {
        return this.to('ListTagsLogGroup');
    }
    /**
     * Grants permission to attach an account policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutAccountPolicy.html
     */
    toPutAccountPolicy() {
        return this.to('PutAccountPolicy');
    }
    /**
     * Grants permission to attach a data protection policy to detect and redact sensitive information from log events
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDataProtectionPolicy.html
     */
    toPutDataProtectionPolicy() {
        return this.to('PutDataProtectionPolicy');
    }
    /**
     * Grants permission to create/update a delivery destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifDeliveryDestinationResourceArn()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliveryDestination.html
     */
    toPutDeliveryDestination() {
        return this.to('PutDeliveryDestination');
    }
    /**
     * Grants permission to attach a delivery destination policy to a delivery destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliveryDestinationPolicy.html
     */
    toPutDeliveryDestinationPolicy() {
        return this.to('PutDeliveryDestinationPolicy');
    }
    /**
     * Grants permission to create/update a delivery source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifLogGeneratingResourceArns()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDeliverySource.html
     */
    toPutDeliverySource() {
        return this.to('PutDeliverySource');
    }
    /**
     * Grants permission to create or update a Destination
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestination.html
     */
    toPutDestination() {
        return this.to('PutDestination');
    }
    /**
     * Grants permission to create or update an access policy associated with an existing Destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutDestinationPolicy.html
     */
    toPutDestinationPolicy() {
        return this.to('PutDestinationPolicy');
    }
    /**
     * Grants permission to attach an index policy at log group level to optimize search and query
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutIndexPolicy.html
     */
    toPutIndexPolicy() {
        return this.to('PutIndexPolicy');
    }
    /**
     * Grants permission to create integration between cloudwatch logs and opensearch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutIntegration.html
     */
    toPutIntegration() {
        return this.to('PutIntegration');
    }
    /**
     * Grants permission to upload a batch of log events to the specified log stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutLogEvents.html
     */
    toPutLogEvents() {
        return this.to('PutLogEvents');
    }
    /**
     * Grants permission to create or update a metric filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutMetricFilter.html
     */
    toPutMetricFilter() {
        return this.to('PutMetricFilter');
    }
    /**
     * Grants permission to create or update a query definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutQueryDefinition.html
     */
    toPutQueryDefinition() {
        return this.to('PutQueryDefinition');
    }
    /**
     * Grants permission to create or update a resource policy allowing other AWS services to put log events to this account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to set the retention of the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutRetentionPolicy.html
     */
    toPutRetentionPolicy() {
        return this.to('PutRetentionPolicy');
    }
    /**
     * Grants permission to create or update a subscription filter and associates it with the specified log group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutSubscriptionFilter.html
     */
    toPutSubscriptionFilter() {
        return this.to('PutSubscriptionFilter');
    }
    /**
     * Grants permission to create or update a transformer and associates it with the specified log group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_PutTransformer.html
     */
    toPutTransformer() {
        return this.to('PutTransformer');
    }
    /**
     * Grants permission to start a Live Tail session in CloudWatch Logs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartLiveTail.html
     */
    toStartLiveTail() {
        return this.to('StartLiveTail');
    }
    /**
     * Grants permission to schedule a query of a log group using CloudWatch Logs Insights
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StartQuery.html
     */
    toStartQuery() {
        return this.to('StartQuery');
    }
    /**
     * Grants permission to stop a Live Tail session that is in progress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/CloudWatchLogs_LiveTail.html
     */
    toStopLiveTail() {
        return this.to('StopLiveTail');
    }
    /**
     * Grants permission to stop a CloudWatch Logs Insights query that is in progress
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_StopQuery.html
     */
    toStopQuery() {
        return this.to('StopQuery');
    }
    /**
     * Grants permission to add or update the specified tags for the specified log group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagLogGroup.html
     */
    toTagLogGroup() {
        return this.to('TagLogGroup');
    }
    /**
     * Grants permission to add or update the specified tags for the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test the filter pattern of a metric filter against a sample of log event messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestMetricFilter.html
     */
    toTestMetricFilter() {
        return this.to('TestMetricFilter');
    }
    /**
     * Grants permission to test the transformer against a sample of log event messages
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_TestTransformer.html
     */
    toTestTransformer() {
        return this.to('TestTransformer');
    }
    /**
     * Grants permission to fetch unmasked log events that have been redacted with a data protection policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/mask-sensitive-log-data.html
     */
    toUnmask() {
        return this.to('Unmask');
    }
    /**
     * Grants permission to remove the specified tags from the specified log group
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagLogGroup.html
     */
    toUntagLogGroup() {
        return this.to('UntagLogGroup');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an anomaly reported by a log anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateAnomaly.html
     */
    toUpdateAnomaly() {
        return this.to('UpdateAnomaly');
    }
    /**
     * Grants permission to update configuration related to a delivery
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateDeliveryConfiguration.html
     */
    toUpdateDeliveryConfiguration() {
        return this.to('UpdateDeliveryConfiguration');
    }
    /**
     * Grants permission to update a log anomaly detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_UpdateLogAnomalyDetector.html
     */
    toUpdateLogAnomalyDetector() {
        return this.to('UpdateLogAnomalyDetector');
    }
    /**
     * Grants permission to update the log delivery information for specified log delivery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/AWS-logs-and-resource-policy.html
     */
    toUpdateLogDelivery() {
        return this.to('UpdateLogDelivery');
    }
    /**
     * Adds a resource of type log-group to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_LogGroup.html
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLogGroup(logGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:log-group:${logGroupName}`);
    }
    /**
     * Adds a resource of type log-stream to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_LogStream.html
     *
     * @param logGroupName - Identifier for the logGroupName.
     * @param logStreamName - Identifier for the logStreamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLogStream(logGroupName, logStreamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:log-group:${logGroupName}:log-stream:${logStreamName}`);
    }
    /**
     * Adds a resource of type destination to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_Destination.html
     *
     * @param destinationName - Identifier for the destinationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDestination(destinationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:destination:${destinationName}`);
    }
    /**
     * Adds a resource of type delivery-source to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeliverySource.html
     *
     * @param deliverySourceName - Identifier for the deliverySourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverySource(deliverySourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:delivery-source:${deliverySourceName}`);
    }
    /**
     * Adds a resource of type delivery to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_Delivery.html
     *
     * @param deliveryName - Identifier for the deliveryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDelivery(deliveryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:delivery:${deliveryName}`);
    }
    /**
     * Adds a resource of type delivery-destination to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_DeliveryDestination.html
     *
     * @param deliveryDestinationName - Identifier for the deliveryDestinationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliveryDestination(deliveryDestinationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:delivery-destination:${deliveryDestinationName}`);
    }
    /**
     * Adds a resource of type anomaly-detector to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatchLogs/latest/APIReference/API_AnomalyDetector.html
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnomalyDetector(detectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:logs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:anomaly-detector:${detectorId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDelivery()
     * - .toCreateLogAnomalyDetector()
     * - .toCreateLogGroup()
     * - .toPutDeliveryDestination()
     * - .toPutDeliverySource()
     * - .toPutDestination()
     * - .toTagLogGroup()
     * - .toTagResource()
     * - .toUpdateDeliveryConfiguration()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - log-group
     * - log-stream
     * - destination
     * - delivery-source
     * - delivery
     * - delivery-destination
     * - anomaly-detector
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDelivery()
     * - .toCreateLogAnomalyDetector()
     * - .toCreateLogGroup()
     * - .toPutDeliveryDestination()
     * - .toPutDeliverySource()
     * - .toPutDestination()
     * - .toTagLogGroup()
     * - .toTagResource()
     * - .toUntagLogGroup()
     * - .toUntagResource()
     * - .toUpdateDeliveryConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Log Destination ARN passed in the request
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/iam-identity-based-access-control-cwl.html
     *
     * Applies to actions:
     * - .toPutDeliveryDestination()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDeliveryDestinationResourceArn(value, operator) {
        return this.if(`DeliveryDestinationResourceArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the Log Generating Resource ARNs passed in the request
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/logs/iam-identity-based-access-control-cwl.html
     *
     * Applies to actions:
     * - .toPutDeliverySource()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifLogGeneratingResourceArns(value, operator) {
        return this.if(`LogGeneratingResourceArns`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [logs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchlogs.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'logs';
        this.accessLevelList = {
            Write: [
                'AssociateKmsKey',
                'CancelExportTask',
                'CreateDelivery',
                'CreateExportTask',
                'CreateLogAnomalyDetector',
                'CreateLogDelivery',
                'CreateLogGroup',
                'CreateLogStream',
                'DeleteAccountPolicy',
                'DeleteDataProtectionPolicy',
                'DeleteDelivery',
                'DeleteDeliveryDestination',
                'DeleteDeliveryDestinationPolicy',
                'DeleteDeliverySource',
                'DeleteDestination',
                'DeleteIndexPolicy',
                'DeleteIntegration',
                'DeleteLogAnomalyDetector',
                'DeleteLogDelivery',
                'DeleteLogGroup',
                'DeleteLogStream',
                'DeleteMetricFilter',
                'DeleteQueryDefinition',
                'DeleteRetentionPolicy',
                'DeleteSubscriptionFilter',
                'DeleteTransformer',
                'DisassociateKmsKey',
                'Link',
                'PutAccountPolicy',
                'PutDataProtectionPolicy',
                'PutDeliveryDestination',
                'PutDeliveryDestinationPolicy',
                'PutDeliverySource',
                'PutDestination',
                'PutDestinationPolicy',
                'PutIndexPolicy',
                'PutIntegration',
                'PutLogEvents',
                'PutMetricFilter',
                'PutQueryDefinition',
                'PutRetentionPolicy',
                'PutSubscriptionFilter',
                'PutTransformer',
                'UpdateAnomaly',
                'UpdateDeliveryConfiguration',
                'UpdateLogAnomalyDetector',
                'UpdateLogDelivery'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            List: [
                'DescribeAccountPolicies',
                'DescribeConfigurationTemplates',
                'DescribeDeliveries',
                'DescribeDeliveryDestinations',
                'DescribeDeliverySources',
                'DescribeDestinations',
                'DescribeExportTasks',
                'DescribeFieldIndexes',
                'DescribeIndexPolicies',
                'DescribeLogGroups',
                'DescribeLogStreams',
                'DescribeMetricFilters',
                'DescribeQueries',
                'DescribeQueryDefinitions',
                'DescribeResourcePolicies',
                'DescribeSubscriptionFilters',
                'ListAnomalies',
                'ListEntitiesForLogGroup',
                'ListIntegrations',
                'ListLogAnomalyDetectors',
                'ListLogDeliveries',
                'ListLogGroupsForEntity',
                'ListLogGroupsForQuery',
                'ListTagsForResource',
                'ListTagsLogGroup'
            ],
            Read: [
                'FilterLogEvents',
                'GetDataProtectionPolicy',
                'GetDelivery',
                'GetDeliveryDestination',
                'GetDeliveryDestinationPolicy',
                'GetDeliverySource',
                'GetIntegration',
                'GetLogAnomalyDetector',
                'GetLogDelivery',
                'GetLogEvents',
                'GetLogGroupFields',
                'GetLogRecord',
                'GetQueryResults',
                'GetTransformer',
                'StartLiveTail',
                'StartQuery',
                'StopLiveTail',
                'StopQuery',
                'TestMetricFilter',
                'TestTransformer',
                'Unmask'
            ],
            Tagging: [
                'TagLogGroup',
                'TagResource',
                'UntagLogGroup',
                'UntagResource'
            ]
        };
    }
}
exports.Logs = Logs;
//# sourceMappingURL=data:application/json;base64,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