import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [oam](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchobservabilityaccessmanager.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Oam extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to create a link between a monitoring account and a source account for cross-account monitoring
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifResourceTypes()
     *
     * Dependent actions:
     * - oam:TagResource
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_CreateLink.html
     */
    toCreateLink(): this;
    /**
     * Grants permission to create a sink in an account so that it can be used as a monitoring account for cross-account monitoring
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - oam:TagResource
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_CreateSink.html
     */
    toCreateSink(): this;
    /**
     * Grants permission to delete a link between a monitoring account and a source account for cross-account monitoring
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_DeleteLink.html
     */
    toDeleteLink(): this;
    /**
     * Grants permission to delete a cross-account monitoring sink in a monitoring account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_DeleteSink.html
     */
    toDeleteSink(): this;
    /**
     * Grants permission to retrieve complete information about one cross-account monitoring link
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_GetLink.html
     */
    toGetLink(): this;
    /**
     * Grants permission to retrieve complete information about one cross-account monitoring sink
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_GetSink.html
     */
    toGetSink(): this;
    /**
     * Grants permission to retrieve information for the IAM policy for a cross-account monitoring sink
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_GetSinkPolicy.html
     */
    toGetSinkPolicy(): this;
    /**
     * Grants permission to retrieve a list of links that are linked for a cross-account monitoring sink
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListAttachedLinks.html
     */
    toListAttachedLinks(): this;
    /**
     * Grants permission to retrieve the ARNs of cross-account monitoring links in this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListLinks.html
     */
    toListLinks(): this;
    /**
     * Grants permission to retrieve the ARNs of cross-account monitoring sinks in this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListSinks.html
     */
    toListSinks(): this;
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to create or update the IAM policy for a cross-account monitoring sink
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_PutSinkPolicy.html
     */
    toPutSinkPolicy(): this;
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update an existing link between a monitoring account and a source account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTypes()
     *
     * https://docs.aws.amazon.com/OAM/latest/APIReference/API_UpdateLink.html
     */
    toUpdateLink(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type Link to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLink(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type Sink to the statement
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSink(resourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateLink()
     * - .toCreateSink()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteLink()
     * - .toDeleteSink()
     * - .toGetLink()
     * - .toGetSink()
     * - .toGetSinkPolicy()
     * - .toListAttachedLinks()
     * - .toPutSinkPolicy()
     * - .toUpdateLink()
     *
     * Applies to resource types:
     * - Link
     * - Sink
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateLink()
     * - .toCreateSink()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of resource types in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchobservabilityaccessmanager.html
     *
     * Applies to actions:
     * - .toCreateLink()
     * - .toUpdateLink()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTypes(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [oam](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncloudwatchobservabilityaccessmanager.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
