"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codebuild = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codebuild extends shared_1.PolicyStatement {
    /**
     * Grants permission to delete one or more builds
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchDeleteBuilds.html
     */
    toBatchDeleteBuilds() {
        return this.to('BatchDeleteBuilds');
    }
    /**
     * Grants permission to get information about one or more build batches
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuildBatches.html
     */
    toBatchGetBuildBatches() {
        return this.to('BatchGetBuildBatches');
    }
    /**
     * Grants permission to get information about one or more builds
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetBuilds.html
     */
    toBatchGetBuilds() {
        return this.to('BatchGetBuilds');
    }
    /**
     * Grants permission to get information about one or more command executions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetCommandExecutions.html
     */
    toBatchGetCommandExecutions() {
        return this.to('BatchGetCommandExecutions');
    }
    /**
     * Grants permission to return an array of the Fleet objects specified by the input parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetFleets.html
     */
    toBatchGetFleets() {
        return this.to('BatchGetFleets');
    }
    /**
     * Grants permission to get information about one or more build projects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetProjects.html
     */
    toBatchGetProjects() {
        return this.to('BatchGetProjects');
    }
    /**
     * Grants permission to return an array of ReportGroup objects that are specified by the input reportGroupArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReportGroups.html
     */
    toBatchGetReportGroups() {
        return this.to('BatchGetReportGroups');
    }
    /**
     * Grants permission to return an array of the Report objects specified by the input reportArns parameter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetReports.html
     */
    toBatchGetReports() {
        return this.to('BatchGetReports');
    }
    /**
     * Grants permission to get information about one or more sandboxes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_BatchGetSandboxes.html
     */
    toBatchGetSandboxes() {
        return this.to('BatchGetSandboxes');
    }
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutCodeCoverages() {
        return this.to('BatchPutCodeCoverages');
    }
    /**
     * Grants permission to add or update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toBatchPutTestCases() {
        return this.to('BatchPutTestCases');
    }
    /**
     * Grants permission to create a compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create a build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to create a report. A report is created when tests specified in the buildspec file for a report groups run during the build of a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toCreateReport() {
        return this.to('CreateReport');
    }
    /**
     * Grants permission to create a report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateReportGroup.html
     */
    toCreateReportGroup() {
        return this.to('CreateReportGroup');
    }
    /**
     * Grants permission to create webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, enables AWS CodeBuild to start rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_CreateWebhook.html
     */
    toCreateWebhook() {
        return this.to('CreateWebhook');
    }
    /**
     * Grants permission to delete a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteBuildBatch.html
     */
    toDeleteBuildBatch() {
        return this.to('DeleteBuildBatch');
    }
    /**
     * Grants permission to delete a compute fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toDeleteOAuthToken() {
        return this.to('DeleteOAuthToken');
    }
    /**
     * Grants permission to delete a build project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to delete a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReport.html
     */
    toDeleteReport() {
        return this.to('DeleteReport');
    }
    /**
     * Grants permission to delete a report group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteReportGroup.html
     */
    toDeleteReportGroup() {
        return this.to('DeleteReportGroup');
    }
    /**
     * Grants permission to delete a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a set of GitHub, GitHub Enterprise, or Bitbucket source credentials
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteSourceCredentials.html
     */
    toDeleteSourceCredentials() {
        return this.to('DeleteSourceCredentials');
    }
    /**
     * Grants permission to delete webhook. For an existing AWS CodeBuild build project that has its source code stored in a GitHub or Bitbucket repository, stops AWS CodeBuild from rebuilding the source code every time a code change is pushed to the repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook() {
        return this.to('DeleteWebhook');
    }
    /**
     * Grants permission to return an array of CodeCoverage objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeCodeCoverages.html
     */
    toDescribeCodeCoverages() {
        return this.to('DescribeCodeCoverages');
    }
    /**
     * Grants permission to return an array of TestCase objects
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_DescribeTestCases.html
     */
    toDescribeTestCases() {
        return this.to('DescribeTestCases');
    }
    /**
     * Grants permission to analyze and accumulate test report values for the test reports in the specified report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetReportGroupTrend.html
     */
    toGetReportGroupTrend() {
        return this.to('GetReportGroupTrend');
    }
    /**
     * Grants permission to return a resource policy for the specified project or report group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to import the source repository credentials for an AWS CodeBuild project that has its source code stored in a GitHub, GitHub Enterprise, or Bitbucket repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ImportSourceCredentials.html
     */
    toImportSourceCredentials() {
        return this.to('ImportSourceCredentials');
    }
    /**
     * Grants permission to reset the cache for a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_InvalidateProjectCache.html
     */
    toInvalidateProjectCache() {
        return this.to('InvalidateProjectCache');
    }
    /**
     * Grants permission to get a list of build batch IDs, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatches.html
     */
    toListBuildBatches() {
        return this.to('ListBuildBatches');
    }
    /**
     * Grants permission to get a list of build batch IDs for the specified build project, with each build batch ID representing a single build batch
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildBatchesForProject.html
     */
    toListBuildBatchesForProject() {
        return this.to('ListBuildBatchesForProject');
    }
    /**
     * Grants permission to get a list of build IDs, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuilds.html
     */
    toListBuilds() {
        return this.to('ListBuilds');
    }
    /**
     * Grants permission to get a list of build IDs for the specified build project, with each build ID representing a single build
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListBuildsForProject.html
     */
    toListBuildsForProject() {
        return this.to('ListBuildsForProject');
    }
    /**
     * Grants permission to get a list of command execution IDs for the specified sandbox, with each command execution ID representing a single command execution
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCommandExecutionsForSandbox.html
     */
    toListCommandExecutionsForSandbox() {
        return this.to('ListCommandExecutionsForSandbox');
    }
    /**
     * Grants permission to list connected third-party OAuth providers. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListConnectedOAuthAccounts() {
        return this.to('ListConnectedOAuthAccounts');
    }
    /**
     * Grants permission to get information about Docker images that are managed by AWS CodeBuild
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListCuratedEnvironmentImages.html
     */
    toListCuratedEnvironmentImages() {
        return this.to('ListCuratedEnvironmentImages');
    }
    /**
     * Grants permission to get a list of compute fleet ARNs, with each compute fleet ARN representing a single fleet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to get a list of build project names, with each build project name representing a single build project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        return this.to('ListProjects');
    }
    /**
     * Grants permission to return a list of report group ARNs. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportGroups.html
     */
    toListReportGroups() {
        return this.to('ListReportGroups');
    }
    /**
     * Grants permission to return a list of report ARNs. Each report ARN representing one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReports.html
     */
    toListReports() {
        return this.to('ListReports');
    }
    /**
     * Grants permission to return a list of report ARNs that belong to the specified report group. Each report ARN represents one report
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListReportsForReportGroup.html
     */
    toListReportsForReportGroup() {
        return this.to('ListReportsForReportGroup');
    }
    /**
     * Grants permission to list source code repositories from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to get a list of sandbox IDs, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxes.html
     */
    toListSandboxes() {
        return this.to('ListSandboxes');
    }
    /**
     * Grants permission to get a list of sandbox IDs for the specified sandbox project, with each sandbox ID representing a single sandbox
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSandboxesForProject.html
     */
    toListSandboxesForProject() {
        return this.to('ListSandboxesForProject');
    }
    /**
     * Grants permission to return a list of project ARNs that have been shared with the requester. Each project ARN represents one project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedProjects.html
     */
    toListSharedProjects() {
        return this.to('ListSharedProjects');
    }
    /**
     * Grants permission to return a list of report group ARNs that have been shared with the requester. Each report group ARN represents one report group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSharedReportGroups.html
     */
    toListSharedReportGroups() {
        return this.to('ListSharedReportGroups');
    }
    /**
     * Grants permission to return a list of SourceCredentialsInfo objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ListSourceCredentials.html
     */
    toListSourceCredentials() {
        return this.to('ListSourceCredentials');
    }
    /**
     * Grants permission to save an OAuth token from a connected third-party OAuth provider. Only used in the AWS CodeBuild console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toPersistOAuthToken() {
        return this.to('PersistOAuthToken');
    }
    /**
     * Grants permission to create a resource policy for the associated project or report group
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to retry a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuild.html
     */
    toRetryBuild() {
        return this.to('RetryBuild');
    }
    /**
     * Grants permission to retry a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_RetryBuildBatch.html
     */
    toRetryBuildBatch() {
        return this.to('RetryBuildBatch');
    }
    /**
     * Grants permission to start running a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html
     */
    toStartBuild() {
        return this.to('StartBuild');
    }
    /**
     * Grants permission to start running a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuildBatch.html
     */
    toStartBuildBatch() {
        return this.to('StartBuildBatch');
    }
    /**
     * Grants permission to start running a command execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartCommandExecution.html
     */
    toStartCommandExecution() {
        return this.to('StartCommandExecution');
    }
    /**
     * Grants permission to start running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandbox.html
     */
    toStartSandbox() {
        return this.to('StartSandbox');
    }
    /**
     * Grants permission to establish a connection to the sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartSandboxConnection.html
     */
    toStartSandboxConnection() {
        return this.to('StartSandboxConnection');
    }
    /**
     * Grants permission to attempt to stop running a build
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuild.html
     */
    toStopBuild() {
        return this.to('StopBuild');
    }
    /**
     * Grants permission to attempt to stop running a build batch
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopBuildBatch.html
     */
    toStopBuildBatch() {
        return this.to('StopBuildBatch');
    }
    /**
     * Grants permission to attempt to stop running a sandbox
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StopSandbox.html
     */
    toStopSandbox() {
        return this.to('StopSandbox');
    }
    /**
     * Grants permission to change the settings of an existing compute fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to change the settings of an existing build project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        return this.to('UpdateProject');
    }
    /**
     * Grants permission to change the public visibility of a project and its builds
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateProjectVisibility.html
     */
    toUpdateProjectVisibility() {
        return this.to('UpdateProjectVisibility');
    }
    /**
     * Grants permission to update information about a report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-identity-based-access-control.html#console-policies
     */
    toUpdateReport() {
        return this.to('UpdateReport');
    }
    /**
     * Grants permission to change the settings of an existing report group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateReportGroup.html
     */
    toUpdateReportGroup() {
        return this.to('UpdateReportGroup');
    }
    /**
     * Grants permission to update the webhook associated with an AWS CodeBuild build project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_UpdateWebhook.html
     */
    toUpdateWebhook() {
        return this.to('UpdateWebhook');
    }
    /**
     * Adds a resource of type build to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildId - Identifier for the buildId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuild(buildId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:build/${buildId}`);
    }
    /**
     * Adds a resource of type build-batch to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param buildBatchId - Identifier for the buildBatchId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBuildBatch(buildBatchId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:build-batch/${buildBatchId}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param projectName - Identifier for the projectName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectName}`);
    }
    /**
     * Adds a resource of type report-group to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReportGroup(reportGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:report-group/${reportGroupName}`);
    }
    /**
     * Adds a resource of type report to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param reportGroupName - Identifier for the reportGroupName.
     * @param reportId - Identifier for the reportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReport(reportGroupName, reportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:report/${reportGroupName}:${reportId}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param fleetName - Identifier for the fleetName.
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFleet(fleetName, fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetName}:${fleetId}`);
    }
    /**
     * Adds a resource of type sandbox to the statement
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-iam-access-control-identity-based.html#arn-formats
     *
     * @param sandboxId - Identifier for the sandboxId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSandbox(sandboxId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codebuild:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sandbox/${sandboxId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - project
     * - report-group
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFleet()
     * - .toCreateProject()
     * - .toCreateReportGroup()
     * - .toUpdateFleet()
     * - .toUpdateProject()
     * - .toUpdateProjectVisibility()
     * - .toUpdateReportGroup()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of the AWS CodeBuild build from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifBuildArn(value, operator) {
        return this.if(`buildArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the ARN of the AWS CodeBuild project from which the request originated
     *
     * https://docs.aws.amazon.com/codebuild/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifProjectArn(value, operator) {
        return this.if(`projectArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [codebuild](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodebuild.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codebuild';
        this.accessLevelList = {
            Write: [
                'BatchDeleteBuilds',
                'BatchPutCodeCoverages',
                'BatchPutTestCases',
                'CreateFleet',
                'CreateProject',
                'CreateReport',
                'CreateReportGroup',
                'CreateWebhook',
                'DeleteBuildBatch',
                'DeleteFleet',
                'DeleteOAuthToken',
                'DeleteProject',
                'DeleteReport',
                'DeleteReportGroup',
                'DeleteSourceCredentials',
                'DeleteWebhook',
                'ImportSourceCredentials',
                'InvalidateProjectCache',
                'PersistOAuthToken',
                'RetryBuild',
                'RetryBuildBatch',
                'StartBuild',
                'StartBuildBatch',
                'StartCommandExecution',
                'StartSandbox',
                'StartSandboxConnection',
                'StopBuild',
                'StopBuildBatch',
                'StopSandbox',
                'UpdateFleet',
                'UpdateProject',
                'UpdateProjectVisibility',
                'UpdateReport',
                'UpdateReportGroup',
                'UpdateWebhook'
            ],
            Read: [
                'BatchGetBuildBatches',
                'BatchGetBuilds',
                'BatchGetCommandExecutions',
                'BatchGetFleets',
                'BatchGetProjects',
                'BatchGetReportGroups',
                'BatchGetReports',
                'BatchGetSandboxes',
                'DescribeCodeCoverages',
                'DescribeTestCases',
                'GetReportGroupTrend',
                'GetResourcePolicy'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            List: [
                'ListBuildBatches',
                'ListBuildBatchesForProject',
                'ListBuilds',
                'ListBuildsForProject',
                'ListCommandExecutionsForSandbox',
                'ListConnectedOAuthAccounts',
                'ListCuratedEnvironmentImages',
                'ListFleets',
                'ListProjects',
                'ListReportGroups',
                'ListReports',
                'ListReportsForReportGroup',
                'ListRepositories',
                'ListSandboxes',
                'ListSandboxesForProject',
                'ListSharedProjects',
                'ListSharedReportGroups',
                'ListSourceCredentials'
            ]
        };
    }
}
exports.Codebuild = Codebuild;
//# sourceMappingURL=data:application/json;base64,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