import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [codeguru-security](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodegurusecurity.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class CodeguruSecurity extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to batch retrieve specific findings generated by CodeGuru Security
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_BatchGetFindings.html
     */
    toBatchGetFindings(): this;
    /**
     * Grants permission to create a CodeGuru Security scan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_CreateScan.html
     */
    toCreateScan(): this;
    /**
     * Grants permission to generate a presigned url for uploading code archives
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_CreateUploadUrl.html
     */
    toCreateUploadUrl(): this;
    /**
     * Grants permission to delete all the scans and related findings from CodeGuru Security by given category
     *
     * Access Level: Write
     */
    toDeleteScansByCategory(): this;
    /**
     * Grants permission to retrieve the account level configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_GetAccountConfiguration.html
     */
    toGetAccountConfiguration(): this;
    /**
     * Grants permission to retrieve findings for a scan generated by CodeGuru Security
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_GetFindings.html
     */
    toGetFindings(): this;
    /**
     * Grants permission to retrieve AWS accout level metrics summary generated by CodeGuru Security
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_GetMetricsSummary.html
     */
    toGetMetricsSummary(): this;
    /**
     * Grants permission to retrieve CodeGuru Security scan metadata
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_GetScan.html
     */
    toGetScan(): this;
    /**
     * Grants permission to retrieve findings generated by CodeGuru Security
     *
     * Access Level: List
     */
    toListFindings(): this;
    /**
     * Grants permission to retrieve a list of account level findings metrics within a date range
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_ListFindingsMetrics.html
     */
    toListFindingsMetrics(): this;
    /**
     * Grants permission to retrieve list of CodeGuru Security scan metadata
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_ListScans.html
     */
    toListScans(): this;
    /**
     * Grants permission to retrieve a list of tags for a scan name ARN
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to add tags to a scan name ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from a scan name ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update the account level configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-api/API_UpdateAccountConfiguration.html
     */
    toUpdateAccountConfiguration(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type ScanName to the statement
     *
     * https://docs.aws.amazon.com/codeguru/latest/security-ug/working-with-code-scans.html
     *
     * @param scanName - Identifier for the scanName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onScanName(scanName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateScan()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetScan()
     * - .toListTagsForResource()
     *
     * Applies to resource types:
     * - ScanName
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateScan()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [codeguru-security](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncodegurusecurity.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
