"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdentity = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitoidentity.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdentity extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new identity pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html
     */
    toCreateIdentityPool() {
        return this.to('CreateIdentityPool');
    }
    /**
     * Grants permission to delete identities from an identity pool. You can specify a list of 1-60 identities that you want to delete
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html
     */
    toDeleteIdentities() {
        return this.to('DeleteIdentities');
    }
    /**
     * Grants permission to delete a user pool. Once a pool is deleted, users will not be able to authenticate with the pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html
     */
    toDeleteIdentityPool() {
        return this.to('DeleteIdentityPool');
    }
    /**
     * Grants permission to return metadata related to the given identity, including when the identity was created and any associated linked logins
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html
     */
    toDescribeIdentity() {
        return this.to('DescribeIdentity');
    }
    /**
     * Grants permission to get details about a particular identity pool, including the pool name, ID description, creation date, and current number of users
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html
     */
    toDescribeIdentityPool() {
        return this.to('DescribeIdentityPool');
    }
    /**
     * Grants permission to return credentials for the provided identity ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html
     */
    toGetCredentialsForIdentity() {
        return this.to('GetCredentialsForIdentity');
    }
    /**
     * Grants permission to generate (or retrieve) a Cognito ID. Supplying multiple logins will create an implicit linked account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html
     */
    toGetId() {
        return this.to('GetId');
    }
    /**
     * Grants permission to get analytics data about the total current identity count for all identity pool identity provider (IdPs)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolAnalytics.html
     */
    toGetIdentityPoolAnalytics() {
        return this.to('GetIdentityPoolAnalytics');
    }
    /**
     * Grants permission to get analytics data about the number of new identities and total identities for all identity pool identity providers (IdPs)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolDailyAnalytics.html
     */
    toGetIdentityPoolDailyAnalytics() {
        return this.to('GetIdentityPoolDailyAnalytics');
    }
    /**
     * Grants permission to get the roles for an identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html
     */
    toGetIdentityPoolRoles() {
        return this.to('GetIdentityPoolRoles');
    }
    /**
     * Grants permission to get analytics data about the number of new identities and total identities for one identity pool identity provider (IdPs)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityProviderDailyAnalytics.html
     */
    toGetIdentityProviderDailyAnalytics() {
        return this.to('GetIdentityProviderDailyAnalytics');
    }
    /**
     * Grants permission to get an OpenID token, using a known Cognito ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html
     */
    toGetOpenIdToken() {
        return this.to('GetOpenIdToken');
    }
    /**
     * Grants permission to register (or retrieve) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html
     */
    toGetOpenIdTokenForDeveloperIdentity() {
        return this.to('GetOpenIdTokenForDeveloperIdentity');
    }
    /**
     * Grants permission to get the principal tags for an identity pool and provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetPrincipalTagAttributeMap.html
     */
    toGetPrincipalTagAttributeMap() {
        return this.to('GetPrincipalTagAttributeMap');
    }
    /**
     * Grants permission to list the identities in an identity pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html
     */
    toListIdentities() {
        return this.to('ListIdentities');
    }
    /**
     * Grants permission to list all of the Cognito identity pools registered for your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html
     */
    toListIdentityPools() {
        return this.to('ListIdentityPools');
    }
    /**
     * Grants permission to list the tags that are assigned to an Amazon Cognito identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve the IdentityId associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html
     */
    toLookupDeveloperIdentity() {
        return this.to('LookupDeveloperIdentity');
    }
    /**
     * Grants permission to merge two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html
     */
    toMergeDeveloperIdentities() {
        return this.to('MergeDeveloperIdentities');
    }
    /**
     * Grants permission to set the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html
     */
    toSetIdentityPoolRoles() {
        return this.to('SetIdentityPoolRoles');
    }
    /**
     * Grants permission to set the principal tags for an identity pool and provider. These tags are used when making calls to GetOpenIdToken action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetPrincipalTagAttributeMap.html
     */
    toSetPrincipalTagAttributeMap() {
        return this.to('SetPrincipalTagAttributeMap');
    }
    /**
     * Grants permission to assign a set of tags to an Amazon Cognito identity pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to unlink a DeveloperUserIdentifier from an existing identity
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html
     */
    toUnlinkDeveloperIdentity() {
        return this.to('UnlinkDeveloperIdentity');
    }
    /**
     * Grants permission to unlink a federated identity from an existing account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html
     */
    toUnlinkIdentity() {
        return this.to('UnlinkIdentity');
    }
    /**
     * Grants permission to remove the specified tags from an Amazon Cognito identity pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an identity pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html
     */
    toUpdateIdentityPool() {
        return this.to('UpdateIdentityPool');
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-identity:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identitypool/${identityPoolId}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateIdentityPool()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - identitypool
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a key that is present in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateIdentityPool()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitoidentity.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-identity';
        this.accessLevelList = {
            Write: [
                'CreateIdentityPool',
                'DeleteIdentities',
                'DeleteIdentityPool',
                'GetId',
                'MergeDeveloperIdentities',
                'SetIdentityPoolRoles',
                'SetPrincipalTagAttributeMap',
                'UnlinkDeveloperIdentity',
                'UnlinkIdentity',
                'UpdateIdentityPool'
            ],
            Read: [
                'DescribeIdentity',
                'DescribeIdentityPool',
                'GetCredentialsForIdentity',
                'GetIdentityPoolAnalytics',
                'GetIdentityPoolDailyAnalytics',
                'GetIdentityPoolRoles',
                'GetIdentityProviderDailyAnalytics',
                'GetOpenIdToken',
                'GetOpenIdTokenForDeveloperIdentity',
                'GetPrincipalTagAttributeMap',
                'ListTagsForResource',
                'LookupDeveloperIdentity'
            ],
            List: [
                'ListIdentities',
                'ListIdentityPools'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CognitoIdentity = CognitoIdentity;
//# sourceMappingURL=data:application/json;base64,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