"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoSync = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitosync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoSync extends shared_1.PolicyStatement {
    /**
     * Grants permission to initiate a bulk publish of all existing datasets for an Identity Pool to the configured stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_BulkPublish.html
     */
    toBulkPublish() {
        return this.to('BulkPublish');
    }
    /**
     * Grants permission to delete a specific dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        return this.to('DeleteDataset');
    }
    /**
     * Grants permission to get metadata about a dataset by identity and dataset name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        return this.to('DescribeDataset');
    }
    /**
     * Grants permission to get usage details (for example, data storage) about a particular identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityPoolUsage.html
     */
    toDescribeIdentityPoolUsage() {
        return this.to('DescribeIdentityPoolUsage');
    }
    /**
     * Grants permission to get usage information for an identity, including number of datasets and data usage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_DescribeIdentityUsage.html
     */
    toDescribeIdentityUsage() {
        return this.to('DescribeIdentityUsage');
    }
    /**
     * Grants permission to get the status of the last BulkPublish operation for an identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetBulkPublishDetails.html
     */
    toGetBulkPublishDetails() {
        return this.to('GetBulkPublishDetails');
    }
    /**
     * Grants permission to get the events and the corresponding Lambda functions associated with an identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetCognitoEvents.html
     */
    toGetCognitoEvents() {
        return this.to('GetCognitoEvents');
    }
    /**
     * Grants permission to get the configuration settings of an identity pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_GetIdentityPoolConfiguration.html
     */
    toGetIdentityPoolConfiguration() {
        return this.to('GetIdentityPoolConfiguration');
    }
    /**
     * Grants permission to list datasets for an identity
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        return this.to('ListDatasets');
    }
    /**
     * Grants permission to get a list of identity pools registered with Cognito
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListIdentityPoolUsage.html
     */
    toListIdentityPoolUsage() {
        return this.to('ListIdentityPoolUsage');
    }
    /**
     * Grants permission to get paginated records, optionally changed after a particular sync count for a dataset and identity
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_ListRecords.html
     */
    toListRecords() {
        return this.to('ListRecords');
    }
    /**
     * Grants permission to query records
     *
     * Access Level: Read
     */
    toQueryRecords() {
        return this.to('QueryRecords');
    }
    /**
     * Grants permission to register a device to receive push sync notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_RegisterDevice.html
     */
    toRegisterDevice() {
        return this.to('RegisterDevice');
    }
    /**
     * Grants permission to set the AWS Lambda function for a given event type for an identity pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetCognitoEvents.html
     */
    toSetCognitoEvents() {
        return this.to('SetCognitoEvents');
    }
    /**
     * Grants permission to configure datasets
     *
     * Access Level: Write
     */
    toSetDatasetConfiguration() {
        return this.to('SetDatasetConfiguration');
    }
    /**
     * Grants permission to set the necessary configuration for push sync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SetIdentityPoolConfiguration.html
     */
    toSetIdentityPoolConfiguration() {
        return this.to('SetIdentityPoolConfiguration');
    }
    /**
     * Grants permission to subscribe to receive notifications when a dataset is modified by another device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_SubscribeToDataset.html
     */
    toSubscribeToDataset() {
        return this.to('SubscribeToDataset');
    }
    /**
     * Grants permission to unsubscribe from receiving notifications when a dataset is modified by another device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UnsubscribeFromDataset.html
     */
    toUnsubscribeFromDataset() {
        return this.to('UnsubscribeFromDataset');
    }
    /**
     * Grants permission to post updates to records and add and delete records for a dataset and user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitosync/latest/APIReference/API_UpdateRecords.html
     */
    toUpdateRecords() {
        return this.to('UpdateRecords');
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/synchronizing-data.html#understanding-datasets
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDataset(identityPoolId, identityId, datasetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-sync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identitypool/${identityPoolId}/identity/${identityId}/dataset/${datasetName}`);
    }
    /**
     * Adds a resource of type identity to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html#authenticated-and-unauthenticated-identities
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param identityId - Identifier for the identityId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIdentity(identityPoolId, identityId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-sync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identitypool/${identityPoolId}/identity/${identityId}`);
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-sync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:identitypool/${identityPoolId}`);
    }
    /**
     * Statement provider for service [cognito-sync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitosync.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-sync';
        this.accessLevelList = {
            Write: [
                'BulkPublish',
                'DeleteDataset',
                'RegisterDevice',
                'SetCognitoEvents',
                'SetDatasetConfiguration',
                'SetIdentityPoolConfiguration',
                'SubscribeToDataset',
                'UnsubscribeFromDataset',
                'UpdateRecords'
            ],
            Read: [
                'DescribeDataset',
                'DescribeIdentityPoolUsage',
                'DescribeIdentityUsage',
                'GetBulkPublishDetails',
                'GetCognitoEvents',
                'GetIdentityPoolConfiguration',
                'ListIdentityPoolUsage',
                'ListRecords',
                'QueryRecords'
            ],
            List: [
                'ListDatasets'
            ]
        };
    }
}
exports.CognitoSync = CognitoSync;
//# sourceMappingURL=data:application/json;base64,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