import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Connect extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to activate an evaluation form in the specified Amazon Connect instance. After the evaluation form is activated, it is available to start new evaluations based on the form
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ActivateEvaluationForm.html
     */
    toActivateEvaluationForm(): this;
    /**
     * Grants permission to federate into an Amazon Connect instance (Log in for emergency access functionality in the Amazon Connect console)
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - connect:ListInstances
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetFederationToken.html
     */
    toAdminGetEmergencyAccessToken(): this;
    /**
     * Grants permission to grant access and to associate a dataset with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toAssociateAnalyticsDataSet(): this;
    /**
     * Grants permission to associate approved origin for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateApprovedOrigin(): this;
    /**
     * Grants permission to associate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:CreateResourcePolicy
     * - lex:DescribeBotAlias
     * - lex:GetBot
     * - lex:UpdateResourcePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateBot(): this;
    /**
     * Grants permission to associate a Customer Profiles domain for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - profile:GetDomain
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateCustomerProfilesDomain(): this;
    /**
     * Grants permission to default vocabulary for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toAssociateDefaultVocabulary(): this;
    /**
     * Grants permission to associate a resource with a flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateFlow.html
     */
    toAssociateFlow(): this;
    /**
     * Grants permission to associate instance storage for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - firehose:DescribeDeliveryStream
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kinesis:DescribeStream
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateInstanceStorageConfig(): this;
    /**
     * Grants permission to associate a Lambda function for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - lambda:AddPermission
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateLambdaFunction(): this;
    /**
     * Grants permission to associate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:GetBot
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateLexBot(): this;
    /**
     * Grants permission to associate contact flow resources to phone number resources in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociatePhoneNumberContactFlow.html
     */
    toAssociatePhoneNumberContactFlow(): this;
    /**
     * Grants permission to associate quick connects with a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateQueueQuickConnects.html
     */
    toAssociateQueueQuickConnects(): this;
    /**
     * Grants permission to associate queues with a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateRoutingProfileQueues.html
     */
    toAssociateRoutingProfileQueues(): this;
    /**
     * Grants permission to associate a security key for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateSecurityKey(): this;
    /**
     * Grants permission to associate a user to a traffic distribution group in the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeUser
     * - connect:SearchUsers
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateTrafficDistributionGroupUser.html
     */
    toAssociateTrafficDistributionGroupUser(): this;
    /**
     * Grants permission to associate user proficiencies to a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateUserProficiencies.html
     */
    toAssociateUserProficiencies(): this;
    /**
     * Grants permission to grant access and to associate the datasets with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toBatchAssociateAnalyticsDataSet(): this;
    /**
     * Grants permission to revoke access and to disassociate the datasets with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toBatchDisassociateAnalyticsDataSet(): this;
    /**
     * Grants permission to get metadata for multiple attached files from an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_BatchGetAttachedFileMetadata.html
     */
    toBatchGetAttachedFileMetadata(): this;
    /**
     * Grants permission to get summary information about the flow associations for the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_BatchGetFlowAssociation.html
     */
    toBatchGetFlowAssociation(): this;
    /**
     * Grants permission to put contacts in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_BatchPutContact.html
     */
    toBatchPutContact(): this;
    /**
     * Grants permission to claim phone number resources in an Amazon Connect instance or traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ClaimPhoneNumber.html
     */
    toClaimPhoneNumber(): this;
    /**
     * Grants permission to complete an attached file upload in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CompleteAttachedFileUpload.html
     */
    toCompleteAttachedFileUpload(): this;
    /**
     * Grants permission to create agent status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateAgentStatus.html
     */
    toCreateAgentStatus(): this;
    /**
     * Grants permission to create authentication profile resources in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateAuthenticationProfile.html
     */
    toCreateAuthenticationProfile(): this;
    /**
     * Grants permission to create a new contact using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifContactInitiationMethod()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContact.html
     */
    toCreateContact(): this;
    /**
     * Grants permission to create a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlow.html
     */
    toCreateContactFlow(): this;
    /**
     * Grants permission to create a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlowModule.html
     */
    toCreateContactFlowModule(): this;
    /**
     * Grants permission to create a version a flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlowVersion.html
     */
    toCreateContactFlowVersion(): this;
    /**
     * Grants permission to create an email address resource in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateEmailAddress.html
     */
    toCreateEmailAddress(): this;
    /**
     * Grants permission to create an evaluation form in the specified Amazon Connect instance. The form can be used to define questions related to agent performance, and create sections to organize such questions. Question and section identifiers cannot be duplicated within the same evaluation form
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateEvaluationForm.html
     */
    toCreateEvaluationForm(): this;
    /**
     * Grants permission to create hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateHoursOfOperation.html
     */
    toCreateHoursOfOperation(): this;
    /**
     * Grants permission to create an hours of operation override in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateHoursOfOperationOverride.html
     */
    toCreateHoursOfOperationOverride(): this;
    /**
     * Grants permission to create a new Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     * - ds:CheckAlias
     * - ds:CreateAlias
     * - ds:CreateDirectory
     * - ds:CreateIdentityPoolDirectory
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - ds:UnauthorizeApplication
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toCreateInstance(): this;
    /**
     * Grants permission to create an integration association with an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - app-integrations:CreateApplicationAssociation
     * - app-integrations:CreateEventIntegrationAssociation
     * - app-integrations:GetApplication
     * - cases:GetDomain
     * - chime:AssociateVoiceConnectorConnect
     * - chime:DisassociateVoiceConnectorConnect
     * - chime:TagResource
     * - chime:UntagResource
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     * - events:PutRule
     * - events:PutTargets
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - mobiletargeting:GetApp
     * - voiceid:DescribeDomain
     * - wisdom:GetAssistant
     * - wisdom:GetKnowledgeBase
     * - wisdom:TagResource
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateIntegrationAssociation.html
     */
    toCreateIntegrationAssociation(): this;
    /**
     * Grants permission to add a participant to an ongoing contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateParticipant.html
     */
    toCreateParticipant(): this;
    /**
     * Grants permission to create persistent contact associations for a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreatePersistentContactAssociation.html
     */
    toCreatePersistentContactAssociation(): this;
    /**
     * Grants permission to create a predefined attribute in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreatePredefinedAttribute.html
     */
    toCreatePredefinedAttribute(): this;
    /**
     * Grants permission to create a prompt in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - kms:Decrypt
     * - s3:GetObject
     * - s3:GetObjectAcl
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreatePrompt.html
     */
    toCreatePrompt(): this;
    /**
     * Grants permission to create a push notification registration for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreatePushNotificationRegistration.html
     */
    toCreatePushNotificationRegistration(): this;
    /**
     * Grants permission to create a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue(): this;
    /**
     * Grants permission to create a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateQuickConnect.html
     */
    toCreateQuickConnect(): this;
    /**
     * Grants permission to create a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateRoutingProfile.html
     */
    toCreateRoutingProfile(): this;
    /**
     * Grants permission to create a rule in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateRule.html
     */
    toCreateRule(): this;
    /**
     * Grants permission to create a security profile for the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateSecurityProfile.html
     */
    toCreateSecurityProfile(): this;
    /**
     * Grants permission to create a task template in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateTaskTemplate.html
     */
    toCreateTaskTemplate(): this;
    /**
     * Grants permission to create a traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateTrafficDistributionGroup.html
     */
    toCreateTrafficDistributionGroup(): this;
    /**
     * Grants permission to create a use case for an integration association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUseCase.html
     */
    toCreateUseCase(): this;
    /**
     * Grants permission to create a user for the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUser.html
     */
    toCreateUser(): this;
    /**
     * Grants permission to create a user hierarchy group in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUserHierarchyGroup.html
     */
    toCreateUserHierarchyGroup(): this;
    /**
     * Grants permission to create a view in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateView.html
     */
    toCreateView(): this;
    /**
     * Grants permission to create a view version in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateViewVersion.html
     */
    toCreateViewVersion(): this;
    /**
     * Grants permission to create a vocabulary in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateVocabulary.html
     */
    toCreateVocabulary(): this;
    /**
     * Grants permission to deactivate an evaluation form in the specified Amazon Connect instance. After a form is deactivated, it is no longer available for users to start new evaluations based on the form
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeactivateEvaluationForm.html
     */
    toDeactivateEvaluationForm(): this;
    /**
     * Grants permission to delete an attached file from an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - cases:DeleteRelatedItem
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteAttachedFile.html
     */
    toDeleteAttachedFile(): this;
    /**
     * Grants permission to delete a contact evaluation in the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactEvaluation.html
     */
    toDeleteContactEvaluation(): this;
    /**
     * Grants permission to delete a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactFlow.html
     */
    toDeleteContactFlow(): this;
    /**
     * Grants permission to delete a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactFlowModule.html
     */
    toDeleteContactFlowModule(): this;
    /**
     * Grants permission to delete a version of a flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactFlowVersion.html
     */
    toDeleteContactFlowVersion(): this;
    /**
     * Grants permission to delete an email address resource in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteEmailAddress.html
     */
    toDeleteEmailAddress(): this;
    /**
     * Grants permission to delete an evaluation form in the specified Amazon Connect instance. If the version property is provided, only the specified version of the evaluation form is deleted
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteEvaluationForm.html
     */
    toDeleteEvaluationForm(): this;
    /**
     * Grants permission to delete hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteHoursOfOperation.html
     */
    toDeleteHoursOfOperation(): this;
    /**
     * Grants permission to delete an hours of operation override in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteHoursOfOperationOverride.html
     */
    toDeleteHoursOfOperationOverride(): this;
    /**
     * Grants permission to delete an Amazon Connect instance. When you remove an instance, the link to an existing AWS directory is also removed
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - ds:UnauthorizeApplication
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDeleteInstance(): this;
    /**
     * Grants permission to delete an integration association from an Amazon Connect instance. The association must not have any use cases associated with it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - app-integrations:DeleteApplicationAssociation
     * - app-integrations:DeleteEventIntegrationAssociation
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     * - events:DeleteRule
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteIntegrationAssociation.html
     */
    toDeleteIntegrationAssociation(): this;
    /**
     * Grants permission to delete a predefined attribute in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeletePredefinedAttribute.html
     */
    toDeletePredefinedAttribute(): this;
    /**
     * Grants permission to delete a prompt in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeletePrompt.html
     */
    toDeletePrompt(): this;
    /**
     * Grants permission to delete a push notification registration for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeletePushNotificationRegistration.html
     */
    toDeletePushNotificationRegistration(): this;
    /**
     * Grants permission to delete a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue(): this;
    /**
     * Grants permission to delete a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteQuickConnect.html
     */
    toDeleteQuickConnect(): this;
    /**
     * Grants permission to delete routing profiles in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteRoutingProfile.html
     */
    toDeleteRoutingProfile(): this;
    /**
     * Grants permission to delete a rule in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule(): this;
    /**
     * Grants permission to delete a security profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteSecurityProfile.html
     */
    toDeleteSecurityProfile(): this;
    /**
     * Grants permission to delete a task template in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteTaskTemplate.html
     */
    toDeleteTaskTemplate(): this;
    /**
     * Grants permission to delete a traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteTrafficDistributionGroup.html
     */
    toDeleteTrafficDistributionGroup(): this;
    /**
     * Grants permission to delete a use case from an integration association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUseCase.html
     */
    toDeleteUseCase(): this;
    /**
     * Grants permission to delete a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser(): this;
    /**
     * Grants permission to delete a user hierarchy group in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUserHierarchyGroup.html
     */
    toDeleteUserHierarchyGroup(): this;
    /**
     * Grants permission to delete a view in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteView.html
     */
    toDeleteView(): this;
    /**
     * Grants permission to delete a view version in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteViewVersion.html
     */
    toDeleteViewVersion(): this;
    /**
     * Grants permission to delete a vocabulary in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteVocabulary.html
     */
    toDeleteVocabulary(): this;
    /**
     * Grants permission to describe agent status in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeAgentStatus.html
     */
    toDescribeAgentStatus(): this;
    /**
     * Grants permission to describe authentication profile resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeAuthenticationProfile.html
     */
    toDescribeAuthenticationProfile(): this;
    /**
     * Grants permission to describe a contact in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContact.html
     */
    toDescribeContact(): this;
    /**
     * Grants permission to describe a contact evaluation in the specified Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactEvaluation.html
     */
    toDescribeContactEvaluation(): this;
    /**
     * Grants permission to describe a contact flow in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactFlow.html
     */
    toDescribeContactFlow(): this;
    /**
     * Grants permission to describe a contact flow module in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactFlowModule.html
     */
    toDescribeContactFlowModule(): this;
    /**
     * Grants permission to describe an email address resource in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeEmailAddress.html
     */
    toDescribeEmailAddress(): this;
    /**
     * Grants permission to describe an evaluation form in the specified Amazon Connect instance. If the version property is not provided, the latest version of the evaluation form is described
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeEvaluationForm.html
     */
    toDescribeEvaluationForm(): this;
    /**
     * Grants permission to describe the status of forecasting, planning, and scheduling integration on an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/optimization-apis.html
     */
    toDescribeForecastingPlanningSchedulingIntegration(): this;
    /**
     * Grants permission to describe hours of operation in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeHoursOfOperation.html
     */
    toDescribeHoursOfOperation(): this;
    /**
     * Grants permission to describe an hours of operation override in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeHoursOfOperationOverride.html
     */
    toDescribeHoursOfOperationOverride(): this;
    /**
     * Grants permission to view details of an Amazon Connect instance and is also required to create an instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstance(): this;
    /**
     * Grants permission to view the attribute details of an existing Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributeType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstanceAttribute(): this;
    /**
     * Grants permission to view the instance storage configuration for an existing Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstanceStorageConfig(): this;
    /**
     * Grants permission to describe phone number resources in an Amazon Connect instance or traffic distribution group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePhoneNumber.html
     */
    toDescribePhoneNumber(): this;
    /**
     * Grants permission to describe a predefined attribute in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePredefinedAttribute.html
     */
    toDescribePredefinedAttribute(): this;
    /**
     * Grants permission to describe a prompt in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribePrompt.html
     */
    toDescribePrompt(): this;
    /**
     * Grants permission to describe a queue in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeQueue.html
     */
    toDescribeQueue(): this;
    /**
     * Grants permission to describe a quick connect in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeQuickConnect.html
     */
    toDescribeQuickConnect(): this;
    /**
     * Grants permission to describe a routing profile in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeRoutingProfile.html
     */
    toDescribeRoutingProfile(): this;
    /**
     * Grants permission to describe a rule in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeRule.html
     */
    toDescribeRule(): this;
    /**
     * Grants permission to describe a security profile in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeSecurityProfile.html
     */
    toDescribeSecurityProfile(): this;
    /**
     * Grants permission to describe a traffic distribution group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeTrafficDistributionGroup.html
     */
    toDescribeTrafficDistributionGroup(): this;
    /**
     * Grants permission to describe a user in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser(): this;
    /**
     * Grants permission to describe a hierarchy group for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyGroup.html
     */
    toDescribeUserHierarchyGroup(): this;
    /**
     * Grants permission to describe the hierarchy structure for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyStructure.html
     */
    toDescribeUserHierarchyStructure(): this;
    /**
     * Grants permission to describe a view in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeView.html
     */
    toDescribeView(): this;
    /**
     * Grants permission to describe a vocabulary in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeVocabulary.html
     */
    toDescribeVocabulary(): this;
    /**
     * Grants permission to revoke access and to disassociate a dataset with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toDisassociateAnalyticsDataSet(): this;
    /**
     * Grants permission to disassociate approved origin for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateApprovedOrigin(): this;
    /**
     * Grants permission to disassociate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:DeleteResourcePolicy
     * - lex:UpdateResourcePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateBot(): this;
    /**
     * Grants permission to disassociate a Customer Profiles domain for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:DeleteRolePolicy
     * - iam:DetachRolePolicy
     * - iam:GetPolicy
     * - iam:GetPolicyVersion
     * - iam:GetRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateCustomerProfilesDomain(): this;
    /**
     * Grants permission to disassociate a resource from a flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateFlow.html
     */
    toDisassociateFlow(): this;
    /**
     * Grants permission to disassociate instance storage for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateInstanceStorageConfig(): this;
    /**
     * Grants permission to disassociate a Lambda function for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - lambda:RemovePermission
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateLambdaFunction(): this;
    /**
     * Grants permission to disassociate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateLexBot(): this;
    /**
     * Grants permission to disassociate contact flow resources from phone number resources in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociatePhoneNumberContactFlow.html
     */
    toDisassociatePhoneNumberContactFlow(): this;
    /**
     * Grants permission to disassociate quick connects from a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateQueueQuickConnects.html
     */
    toDisassociateQueueQuickConnects(): this;
    /**
     * Grants permission to disassociate queues from a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateRoutingProfileQueues.html
     */
    toDisassociateRoutingProfileQueues(): this;
    /**
     * Grants permission to disassociate the security key for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateSecurityKey(): this;
    /**
     * Grants permission to disassociate a user from a traffic distribution group in the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateTrafficDistributionGroupUser.html
     */
    toDisassociateTrafficDistributionGroupUser(): this;
    /**
     * Grants permission to disassociate user proficiencies from a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateUserProficiencies.html
     */
    toDisassociateUserProficiencies(): this;
    /**
     * Grants permission to dismiss terminated Contact from Agent CCP
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DismissUserContact.html
     */
    toDismissUserContact(): this;
    /**
     * Grants permission to get an attached file from an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetAttachedFile.html
     */
    toGetAttachedFile(): this;
    /**
     * Grants permission to retrieve the contact attributes for the specified contact
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetContactAttributes.html
     */
    toGetContactAttributes(): this;
    /**
     * Grants permission to retrieve current metric data for queues and routing profiles in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetCurrentMetricData.html
     */
    toGetCurrentMetricData(): this;
    /**
     * Grants permission to retrieve current user data in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetCurrentUserData.html
     */
    toGetCurrentUserData(): this;
    /**
     * Grants permission to get effective hours of operation resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetEffectiveHoursOfOperations.html
     */
    toGetEffectiveHoursOfOperations(): this;
    /**
     * Grants permission to federate into an Amazon Connect instance when using SAML-based authentication for identity management
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetFederationToken.html
     */
    toGetFederationToken(): this;
    /**
     * Grants permission to get information about the flow associations for the specified Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetFlowAssociation.html
     */
    toGetFlowAssociation(): this;
    /**
     * Grants permission to retrieve historical metric data for queues in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData(): this;
    /**
     * Grants permission to retrieve metric data in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetMetricDataV2.html
     */
    toGetMetricDataV2(): this;
    /**
     * Grants permission to get details about a prompt's presigned Amazon S3 URL in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetPromptFile.html
     */
    toGetPromptFile(): this;
    /**
     * Grants permission to get details about specified task template in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetTaskTemplate.html
     */
    toGetTaskTemplate(): this;
    /**
     * Grants permission to read traffic distribution for a traffic distribution group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetTrafficDistribution.html
     */
    toGetTrafficDistribution(): this;
    /**
     * Grants permission to import phone number resources to an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:DescribePhoneNumbers
     * - social-messaging:GetLinkedWhatsAppBusinessAccountPhoneNumber
     * - social-messaging:TagResource
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ImportPhoneNumber.html
     */
    toImportPhoneNumber(): this;
    /**
     * Grants permission to list agent statuses in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListAgentStatuses.html
     */
    toListAgentStatuses(): this;
    /**
     * Grants permission to list the association status of a dataset for a given Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toListAnalyticsDataAssociations(): this;
    /**
     * Grants permission to list data lake datasets available to associate with for a given Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toListAnalyticsDataLakeDataSets(): this;
    /**
     * Grants permission to view approved origins of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListApprovedOrigins(): this;
    /**
     * Grants permission to list the contacts associated with an email address in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListAssociatedContacts.html
     */
    toListAssociatedContacts(): this;
    /**
     * Grants permission to list authentication profile resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListAuthenticationProfiles.html
     */
    toListAuthenticationProfiles(): this;
    /**
     * Grants permission to view the Lex bots of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListBots(): this;
    /**
     * Grants permission to list contact evaluations in the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactEvaluations.html
     */
    toListContactEvaluations(): this;
    /**
     * Grants permission to list contact flow module resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlowModules.html
     */
    toListContactFlowModules(): this;
    /**
     * Grants permission to list all the versions a flow in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlowVersions.html
     */
    toListContactFlowVersions(): this;
    /**
     * Grants permission to list contact flow resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlows.html
     */
    toListContactFlows(): this;
    /**
     * Grants permission to list references associated with a contact in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactReferences.html
     */
    toListContactReferences(): this;
    /**
     * Grants permission to list default vocabularies associated with a Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListDefaultVocabularies.html
     */
    toListDefaultVocabularies(): this;
    /**
     * Grants permission to list versions of an evaluation form in the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListEvaluationFormVersions.html
     */
    toListEvaluationFormVersions(): this;
    /**
     * Grants permission to list evaluation forms in the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListEvaluationForms.html
     */
    toListEvaluationForms(): this;
    /**
     * Grants permission to list summary information about the flow associations for the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListFlowAssociations.html
     */
    toListFlowAssociations(): this;
    /**
     * Grants permission to list hours of operation override resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListHoursOfOperationOverrides.html
     */
    toListHoursOfOperationOverrides(): this;
    /**
     * Grants permission to list hours of operation resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListHoursOfOperations.html
     */
    toListHoursOfOperations(): this;
    /**
     * Grants permission to view the attributes of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstanceAttributes(): this;
    /**
     * Grants permission to view storage configurations of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstanceStorageConfigs(): this;
    /**
     * Grants permission to view the Amazon Connect instances associated with an AWS account
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstances(): this;
    /**
     * Grants permission to list summary information about the integration associations for the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListIntegrationAssociations.html
     */
    toListIntegrationAssociations(): this;
    /**
     * Grants permission to view the Lambda functions of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListLambdaFunctions(): this;
    /**
     * Grants permission to view the Lex bots of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListLexBots(): this;
    /**
     * Grants permission to list phone number resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers(): this;
    /**
     * Grants permission to list phone number resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPhoneNumbersV2.html
     */
    toListPhoneNumbersV2(): this;
    /**
     * Grants permission to list predefined attributes in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPredefinedAttributes.html
     */
    toListPredefinedAttributes(): this;
    /**
     * Grants permission to list prompt resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPrompts.html
     */
    toListPrompts(): this;
    /**
     * Grants permission to list quick connect resources in a queue in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQueueQuickConnects.html
     */
    toListQueueQuickConnects(): this;
    /**
     * Grants permission to list queue resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQueues.html
     */
    toListQueues(): this;
    /**
     * Grants permission to list quick connect resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQuickConnects.html
     */
    toListQuickConnects(): this;
    /**
     * Grants permission to list the analysis segments for a real-time analysis session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/contact-lens/latest/APIReference/API_ListRealtimeContactAnalysisSegments.html
     */
    toListRealtimeContactAnalysisSegments(): this;
    /**
     * Grants permission to list the analysis segments for a real-time chat analytics session
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRealtimeContactAnalysisSegmentsV2.html
     */
    toListRealtimeContactAnalysisSegmentsV2(): this;
    /**
     * Grants permission to list queue resources in a routing profile in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfileQueues.html
     */
    toListRoutingProfileQueues(): this;
    /**
     * Grants permission to list routing profile resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfiles.html
     */
    toListRoutingProfiles(): this;
    /**
     * Grants permission to list rules associated with a Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRules.html
     */
    toListRules(): this;
    /**
     * Grants permission to view the security keys of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListSecurityKeys(): this;
    /**
     * Grants permission to list applications associated with a specific security profile in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfileApplications.html
     */
    toListSecurityProfileApplications(): this;
    /**
     * Grants permission to list permissions associated with security profile in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfilePermissions.html
     */
    toListSecurityProfilePermissions(): this;
    /**
     * Grants permission to list security profile resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfiles.html
     */
    toListSecurityProfiles(): this;
    /**
     * Grants permission to list tags for an Amazon Connect resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list task template resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTaskTemplates.html
     */
    toListTaskTemplates(): this;
    /**
     * Grants permission to list the active user associations for a traffic distribution group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTrafficDistributionGroupUsers.html
     */
    toListTrafficDistributionGroupUsers(): this;
    /**
     * Grants permission to list traffic distribution groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTrafficDistributionGroups.html
     */
    toListTrafficDistributionGroups(): this;
    /**
     * Grants permission to list the use cases of an integration association
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUseCases.html
     */
    toListUseCases(): this;
    /**
     * Grants permission to list the hierarchy group resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUserHierarchyGroups.html
     */
    toListUserHierarchyGroups(): this;
    /**
     * Grants permission to list user proficiencies from a user in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUserProficiencies.html
     */
    toListUserProficiencies(): this;
    /**
     * Grants permission to list user resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUsers.html
     */
    toListUsers(): this;
    /**
     * Grants permission to list the view versions in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListViewVersions.html
     */
    toListViewVersions(): this;
    /**
     * Grants permission to list the views in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListViews.html
     */
    toListViews(): this;
    /**
     * Grants permission to monitor an ongoing contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifMonitorCapabilities()
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_MonitorContact.html
     */
    toMonitorContact(): this;
    /**
     * Grants permission to pause an ongoing contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_PauseContact.html
     */
    toPauseContact(): this;
    /**
     * Grants permission to switch User Status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_PutUserStatus.html
     */
    toPutUserStatus(): this;
    /**
     * Grants permission to release phone number resources in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ReleasePhoneNumber.html
     */
    toReleasePhoneNumber(): this;
    /**
     * Grants permission to create a replica of an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     * - ds:CheckAlias
     * - ds:CreateAlias
     * - ds:CreateDirectory
     * - ds:CreateIdentityPoolDirectory
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - ds:UnauthorizeApplication
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toReplicateInstance(): this;
    /**
     * Grants permission to resume a paused contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ResumeContact.html
     */
    toResumeContact(): this;
    /**
     * Grants permission to resume recording for the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ResumeContactRecording.html
     */
    toResumeContactRecording(): this;
    /**
     * Grants permission to search agent status resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeAgentStatus
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchAgentStatuses.html
     */
    toSearchAgentStatuses(): this;
    /**
     * Grants permission to search phone number resources in an Amazon Connect instance or traffic distribution group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchAvailablePhoneNumbers.html
     */
    toSearchAvailablePhoneNumbers(): this;
    /**
     * Grants permission to search contact flow module resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeContactFlowModule
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchContactFlowModules.html
     */
    toSearchContactFlowModules(): this;
    /**
     * Grants permission to search contact flow resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     * - .ifFlowType()
     *
     * Dependent actions:
     * - connect:DescribeContactFlow
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchContactFlows.html
     */
    toSearchContactFlows(): this;
    /**
     * Grants permission to search contacts in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchContactsByContactAnalysis()
     *
     * Dependent actions:
     * - connect:DescribeContact
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchContacts.html
     */
    toSearchContacts(): this;
    /**
     * Grants permission to search email address resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeEmailAddress
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchEmailAddresses.html
     */
    toSearchEmailAddresses(): this;
    /**
     * Grants permission to search hours of operation override resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeHoursOfOperation
     * - connect:ListHoursOfOperationOverrides
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchHoursOfOperationOverrides.html
     */
    toSearchHoursOfOperationOverrides(): this;
    /**
     * Grants permission to search hours of operation resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeHoursOfOperation
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchHoursOfOperations.html
     */
    toSearchHoursOfOperations(): this;
    /**
     * Grants permission to search predefined attributes in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribePredefinedAttribute
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchPredefinedAttributes.html
     */
    toSearchPredefinedAttributes(): this;
    /**
     * Grants permission to search prompt resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribePrompt
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchPrompts.html
     */
    toSearchPrompts(): this;
    /**
     * Grants permission to search queue resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeQueue
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchQueues.html
     */
    toSearchQueues(): this;
    /**
     * Grants permission to search quick connect resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeQuickConnect
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchQuickConnects.html
     */
    toSearchQuickConnects(): this;
    /**
     * Grants permission to search tags that are used in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchResourceTags.html
     */
    toSearchResourceTags(): this;
    /**
     * Grants permission to search routing profile resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeRoutingProfile
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchRoutingProfiles.html
     */
    toSearchRoutingProfiles(): this;
    /**
     * Grants permission to search security profile resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeSecurityProfile
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchSecurityProfiles.html
     */
    toSearchSecurityProfiles(): this;
    /**
     * Grants permission to search user hierarchy group resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeUserHierarchyGroup
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchUserHierarchyGroups.html
     */
    toSearchUserHierarchyGroups(): this;
    /**
     * Grants permission to search user resources in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSearchTag()
     *
     * Dependent actions:
     * - connect:DescribeUser
     * - connect:ListUserProficiencies
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchUsers.html
     */
    toSearchUsers(): this;
    /**
     * Grants permission to search vocabularies in a Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchVocabularies.html
     */
    toSearchVocabularies(): this;
    /**
     * Grants permission to send chat integration events using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SendChatIntegrationEvent.html
     */
    toSendChatIntegrationEvent(): this;
    /**
     * Grants permission to send integration events using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/whatsapp-integration.html
     */
    toSendIntegrationEvent(): this;
    /**
     * Grants permission to send outbound email using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SendOutboundEmail.html
     */
    toSendOutboundEmail(): this;
    /**
     * Grants permission to start an attached file upload in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     * - .ifUserArn()
     *
     * Dependent actions:
     * - cases:CreateRelatedItem
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartAttachedFileUpload.html
     */
    toStartAttachedFileUpload(): this;
    /**
     * Grants permission to initiate a chat using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartChatContact.html
     */
    toStartChatContact(): this;
    /**
     * Grants permission to start an empty evaluation in the specified Amazon Connect instance, using the given evaluation form for the particular contact. The evaluation form version used for the contact evaluation corresponds to the currently activated version. If no version is activated for the evaluation form, the contact evaluation cannot be started
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactEvaluation.html
     */
    toStartContactEvaluation(): this;
    /**
     * Grants permission to start recording for the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactRecording.html
     */
    toStartContactRecording(): this;
    /**
     * Grants permission to start chat streaming using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactStreaming.html
     */
    toStartContactStreaming(): this;
    /**
     * Grants permission to initiate an inbound email using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartEmailContact.html
     */
    toStartEmailContact(): this;
    /**
     * Grants permission to enable forecasting, planning, and scheduling integration on an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/optimization-apis.html
     */
    toStartForecastingPlanningSchedulingIntegration(): this;
    /**
     * Grants permission to initiate an outbound chat using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifSubtype()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartOutboundChatContact.html
     */
    toStartOutboundChatContact(): this;
    /**
     * Grants permission to initiate an outbound email using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartOutboundEmailContact.html
     */
    toStartOutboundEmailContact(): this;
    /**
     * Grants permission to initiate outbound calls using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartOutboundVoiceContact.html
     */
    toStartOutboundVoiceContact(): this;
    /**
     * Grants permission to start screen sharing for contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartScreenSharing.html
     */
    toStartScreenSharing(): this;
    /**
     * Grants permission to initiate a task using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifAssignmentType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartTaskContact.html
     */
    toStartTaskContact(): this;
    /**
     * Grants permission to initiate a WebRTC contact using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartWebRTCContact.html
     */
    toStartWebRTCContact(): this;
    /**
     * Grants permission to stop contacts that were initiated using the Amazon Connect API. If you use this operation on an active contact the contact ends, even if the agent is active on a call with a customer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContact.html
     */
    toStopContact(): this;
    /**
     * Grants permission to stop recording for the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContactRecording.html
     */
    toStopContactRecording(): this;
    /**
     * Grants permission to stop chat streaming using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContactStreaming.html
     */
    toStopContactStreaming(): this;
    /**
     * Grants permission to disable forecasting, planning, and scheduling integration on an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/optimization-apis.html
     */
    toStopForecastingPlanningSchedulingIntegration(): this;
    /**
     * Grants permission to submit a contact evaluation in the specified Amazon Connect instance. Answers included in the request are merged with existing answers for the given evaluation. If no answers or notes are passed, the evaluation is submitted with the existing answers and notes. You can delete an answer or note by passing an empty object ({}) to the question identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SubmitContactEvaluation.html
     */
    toSubmitContactEvaluation(): this;
    /**
     * Grants permission to suspend recording for the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SuspendContactRecording.html
     */
    toSuspendContactRecording(): this;
    /**
     * Grants permission to tag a contact in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_TagContact.html
     */
    toTagContact(): this;
    /**
     * Grants permission to tag an Amazon Connect resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to transfer the contact to another queue or agent
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_TransferContact.html
     */
    toTransferContact(): this;
    /**
     * Grants permission to untag a contact in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UntagContact.html
     */
    toUntagContact(): this;
    /**
     * Grants permission to untag an Amazon Connect resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update agent status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateAgentStatus.html
     */
    toUpdateAgentStatus(): this;
    /**
     * Grants permission to update authentication profile resources in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateAuthenticationProfile.html
     */
    toUpdateAuthenticationProfile(): this;
    /**
     * Grants permission to update a contact in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContact.html
     */
    toUpdateContact(): this;
    /**
     * Grants permission to create or update the contact attributes associated with the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactAttributes.html
     */
    toUpdateContactAttributes(): this;
    /**
     * Grants permission to update details about a contact evaluation in the specified Amazon Connect instance. A contact evaluation must be in the draft state. Answers included in the request are merged with existing answers for the given evaluation. An answer or note can be deleted by passing an empty object ({}) to the question identifier
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactEvaluation.html
     */
    toUpdateContactEvaluation(): this;
    /**
     * Grants permission to update contact flow content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowContent.html
     */
    toUpdateContactFlowContent(): this;
    /**
     * Grants permission to update the metadata of a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowMetadata.html
     */
    toUpdateContactFlowMetadata(): this;
    /**
     * Grants permission to update contact flow module content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowModuleContent.html
     */
    toUpdateContactFlowModuleContent(): this;
    /**
     * Grants permission to update the metadata of a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowModuleMetadata.html
     */
    toUpdateContactFlowModuleMetadata(): this;
    /**
     * Grants permission to update the name and description of a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     * - .ifFlowType()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowName.html
     */
    toUpdateContactFlowName(): this;
    /**
     * Grants permission to update routing properties on a contact in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactRoutingData.html
     */
    toUpdateContactRoutingData(): this;
    /**
     * Grants permission to update the schedule of a contact that is already scheduled in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifContactAssociationId()
     * - .ifChannel()
     * - .ifUserArn()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactSchedule.html
     */
    toUpdateContactSchedule(): this;
    /**
     * Grants permission to update the metadata of an email address resource in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateEmailAddressMetadata.html
     */
    toUpdateEmailAddressMetadata(): this;
    /**
     * Grants permission to update details about a specific evaluation form version in the specified Amazon Connect instance. Question and section identifiers cannot be duplicated within the same evaluation form
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateEvaluationForm.html
     */
    toUpdateEvaluationForm(): this;
    /**
     * Grants permission to update hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateHoursOfOperation.html
     */
    toUpdateHoursOfOperation(): this;
    /**
     * Grants permission to update an hours of operation override in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateHoursOfOperationOverride.html
     */
    toUpdateHoursOfOperationOverride(): this;
    /**
     * Grants permission to update the attribute for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributeType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - logs:CreateLogGroup
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toUpdateInstanceAttribute(): this;
    /**
     * Grants permission to update the storage configuration for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - firehose:DescribeDeliveryStream
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kinesis:DescribeStream
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toUpdateInstanceStorageConfig(): this;
    /**
     * Grants permission to update and continue authentication for a specific contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateParticipantAuthentication.html
     */
    toUpdateParticipantAuthentication(): this;
    /**
     * Grants permission to update participant role configurations associated with a contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateParticipantRoleConfig.html
     */
    toUpdateParticipantRoleConfig(): this;
    /**
     * Grants permission to update phone number resources in an Amazon Connect instance or traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumber.html
     */
    toUpdatePhoneNumber(): this;
    /**
     * Grants permission to update the metadata of a phone number resource in an Amazon Connect instance or traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePhoneNumberMetadata.html
     */
    toUpdatePhoneNumberMetadata(): this;
    /**
     * Grants permission to update a predefined attribute in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePredefinedAttribute.html
     */
    toUpdatePredefinedAttribute(): this;
    /**
     * Grants permission to update a prompt's name, description, and Amazon S3 URI in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - kms:Decrypt
     * - s3:GetObject
     * - s3:GetObjectAcl
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatePrompt.html
     */
    toUpdatePrompt(): this;
    /**
     * Grants permission to update queue hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueHoursOfOperation.html
     */
    toUpdateQueueHoursOfOperation(): this;
    /**
     * Grants permission to update queue capacity in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueMaxContacts.html
     */
    toUpdateQueueMaxContacts(): this;
    /**
     * Grants permission to update a queue name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueName.html
     */
    toUpdateQueueName(): this;
    /**
     * Grants permission to update queue outbound caller config in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueOutboundCallerConfig.html
     */
    toUpdateQueueOutboundCallerConfig(): this;
    /**
     * Grants permission to update the outbound email configuration for a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueOutboundEmailConfig.html
     */
    toUpdateQueueOutboundEmailConfig(): this;
    /**
     * Grants permission to update queue status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueStatus.html
     */
    toUpdateQueueStatus(): this;
    /**
     * Grants permission to update the configuration of a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQuickConnectConfig.html
     */
    toUpdateQuickConnectConfig(): this;
    /**
     * Grants permission to update a quick connect name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQuickConnectName.html
     */
    toUpdateQuickConnectName(): this;
    /**
     * Grants permission to update a routing profile agent availability timer in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileAgentAvailabilityTimer.html
     */
    toUpdateRoutingProfileAgentAvailabilityTimer(): this;
    /**
     * Grants permission to update the concurrency in a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileConcurrency.html
     */
    toUpdateRoutingProfileConcurrency(): this;
    /**
     * Grants permission to update the outbound queue in a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileDefaultOutboundQueue.html
     */
    toUpdateRoutingProfileDefaultOutboundQueue(): this;
    /**
     * Grants permission to update a routing profile name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileName.html
     */
    toUpdateRoutingProfileName(): this;
    /**
     * Grants permission to update the queues in routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileQueues.html
     */
    toUpdateRoutingProfileQueues(): this;
    /**
     * Grants permission to update a rule for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRule.html
     */
    toUpdateRule(): this;
    /**
     * Grants permission to update a security profile group for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateSecurityProfile.html
     */
    toUpdateSecurityProfile(): this;
    /**
     * Grants permission to update task template belonging to a Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateTaskTemplate.html
     */
    toUpdateTaskTemplate(): this;
    /**
     * Grants permission to update traffic distribution for a traffic distribution group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateTrafficDistribution.html
     */
    toUpdateTrafficDistribution(): this;
    /**
     * Grants permission to update a hierarchy group for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchy.html
     */
    toUpdateUserHierarchy(): this;
    /**
     * Grants permission to update a user hierarchy group name in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchyGroupName.html
     */
    toUpdateUserHierarchyGroupName(): this;
    /**
     * Grants permission to update user hierarchy structure in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchyStructure.html
     */
    toUpdateUserHierarchyStructure(): this;
    /**
     * Grants permission to update identity information for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserIdentityInfo.html
     */
    toUpdateUserIdentityInfo(): this;
    /**
     * Grants permission to update phone configuration settings for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserPhoneConfig.html
     */
    toUpdateUserPhoneConfig(): this;
    /**
     * Grants permission to update user proficiencies from a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserProficiencies.html
     */
    toUpdateUserProficiencies(): this;
    /**
     * Grants permission to update a routing profile for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserRoutingProfile.html
     */
    toUpdateUserRoutingProfile(): this;
    /**
     * Grants permission to update security profiles for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserSecurityProfiles.html
     */
    toUpdateUserSecurityProfiles(): this;
    /**
     * Grants permission to update a view's content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateViewContent.html
     */
    toUpdateViewContent(): this;
    /**
     * Grants permission to update a view's metadata in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateViewMetadata.html
     */
    toUpdateViewMetadata(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(instanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-attributes.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactId - Identifier for the contactId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onContact(instanceId: string, contactId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-agents.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(instanceId: string, userId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type routing-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/routing-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param routingProfileId - Identifier for the routingProfileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoutingProfile(instanceId: string, routingProfileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type security-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-security-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param securityProfileId - Identifier for the securityProfileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecurityProfile(instanceId: string, securityProfileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type authentication-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-authentication-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param authenticationProfileId - Identifier for the authenticationProfileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAuthenticationProfile(instanceId: string, authenticationProfileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hierarchy-group to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/agent-hierarchy.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hierarchyGroupId - Identifier for the hierarchyGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHierarchyGroup(instanceId: string, hierarchyGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/create-queue.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param queueId - Identifier for the queueId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(instanceId: string, queueId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-queue to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/create-queue.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWildcardQueue(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type quick-connect to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/quick-connects.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param quickConnectId - Identifier for the quickConnectId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuickConnect(instanceId: string, quickConnectId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-quick-connect to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/quick-connects.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWildcardQuickConnect(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-flow to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-flows.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactFlowId - Identifier for the contactFlowId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactFlow(instanceId: string, contactFlowId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type task-template to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/task-templates.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param taskTemplateId - Identifier for the taskTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTaskTemplate(instanceId: string, taskTemplateId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-flow-module to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/contact-flow-modules.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactFlowModuleId - Identifier for the contactFlowModuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactFlowModule(instanceId: string, contactFlowModuleId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-contact-flow to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-flows.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWildcardContactFlow(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hours-of-operation to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/set-hours-operation.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hoursOfOperationId - Identifier for the hoursOfOperationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHoursOfOperation(instanceId: string, hoursOfOperationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type agent-status to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/metrics-agent-status.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param agentStatusId - Identifier for the agentStatusId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgentStatus(instanceId: string, agentStatusId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-agent-status to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/metrics-agent-status.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWildcardAgentStatus(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type legacy-phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/ag-overview-numbers.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param phoneNumberId - Identifier for the phoneNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLegacyPhoneNumber(instanceId: string, phoneNumberId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-legacy-phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/ag-overview-numbers.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWildcardLegacyPhoneNumber(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/ag-overview-numbers.html
     *
     * @param phoneNumberId - Identifier for the phoneNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPhoneNumber(phoneNumberId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/ag-overview-numbers.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWildcardPhoneNumber(resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type integration-association to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-rules.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param integrationAssociationId - Identifier for the integrationAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrationAssociation(instanceId: string, integrationAssociationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type use-case to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/add-rules-task-creation.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param useCaseId - Identifier for the useCaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUseCase(instanceId: string, useCaseId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type vocabulary to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/add-custom-vocabulary.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param vocabularyId - Identifier for the vocabularyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabulary(instanceId: string, vocabularyId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type traffic-distribution-group to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/setup-traffic-distribution-groups.html
     *
     * @param trafficDistributionGroupId - Identifier for the trafficDistributionGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrafficDistributionGroup(trafficDistributionGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-rules.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param ruleId - Identifier for the ruleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(instanceId: string, ruleId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type evaluation-form to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/create-evaluation-forms.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param formId - Identifier for the formId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEvaluationForm(instanceId: string, formId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-evaluation to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/evaluations.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param evaluationId - Identifier for the evaluationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactEvaluation(instanceId: string, evaluationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type prompt to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/prompts.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param promptId - Identifier for the promptId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPrompt(instanceId: string, promptId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type customer-managed-view to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/view-resources-sg.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param viewId - Identifier for the viewId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomerManagedView(instanceId: string, viewId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type aws-managed-view to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/view-resources-sg.html
     *
     * @param viewId - Identifier for the viewId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAwsManagedView(viewId: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type qualified-customer-managed-view to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/view-resources-sg.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param viewId - Identifier for the viewId.
     * @param viewQualifier - Identifier for the viewQualifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQualifiedCustomerManagedView(instanceId: string, viewId: string, viewQualifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type qualified-aws-managed-view to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/view-resources-sg.html
     *
     * @param viewId - Identifier for the viewId.
     * @param viewQualifier - Identifier for the viewQualifier.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onQualifiedAwsManagedView(viewId: string, viewQualifier: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type customer-managed-view-version to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/view-resources-sg.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param viewId - Identifier for the viewId.
     * @param viewVersion - Identifier for the viewVersion.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCustomerManagedViewVersion(instanceId: string, viewId: string, viewVersion: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type attached-file to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-attachments.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param fileId - Identifier for the fileId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAttachedFile(instanceId: string, fileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type email-address to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/email-address.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param emailAddressId - Identifier for the emailAddressId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEmailAddress(instanceId: string, emailAddressId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by using tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toBatchGetAttachedFileMetadata()
     * - .toClaimPhoneNumber()
     * - .toCompleteAttachedFileUpload()
     * - .toCreateAgentStatus()
     * - .toCreateContactFlow()
     * - .toCreateContactFlowModule()
     * - .toCreateEmailAddress()
     * - .toCreateHoursOfOperation()
     * - .toCreateInstance()
     * - .toCreateIntegrationAssociation()
     * - .toCreatePrompt()
     * - .toCreateQueue()
     * - .toCreateQuickConnect()
     * - .toCreateRoutingProfile()
     * - .toCreateSecurityProfile()
     * - .toCreateTrafficDistributionGroup()
     * - .toCreateUseCase()
     * - .toCreateUser()
     * - .toCreateUserHierarchyGroup()
     * - .toCreateView()
     * - .toCreateVocabulary()
     * - .toDeleteAttachedFile()
     * - .toGetAttachedFile()
     * - .toImportPhoneNumber()
     * - .toReplicateInstance()
     * - .toStartAttachedFileUpload()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by using tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateFlow()
     * - .toAssociatePhoneNumberContactFlow()
     * - .toAssociateQueueQuickConnects()
     * - .toAssociateRoutingProfileQueues()
     * - .toAssociateTrafficDistributionGroupUser()
     * - .toBatchGetFlowAssociation()
     * - .toCreateContact()
     * - .toCreateContactFlowVersion()
     * - .toCreateViewVersion()
     * - .toDeleteContactEvaluation()
     * - .toDeleteContactFlow()
     * - .toDeleteContactFlowModule()
     * - .toDeleteContactFlowVersion()
     * - .toDeleteEmailAddress()
     * - .toDeleteEvaluationForm()
     * - .toDeleteHoursOfOperation()
     * - .toDeleteInstance()
     * - .toDeletePrompt()
     * - .toDeleteQueue()
     * - .toDeleteQuickConnect()
     * - .toDeleteRoutingProfile()
     * - .toDeleteRule()
     * - .toDeleteSecurityProfile()
     * - .toDeleteTaskTemplate()
     * - .toDeleteTrafficDistributionGroup()
     * - .toDeleteUser()
     * - .toDeleteView()
     * - .toDeleteViewVersion()
     * - .toDeleteVocabulary()
     * - .toDescribeAgentStatus()
     * - .toDescribeContactEvaluation()
     * - .toDescribeContactFlow()
     * - .toDescribeContactFlowModule()
     * - .toDescribeEmailAddress()
     * - .toDescribeEvaluationForm()
     * - .toDescribeHoursOfOperation()
     * - .toDescribeInstance()
     * - .toDescribePhoneNumber()
     * - .toDescribePrompt()
     * - .toDescribeQueue()
     * - .toDescribeQuickConnect()
     * - .toDescribeRoutingProfile()
     * - .toDescribeRule()
     * - .toDescribeSecurityProfile()
     * - .toDescribeTrafficDistributionGroup()
     * - .toDescribeUser()
     * - .toDescribeView()
     * - .toDescribeVocabulary()
     * - .toDisassociateFlow()
     * - .toDisassociatePhoneNumberContactFlow()
     * - .toDisassociateQueueQuickConnects()
     * - .toDisassociateRoutingProfileQueues()
     * - .toDisassociateTrafficDistributionGroupUser()
     * - .toDismissUserContact()
     * - .toGetCurrentMetricData()
     * - .toGetCurrentUserData()
     * - .toGetFlowAssociation()
     * - .toGetMetricData()
     * - .toGetMetricDataV2()
     * - .toGetPromptFile()
     * - .toGetTaskTemplate()
     * - .toGetTrafficDistribution()
     * - .toListContactFlowVersions()
     * - .toListQueueQuickConnects()
     * - .toListRoutingProfileQueues()
     * - .toListSecurityProfileApplications()
     * - .toListSecurityProfilePermissions()
     * - .toListTagsForResource()
     * - .toListTrafficDistributionGroupUsers()
     * - .toListViewVersions()
     * - .toMonitorContact()
     * - .toPauseContact()
     * - .toPutUserStatus()
     * - .toReleasePhoneNumber()
     * - .toResumeContact()
     * - .toSearchResourceTags()
     * - .toStartEmailContact()
     * - .toStartOutboundEmailContact()
     * - .toStartTaskContact()
     * - .toUpdateAgentStatus()
     * - .toUpdateContactFlowContent()
     * - .toUpdateContactFlowMetadata()
     * - .toUpdateContactFlowModuleContent()
     * - .toUpdateContactFlowModuleMetadata()
     * - .toUpdateContactFlowName()
     * - .toUpdateEmailAddressMetadata()
     * - .toUpdateHoursOfOperation()
     * - .toUpdatePhoneNumber()
     * - .toUpdatePhoneNumberMetadata()
     * - .toUpdatePrompt()
     * - .toUpdateQueueHoursOfOperation()
     * - .toUpdateQueueMaxContacts()
     * - .toUpdateQueueName()
     * - .toUpdateQueueOutboundCallerConfig()
     * - .toUpdateQueueOutboundEmailConfig()
     * - .toUpdateQueueStatus()
     * - .toUpdateQuickConnectConfig()
     * - .toUpdateQuickConnectName()
     * - .toUpdateRoutingProfileAgentAvailabilityTimer()
     * - .toUpdateRoutingProfileConcurrency()
     * - .toUpdateRoutingProfileDefaultOutboundQueue()
     * - .toUpdateRoutingProfileName()
     * - .toUpdateRoutingProfileQueues()
     * - .toUpdateSecurityProfile()
     * - .toUpdateTaskTemplate()
     * - .toUpdateTrafficDistribution()
     * - .toUpdateUserHierarchy()
     * - .toUpdateUserIdentityInfo()
     * - .toUpdateUserPhoneConfig()
     * - .toUpdateUserRoutingProfile()
     * - .toUpdateUserSecurityProfiles()
     * - .toUpdateViewContent()
     * - .toUpdateViewMetadata()
     *
     * Applies to resource types:
     * - instance
     * - user
     * - routing-profile
     * - security-profile
     * - hierarchy-group
     * - queue
     * - quick-connect
     * - contact-flow
     * - task-template
     * - contact-flow-module
     * - hours-of-operation
     * - agent-status
     * - phone-number
     * - wildcard-phone-number
     * - integration-association
     * - use-case
     * - vocabulary
     * - traffic-distribution-group
     * - rule
     * - evaluation-form
     * - contact-evaluation
     * - prompt
     * - customer-managed-view
     * - qualified-customer-managed-view
     * - customer-managed-view-version
     * - attached-file
     * - email-address
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by using tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toBatchGetAttachedFileMetadata()
     * - .toClaimPhoneNumber()
     * - .toCompleteAttachedFileUpload()
     * - .toCreateAgentStatus()
     * - .toCreateContactFlow()
     * - .toCreateContactFlowModule()
     * - .toCreateEmailAddress()
     * - .toCreateHoursOfOperation()
     * - .toCreateInstance()
     * - .toCreateIntegrationAssociation()
     * - .toCreatePrompt()
     * - .toCreateQueue()
     * - .toCreateQuickConnect()
     * - .toCreateRoutingProfile()
     * - .toCreateSecurityProfile()
     * - .toCreateTrafficDistributionGroup()
     * - .toCreateUseCase()
     * - .toCreateUser()
     * - .toCreateUserHierarchyGroup()
     * - .toCreateView()
     * - .toCreateVocabulary()
     * - .toDeleteAttachedFile()
     * - .toGetAttachedFile()
     * - .toImportPhoneNumber()
     * - .toReplicateInstance()
     * - .toStartAttachedFileUpload()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting access to create contacts based on Assignment Type
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toStartTaskContact()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssignmentType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the attribute type of the Amazon Connect instance
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toDescribeInstanceAttribute()
     * - .toUpdateInstanceAttribute()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributeType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by Channel
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toDescribeContact()
     * - .toGetContactAttributes()
     * - .toListContactReferences()
     * - .toResumeContactRecording()
     * - .toStartContactRecording()
     * - .toStartScreenSharing()
     * - .toStopContact()
     * - .toStopContactRecording()
     * - .toSuspendContactRecording()
     * - .toTagContact()
     * - .toTransferContact()
     * - .toUntagContact()
     * - .toUpdateContact()
     * - .toUpdateContactAttributes()
     * - .toUpdateContactRoutingData()
     * - .toUpdateContactSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChannel(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by ContactAssociationId
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toDescribeContact()
     * - .toGetContactAttributes()
     * - .toListContactReferences()
     * - .toResumeContactRecording()
     * - .toStartContactRecording()
     * - .toStartScreenSharing()
     * - .toStopContact()
     * - .toStopContactRecording()
     * - .toSuspendContactRecording()
     * - .toTagContact()
     * - .toTransferContact()
     * - .toUntagContact()
     * - .toUpdateContact()
     * - .toUpdateContactAttributes()
     * - .toUpdateContactRoutingData()
     * - .toUpdateContactSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifContactAssociationId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting access to create contacts based on the initiation method of the contact
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateContact()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifContactInitiationMethod(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by Flow type
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toCreateContactFlow()
     * - .toCreateContactFlowVersion()
     * - .toDeleteContactFlow()
     * - .toDeleteContactFlowVersion()
     * - .toDescribeContactFlow()
     * - .toListContactFlowVersions()
     * - .toSearchContactFlows()
     * - .toUpdateContactFlowContent()
     * - .toUpdateContactFlowMetadata()
     * - .toUpdateContactFlowName()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFlowType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting federation into specified Amazon Connect instances
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toActivateEvaluationForm()
     * - .toAssociateAnalyticsDataSet()
     * - .toAssociateApprovedOrigin()
     * - .toAssociateBot()
     * - .toAssociateDefaultVocabulary()
     * - .toAssociateFlow()
     * - .toAssociateInstanceStorageConfig()
     * - .toAssociateLambdaFunction()
     * - .toAssociateLexBot()
     * - .toAssociatePhoneNumberContactFlow()
     * - .toAssociateQueueQuickConnects()
     * - .toAssociateRoutingProfileQueues()
     * - .toAssociateSecurityKey()
     * - .toAssociateTrafficDistributionGroupUser()
     * - .toAssociateUserProficiencies()
     * - .toBatchAssociateAnalyticsDataSet()
     * - .toBatchDisassociateAnalyticsDataSet()
     * - .toBatchGetAttachedFileMetadata()
     * - .toBatchGetFlowAssociation()
     * - .toBatchPutContact()
     * - .toClaimPhoneNumber()
     * - .toCompleteAttachedFileUpload()
     * - .toCreateAgentStatus()
     * - .toCreateAuthenticationProfile()
     * - .toCreateContact()
     * - .toCreateContactFlow()
     * - .toCreateContactFlowModule()
     * - .toCreateContactFlowVersion()
     * - .toCreateEmailAddress()
     * - .toCreateEvaluationForm()
     * - .toCreateHoursOfOperation()
     * - .toCreateHoursOfOperationOverride()
     * - .toCreateIntegrationAssociation()
     * - .toCreateParticipant()
     * - .toCreatePersistentContactAssociation()
     * - .toCreatePredefinedAttribute()
     * - .toCreatePrompt()
     * - .toCreatePushNotificationRegistration()
     * - .toCreateQueue()
     * - .toCreateQuickConnect()
     * - .toCreateRoutingProfile()
     * - .toCreateRule()
     * - .toCreateSecurityProfile()
     * - .toCreateTrafficDistributionGroup()
     * - .toCreateUseCase()
     * - .toCreateUser()
     * - .toCreateUserHierarchyGroup()
     * - .toCreateView()
     * - .toCreateViewVersion()
     * - .toCreateVocabulary()
     * - .toDeactivateEvaluationForm()
     * - .toDeleteAttachedFile()
     * - .toDeleteContactEvaluation()
     * - .toDeleteContactFlow()
     * - .toDeleteContactFlowModule()
     * - .toDeleteContactFlowVersion()
     * - .toDeleteEmailAddress()
     * - .toDeleteEvaluationForm()
     * - .toDeleteHoursOfOperation()
     * - .toDeleteHoursOfOperationOverride()
     * - .toDeleteInstance()
     * - .toDeleteIntegrationAssociation()
     * - .toDeletePredefinedAttribute()
     * - .toDeletePrompt()
     * - .toDeletePushNotificationRegistration()
     * - .toDeleteQueue()
     * - .toDeleteQuickConnect()
     * - .toDeleteRoutingProfile()
     * - .toDeleteRule()
     * - .toDeleteSecurityProfile()
     * - .toDeleteTaskTemplate()
     * - .toDeleteUseCase()
     * - .toDeleteUser()
     * - .toDeleteUserHierarchyGroup()
     * - .toDeleteView()
     * - .toDeleteViewVersion()
     * - .toDeleteVocabulary()
     * - .toDescribeAgentStatus()
     * - .toDescribeAuthenticationProfile()
     * - .toDescribeContact()
     * - .toDescribeContactEvaluation()
     * - .toDescribeContactFlow()
     * - .toDescribeContactFlowModule()
     * - .toDescribeEmailAddress()
     * - .toDescribeEvaluationForm()
     * - .toDescribeForecastingPlanningSchedulingIntegration()
     * - .toDescribeHoursOfOperation()
     * - .toDescribeHoursOfOperationOverride()
     * - .toDescribeInstance()
     * - .toDescribeInstanceAttribute()
     * - .toDescribeInstanceStorageConfig()
     * - .toDescribePredefinedAttribute()
     * - .toDescribePrompt()
     * - .toDescribeQueue()
     * - .toDescribeQuickConnect()
     * - .toDescribeRoutingProfile()
     * - .toDescribeRule()
     * - .toDescribeSecurityProfile()
     * - .toDescribeUser()
     * - .toDescribeUserHierarchyGroup()
     * - .toDescribeUserHierarchyStructure()
     * - .toDescribeView()
     * - .toDescribeVocabulary()
     * - .toDisassociateAnalyticsDataSet()
     * - .toDisassociateApprovedOrigin()
     * - .toDisassociateBot()
     * - .toDisassociateFlow()
     * - .toDisassociateInstanceStorageConfig()
     * - .toDisassociateLambdaFunction()
     * - .toDisassociateLexBot()
     * - .toDisassociatePhoneNumberContactFlow()
     * - .toDisassociateQueueQuickConnects()
     * - .toDisassociateRoutingProfileQueues()
     * - .toDisassociateSecurityKey()
     * - .toDisassociateTrafficDistributionGroupUser()
     * - .toDisassociateUserProficiencies()
     * - .toDismissUserContact()
     * - .toGetAttachedFile()
     * - .toGetContactAttributes()
     * - .toGetCurrentMetricData()
     * - .toGetCurrentUserData()
     * - .toGetEffectiveHoursOfOperations()
     * - .toGetFederationToken()
     * - .toGetFlowAssociation()
     * - .toGetMetricData()
     * - .toGetMetricDataV2()
     * - .toGetPromptFile()
     * - .toGetTaskTemplate()
     * - .toListAnalyticsDataAssociations()
     * - .toListAnalyticsDataLakeDataSets()
     * - .toListApprovedOrigins()
     * - .toListAssociatedContacts()
     * - .toListAuthenticationProfiles()
     * - .toListBots()
     * - .toListContactEvaluations()
     * - .toListContactFlowVersions()
     * - .toListContactFlows()
     * - .toListContactReferences()
     * - .toListDefaultVocabularies()
     * - .toListEvaluationFormVersions()
     * - .toListEvaluationForms()
     * - .toListFlowAssociations()
     * - .toListHoursOfOperationOverrides()
     * - .toListHoursOfOperations()
     * - .toListInstanceAttributes()
     * - .toListInstanceStorageConfigs()
     * - .toListIntegrationAssociations()
     * - .toListLambdaFunctions()
     * - .toListLexBots()
     * - .toListPredefinedAttributes()
     * - .toListPrompts()
     * - .toListQueueQuickConnects()
     * - .toListRoutingProfileQueues()
     * - .toListRoutingProfiles()
     * - .toListRules()
     * - .toListSecurityKeys()
     * - .toListSecurityProfileApplications()
     * - .toListSecurityProfilePermissions()
     * - .toListSecurityProfiles()
     * - .toListUseCases()
     * - .toListUserHierarchyGroups()
     * - .toListUserProficiencies()
     * - .toListUsers()
     * - .toListViewVersions()
     * - .toListViews()
     * - .toMonitorContact()
     * - .toPauseContact()
     * - .toPutUserStatus()
     * - .toReplicateInstance()
     * - .toResumeContact()
     * - .toSearchAgentStatuses()
     * - .toSearchContactFlowModules()
     * - .toSearchContactFlows()
     * - .toSearchContacts()
     * - .toSearchEmailAddresses()
     * - .toSearchHoursOfOperationOverrides()
     * - .toSearchHoursOfOperations()
     * - .toSearchPredefinedAttributes()
     * - .toSearchPrompts()
     * - .toSearchQueues()
     * - .toSearchQuickConnects()
     * - .toSearchResourceTags()
     * - .toSearchRoutingProfiles()
     * - .toSearchSecurityProfiles()
     * - .toSearchUserHierarchyGroups()
     * - .toSearchUsers()
     * - .toSearchVocabularies()
     * - .toSendOutboundEmail()
     * - .toStartAttachedFileUpload()
     * - .toStartChatContact()
     * - .toStartContactEvaluation()
     * - .toStartEmailContact()
     * - .toStartForecastingPlanningSchedulingIntegration()
     * - .toStartOutboundChatContact()
     * - .toStartOutboundEmailContact()
     * - .toStartScreenSharing()
     * - .toStartTaskContact()
     * - .toStartWebRTCContact()
     * - .toStopContact()
     * - .toStopForecastingPlanningSchedulingIntegration()
     * - .toSubmitContactEvaluation()
     * - .toTagContact()
     * - .toTransferContact()
     * - .toUntagContact()
     * - .toUpdateAgentStatus()
     * - .toUpdateAuthenticationProfile()
     * - .toUpdateContact()
     * - .toUpdateContactAttributes()
     * - .toUpdateContactEvaluation()
     * - .toUpdateContactFlowContent()
     * - .toUpdateContactFlowMetadata()
     * - .toUpdateContactFlowModuleContent()
     * - .toUpdateContactFlowModuleMetadata()
     * - .toUpdateContactFlowName()
     * - .toUpdateContactRoutingData()
     * - .toUpdateContactSchedule()
     * - .toUpdateEmailAddressMetadata()
     * - .toUpdateEvaluationForm()
     * - .toUpdateHoursOfOperation()
     * - .toUpdateHoursOfOperationOverride()
     * - .toUpdateInstanceAttribute()
     * - .toUpdateInstanceStorageConfig()
     * - .toUpdateParticipantAuthentication()
     * - .toUpdateParticipantRoleConfig()
     * - .toUpdatePhoneNumber()
     * - .toUpdatePredefinedAttribute()
     * - .toUpdatePrompt()
     * - .toUpdateQueueHoursOfOperation()
     * - .toUpdateQueueMaxContacts()
     * - .toUpdateQueueName()
     * - .toUpdateQueueOutboundCallerConfig()
     * - .toUpdateQueueOutboundEmailConfig()
     * - .toUpdateQueueStatus()
     * - .toUpdateQuickConnectConfig()
     * - .toUpdateQuickConnectName()
     * - .toUpdateRoutingProfileAgentAvailabilityTimer()
     * - .toUpdateRoutingProfileConcurrency()
     * - .toUpdateRoutingProfileDefaultOutboundQueue()
     * - .toUpdateRoutingProfileName()
     * - .toUpdateRoutingProfileQueues()
     * - .toUpdateRule()
     * - .toUpdateSecurityProfile()
     * - .toUpdateTaskTemplate()
     * - .toUpdateUserHierarchy()
     * - .toUpdateUserHierarchyGroupName()
     * - .toUpdateUserHierarchyStructure()
     * - .toUpdateUserIdentityInfo()
     * - .toUpdateUserPhoneConfig()
     * - .toUpdateUserProficiencies()
     * - .toUpdateUserRoutingProfile()
     * - .toUpdateUserSecurityProfiles()
     * - .toUpdateViewContent()
     * - .toUpdateViewMetadata()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting the monitor capabilities of the user in the request
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toMonitorContact()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMonitorCapabilities(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting searches using analysis outputs from Amazon Connect Contact Lens
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toSearchContacts()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSearchContactsByContactAnalysis(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by TagFilter condition passed in the search request
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateTrafficDistributionGroupUser()
     * - .toSearchAgentStatuses()
     * - .toSearchContactFlowModules()
     * - .toSearchContactFlows()
     * - .toSearchEmailAddresses()
     * - .toSearchHoursOfOperationOverrides()
     * - .toSearchHoursOfOperations()
     * - .toSearchPrompts()
     * - .toSearchQueues()
     * - .toSearchQuickConnects()
     * - .toSearchRoutingProfiles()
     * - .toSearchSecurityProfiles()
     * - .toSearchUserHierarchyGroups()
     * - .toSearchUsers()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSearchTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting the storage resource type of the Amazon Connect instance storage configuration
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateInstanceStorageConfig()
     * - .toDescribeInstanceStorageConfig()
     * - .toDisassociateInstanceStorageConfig()
     * - .toUpdateInstanceStorageConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStorageResourceType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting creation of a contact for specific subtypes
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toStartOutboundChatContact()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubtype(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by UserArn
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toDescribeContact()
     * - .toGetContactAttributes()
     * - .toListContactReferences()
     * - .toResumeContactRecording()
     * - .toStartAttachedFileUpload()
     * - .toStartContactRecording()
     * - .toStartScreenSharing()
     * - .toStopContact()
     * - .toStopContactRecording()
     * - .toSuspendContactRecording()
     * - .toTagContact()
     * - .toTransferContact()
     * - .toUntagContact()
     * - .toUpdateContact()
     * - .toUpdateContactAttributes()
     * - .toUpdateContactRoutingData()
     * - .toUpdateContactSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUserArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
