"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Controlcatalog = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [controlcatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscontrolcatalog.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Controlcatalog extends shared_1.PolicyStatement {
    /**
     * Grants permission to return details about a specific control
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_GetControl.html
     */
    toGetControl() {
        return this.to('GetControl');
    }
    /**
     * Grants permission to return a paginated list of common controls from the AWS Control Catalog
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ListCommonControls.html
     */
    toListCommonControls() {
        return this.to('ListCommonControls');
    }
    /**
     * Grants permission to return a paginated list of all available controls in the AWS Control Catalog library
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ListControls.html
     */
    toListControls() {
        return this.to('ListControls');
    }
    /**
     * Grants permission to return a paginated list of domains from the AWS Control Catalog
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to return a paginated list of objectives from the AWS Control Catalog
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ListObjectives.html
     */
    toListObjectives() {
        return this.to('ListObjectives');
    }
    /**
     * Adds a resource of type common-control to the statement
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_CommonControlSummary.html
     *
     * @param commonControlId - Identifier for the commonControlId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCommonControl(commonControlId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controlcatalog:::common-control/${commonControlId}`);
    }
    /**
     * Adds a resource of type control to the statement
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ControlSummary.html
     *
     * @param controlId - Identifier for the controlId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onControl(controlId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controlcatalog:::control/${controlId}`);
    }
    /**
     * Adds a resource of type domain to the statement
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_DomainSummary.html
     *
     * @param domainId - Identifier for the domainId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDomain(domainId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controlcatalog:::domain/${domainId}`);
    }
    /**
     * Adds a resource of type objective to the statement
     *
     * https://docs.aws.amazon.com/controlcatalog/latest/APIReference/API_ObjectiveSummary.html
     *
     * @param objectiveId - Identifier for the objectiveId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObjective(objectiveId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:controlcatalog:::objective/${objectiveId}`);
    }
    /**
     * Statement provider for service [controlcatalog](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscontrolcatalog.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'controlcatalog';
        this.accessLevelList = {
            Read: [
                'GetControl'
            ],
            List: [
                'ListCommonControls',
                'ListControls',
                'ListDomains',
                'ListObjectives'
            ]
        };
    }
}
exports.Controlcatalog = Controlcatalog;
//# sourceMappingURL=data:application/json;base64,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