"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dms = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dms extends shared_1.PolicyStatement {
    /**
     * Grants permission to add metadata tags to DMS resources, including replication instances, endpoints, security groups, and migration tasks
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to apply a pending maintenance action to a resource (for example, to a replication instance)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ApplyPendingMaintenanceAction.html
     */
    toApplyPendingMaintenanceAction() {
        return this.to('ApplyPendingMaintenanceAction');
    }
    /**
     * Grants permission to associate a extension pack
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartExtensionPackAssociation.html
     */
    toAssociateExtensionPack() {
        return this.to('AssociateExtensionPack');
    }
    /**
     * Grants permission to start the analysis of up to 20 source databases to recommend target engines for each source database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_BatchStartRecommendations.html
     */
    toBatchStartRecommendations() {
        return this.to('BatchStartRecommendations');
    }
    /**
     * Grants permission to cancel a single premigration assessment run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CancelReplicationTaskAssessmentRun.html
     */
    toCancelReplicationTaskAssessmentRun() {
        return this.to('CancelReplicationTaskAssessmentRun');
    }
    /**
     * Grants permission to create a database migration using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toCreateDataMigration() {
        return this.to('CreateDataMigration');
    }
    /**
     * Grants permission to create an data provider using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateDataProvider.html
     */
    toCreateDataProvider() {
        return this.to('CreateDataProvider');
    }
    /**
     * Grants permission to create an endpoint using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        return this.to('CreateEndpoint');
    }
    /**
     * Grants permission to create an AWS DMS event notification subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateEventSubscription.html
     */
    toCreateEventSubscription() {
        return this.to('CreateEventSubscription');
    }
    /**
     * Grants permission to create a Fleet Advisor collector using the specified parameters
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateFleetAdvisorCollector.html
     */
    toCreateFleetAdvisorCollector() {
        return this.to('CreateFleetAdvisorCollector');
    }
    /**
     * Grants permission to create an instance profile using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateInstanceProfile.html
     */
    toCreateInstanceProfile() {
        return this.to('CreateInstanceProfile');
    }
    /**
     * Grants permission to create an migration project using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateMigrationProject.html
     */
    toCreateMigrationProject() {
        return this.to('CreateMigrationProject');
    }
    /**
     * Grants permission to create a replication config using the provided settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toCreateReplicationConfig() {
        return this.to('CreateReplicationConfig');
    }
    /**
     * Grants permission to create a replication instance using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationInstance.html
     */
    toCreateReplicationInstance() {
        return this.to('CreateReplicationInstance');
    }
    /**
     * Grants permission to create a replication subnet group given a list of the subnet IDs in a VPC
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationSubnetGroup.html
     */
    toCreateReplicationSubnetGroup() {
        return this.to('CreateReplicationSubnetGroup');
    }
    /**
     * Grants permission to create a replication task using the specified parameters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifReqTag()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_CreateReplicationTask.html
     */
    toCreateReplicationTask() {
        return this.to('CreateReplicationTask');
    }
    /**
     * Grants permission to delete the specified certificate
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteCertificate.html
     */
    toDeleteCertificate() {
        return this.to('DeleteCertificate');
    }
    /**
     * Grants permission to delete the specified connection between a replication instance and an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete the specified database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDeleteDataMigration() {
        return this.to('DeleteDataMigration');
    }
    /**
     * Grants permission to delete the specified data provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteDataProvider.html
     */
    toDeleteDataProvider() {
        return this.to('DeleteDataProvider');
    }
    /**
     * Grants permission to delete the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete an AWS DMS event subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteEventSubscription.html
     */
    toDeleteEventSubscription() {
        return this.to('DeleteEventSubscription');
    }
    /**
     * Grants permission to delete the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteFleetAdvisorCollector.html
     */
    toDeleteFleetAdvisorCollector() {
        return this.to('DeleteFleetAdvisorCollector');
    }
    /**
     * Grants permission to delete the specified Fleet Advisor databases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteFleetAdvisorDatabases.html
     */
    toDeleteFleetAdvisorDatabases() {
        return this.to('DeleteFleetAdvisorDatabases');
    }
    /**
     * Grants permission to delete the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteInstanceProfile.html
     */
    toDeleteInstanceProfile() {
        return this.to('DeleteInstanceProfile');
    }
    /**
     * Grants permission to delete the specified migration project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteMigrationProject.html
     */
    toDeleteMigrationProject() {
        return this.to('DeleteMigrationProject');
    }
    /**
     * Grants permission to delete the specified replication config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDeleteReplicationConfig() {
        return this.to('DeleteReplicationConfig');
    }
    /**
     * Grants permission to delete the specified replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationInstance.html
     */
    toDeleteReplicationInstance() {
        return this.to('DeleteReplicationInstance');
    }
    /**
     * Grants permission to deletes a subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationSubnetGroup.html
     */
    toDeleteReplicationSubnetGroup() {
        return this.to('DeleteReplicationSubnetGroup');
    }
    /**
     * Grants permission to delete the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTask.html
     */
    toDeleteReplicationTask() {
        return this.to('DeleteReplicationTask');
    }
    /**
     * Grants permission to delete the record of a single premigration assessment run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DeleteReplicationTaskAssessmentRun.html
     */
    toDeleteReplicationTaskAssessmentRun() {
        return this.to('DeleteReplicationTaskAssessmentRun');
    }
    /**
     * Grants permission to list all of the AWS DMS attributes for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        return this.to('DescribeAccountAttributes');
    }
    /**
     * Grants permission to list individual assessments that you can specify for a new premigration assessment run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeApplicableIndividualAssessments.html
     */
    toDescribeApplicableIndividualAssessments() {
        return this.to('DescribeApplicableIndividualAssessments');
    }
    /**
     * Grants permission to provide a description of the certificate
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeCertificates.html
     */
    toDescribeCertificates() {
        return this.to('DescribeCertificates');
    }
    /**
     * Grants permission to describe the status of the connections that have been made between the replication instance and an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConnections.html
     */
    toDescribeConnections() {
        return this.to('DescribeConnections');
    }
    /**
     * Grants permission to return information about DMS Schema Conversion project configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeConversionConfiguration.html
     */
    toDescribeConversionConfiguration() {
        return this.to('DescribeConversionConfiguration');
    }
    /**
     * Grants permission to return information about database migrations for your account in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeDataMigrations() {
        return this.to('DescribeDataMigrations');
    }
    /**
     * Grants permission to return the possible endpoint settings available when you create an endpoint for a specific database engine
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointSettings.html
     */
    toDescribeEndpointSettings() {
        return this.to('DescribeEndpointSettings');
    }
    /**
     * Grants permission to return information about the type of endpoints available
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpointTypes.html
     */
    toDescribeEndpointTypes() {
        return this.to('DescribeEndpointTypes');
    }
    /**
     * Grants permission to return information about the endpoints for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        return this.to('DescribeEndpoints');
    }
    /**
     * Grants permission to return information about the available versions for DMS replication instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEngineVersions.html
     */
    toDescribeEngineVersions() {
        return this.to('DescribeEngineVersions');
    }
    /**
     * Grants permission to list categories for all event source types, or, if specified, for a specified source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventCategories.html
     */
    toDescribeEventCategories() {
        return this.to('DescribeEventCategories');
    }
    /**
     * Grants permission to list all the event subscriptions for a customer account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEventSubscriptions.html
     */
    toDescribeEventSubscriptions() {
        return this.to('DescribeEventSubscriptions');
    }
    /**
     * Grants permission to list events for a given source identifier and source type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to return a paginated list of Fleet Advisor collectors in your account based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorCollectors.html
     */
    toDescribeFleetAdvisorCollectors() {
        return this.to('DescribeFleetAdvisorCollectors');
    }
    /**
     * Grants permission to return a paginated list of Fleet Advisor databases in your account based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorDatabases.html
     */
    toDescribeFleetAdvisorDatabases() {
        return this.to('DescribeFleetAdvisorDatabases');
    }
    /**
     * Grants permission to return a paginated list of descriptions of large-scale assessment (LSA) analyses produced by your Fleet Advisor collectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorLsaAnalysis.html
     */
    toDescribeFleetAdvisorLsaAnalysis() {
        return this.to('DescribeFleetAdvisorLsaAnalysis');
    }
    /**
     * Grants permission to return a paginated list of descriptions of schemas discovered by your Fleet Advisor collectors based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorSchemaObjectSummary.html
     */
    toDescribeFleetAdvisorSchemaObjectSummary() {
        return this.to('DescribeFleetAdvisorSchemaObjectSummary');
    }
    /**
     * Grants permission to return a paginated list of schemas discovered by your Fleet Advisor collectors based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorSchemas.html
     */
    toDescribeFleetAdvisorSchemas() {
        return this.to('DescribeFleetAdvisorSchemas');
    }
    /**
     * Grants permission to return information about start metadata model import operations for a migration project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeMetadataModelImports.html
     */
    toDescribeMetadataModelImports() {
        return this.to('DescribeMetadataModelImports');
    }
    /**
     * Grants permission to return information about the replication instance types that can be created in the specified region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeOrderableReplicationInstances.html
     */
    toDescribeOrderableReplicationInstances() {
        return this.to('DescribeOrderableReplicationInstances');
    }
    /**
     * Grants permission to return information about pending maintenance actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribePendingMaintenanceActions.html
     */
    toDescribePendingMaintenanceActions() {
        return this.to('DescribePendingMaintenanceActions');
    }
    /**
     * Grants permission to return a paginated list of descriptions of limitations for recommendations of target AWS engines
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeFleetAdvisorLsaAnalysis.html
     */
    toDescribeRecommendationLimitations() {
        return this.to('DescribeRecommendationLimitations');
    }
    /**
     * Grants permission to return a paginated list of descriptions of target engine recommendations for your source databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRecommendations.html
     */
    toDescribeRecommendations() {
        return this.to('DescribeRecommendations');
    }
    /**
     * Grants permission to returns the status of the RefreshSchemas operation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeRefreshSchemasStatus.html
     */
    toDescribeRefreshSchemasStatus() {
        return this.to('DescribeRefreshSchemasStatus');
    }
    /**
     * Grants permission to describe replication configs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplicationConfigs() {
        return this.to('DescribeReplicationConfigs');
    }
    /**
     * Grants permission to return information about the task logs for the specified task
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstanceTaskLogs.html
     */
    toDescribeReplicationInstanceTaskLogs() {
        return this.to('DescribeReplicationInstanceTaskLogs');
    }
    /**
     * Grants permission to return information about replication instances for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationInstances.html
     */
    toDescribeReplicationInstances() {
        return this.to('DescribeReplicationInstances');
    }
    /**
     * Grants permission to return information about the replication subnet groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationSubnetGroups.html
     */
    toDescribeReplicationSubnetGroups() {
        return this.to('DescribeReplicationSubnetGroups');
    }
    /**
     * Grants permission to describe replication table statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplicationTableStatistics() {
        return this.to('DescribeReplicationTableStatistics');
    }
    /**
     * Grants permission to return the latest task assessment results from Amazon S3
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentResults.html
     */
    toDescribeReplicationTaskAssessmentResults() {
        return this.to('DescribeReplicationTaskAssessmentResults');
    }
    /**
     * Grants permission to return a paginated list of premigration assessment runs based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskAssessmentRuns.html
     */
    toDescribeReplicationTaskAssessmentRuns() {
        return this.to('DescribeReplicationTaskAssessmentRuns');
    }
    /**
     * Grants permission to return a paginated list of individual assessments based on filter settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTaskIndividualAssessments.html
     */
    toDescribeReplicationTaskIndividualAssessments() {
        return this.to('DescribeReplicationTaskIndividualAssessments');
    }
    /**
     * Grants permission to return information about replication tasks for your account in the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeReplicationTasks.html
     */
    toDescribeReplicationTasks() {
        return this.to('DescribeReplicationTasks');
    }
    /**
     * Grants permission to describe replications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toDescribeReplications() {
        return this.to('DescribeReplications');
    }
    /**
     * Grants permission to return information about the schema for the specified endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeSchemas.html
     */
    toDescribeSchemas() {
        return this.to('DescribeSchemas');
    }
    /**
     * Grants permission to return table statistics on the database migration task, including table name, rows inserted, rows updated, and rows deleted
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeTableStatistics.html
     */
    toDescribeTableStatistics() {
        return this.to('DescribeTableStatistics');
    }
    /**
     * Grants permission to export the specified metadata model assessment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ExportMetadataModelAssessment.html
     */
    toExportMetadataModelAssessment() {
        return this.to('ExportMetadataModelAssessment');
    }
    /**
     * Grants permission to upload the specified certificate
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ImportCertificate.html
     */
    toImportCertificate() {
        return this.to('ImportCertificate');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a data providers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeDataProviders.html
     */
    toListDataProviders() {
        return this.to('ListDataProviders');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a extension packs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeExtensionPackAssociations.html
     */
    toListExtensionPacks() {
        return this.to('ListExtensionPacks');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a instance profiles
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeInstanceProfiles.html
     */
    toListInstanceProfiles() {
        return this.to('ListInstanceProfiles');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model assessment action items. Note. Despite this action requires StartMetadataModelImport, the latter does not currently authorize the described Schema Conversion operation
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - dms:StartMetadataModelImport
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMetadataModelAssessmentActionItems() {
        return this.to('ListMetadataModelAssessmentActionItems');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model assessments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListMetadataModelExports.html
     */
    toListMetadataModelAssessments() {
        return this.to('ListMetadataModelAssessments');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model conversions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_DescribeMetadataModelConversions.html
     */
    toListMetadataModelConversions() {
        return this.to('ListMetadataModelConversions');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a metadata model exports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMetadataModelExports() {
        return this.to('ListMetadataModelExports');
    }
    /**
     * Grants permission to list the AWS DMS attributes for a migration projects. Note. Despite this action requires DescribeMigrationProjects and DescribeConversionConfiguration, both required actions do not currently authorize the described Schema Conversion operation
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - dms:DescribeConversionConfiguration
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toListMigrationProjects() {
        return this.to('ListMigrationProjects');
    }
    /**
     * Grants permission to list all tags for an AWS DMS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to modify the specified database migration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyDataMigration() {
        return this.to('ModifyDataMigration');
    }
    /**
     * Grants permission to modify the specified endpoint
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEndpoint.html
     */
    toModifyEndpoint() {
        return this.to('ModifyEndpoint');
    }
    /**
     * Grants permission to modify an existing AWS DMS event notification subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyEventSubscription.html
     */
    toModifyEventSubscription() {
        return this.to('ModifyEventSubscription');
    }
    /**
     * Grants permission to modify the name and description of the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyFleetAdvisorCollector() {
        return this.to('ModifyFleetAdvisorCollector');
    }
    /**
     * Grants permission to modify the status of the specified Fleet Advisor collector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyFleetAdvisorCollectorStatuses() {
        return this.to('ModifyFleetAdvisorCollectorStatuses');
    }
    /**
     * Grants permission to modify the specified replication config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toModifyReplicationConfig() {
        return this.to('ModifyReplicationConfig');
    }
    /**
     * Grants permission to modify the replication instance to apply new settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationInstance.html
     */
    toModifyReplicationInstance() {
        return this.to('ModifyReplicationInstance');
    }
    /**
     * Grants permission to modify the settings for the specified replication subnet group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationSubnetGroup.html
     */
    toModifyReplicationSubnetGroup() {
        return this.to('ModifyReplicationSubnetGroup');
    }
    /**
     * Grants permission to modify the specified replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyReplicationTask.html
     */
    toModifyReplicationTask() {
        return this.to('ModifyReplicationTask');
    }
    /**
     * Grants permission to move the specified replication task to a different replication instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_MoveReplicationTask.html
     */
    toMoveReplicationTask() {
        return this.to('MoveReplicationTask');
    }
    /**
     * Grants permission to reboot a replication instance. Rebooting results in a momentary outage, until the replication instance becomes available again
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RebootReplicationInstance.html
     */
    toRebootReplicationInstance() {
        return this.to('RebootReplicationInstance');
    }
    /**
     * Grants permission to populate the schema for the specified endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RefreshSchemas.html
     */
    toRefreshSchemas() {
        return this.to('RefreshSchemas');
    }
    /**
     * Grants permission to reload the target database table with the source for a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toReloadReplicationTables() {
        return this.to('ReloadReplicationTables');
    }
    /**
     * Grants permission to reload the target database table with the source data
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReloadTables.html
     */
    toReloadTables() {
        return this.to('ReloadTables');
    }
    /**
     * Grants permission to remove metadata tags from a DMS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to run a large-scale assessment (LSA) analysis on every Fleet Advisor collector in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_RunFleetAdvisorLsaAnalysis.html
     */
    toRunFleetAdvisorLsaAnalysis() {
        return this.to('RunFleetAdvisorLsaAnalysis');
    }
    /**
     * Grants permission to start the database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStartDataMigration() {
        return this.to('StartDataMigration');
    }
    /**
     * Grants permission to start a new assessment of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelAssessment.html
     */
    toStartMetadataModelAssessment() {
        return this.to('StartMetadataModelAssessment');
    }
    /**
     * Grants permission to start a new conversion of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelConversion.html
     */
    toStartMetadataModelConversion() {
        return this.to('StartMetadataModelConversion');
    }
    /**
     * Grants permission to start a new export of metadata model as script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelExportAsScript.html
     */
    toStartMetadataModelExportAsScripts() {
        return this.to('StartMetadataModelExportAsScripts');
    }
    /**
     * Grants permission to start a new export of metadata model to target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelExportToTarget.html
     */
    toStartMetadataModelExportToTarget() {
        return this.to('StartMetadataModelExportToTarget');
    }
    /**
     * Grants permission to start a new import of metadata model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartMetadataModelImport.html
     */
    toStartMetadataModelImport() {
        return this.to('StartMetadataModelImport');
    }
    /**
     * Grants permission to start the analysis of your source database to provide recommendations of target engines
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartRecommendations.html
     */
    toStartRecommendations() {
        return this.to('StartRecommendations');
    }
    /**
     * Grants permission to start a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStartReplication() {
        return this.to('StartReplication');
    }
    /**
     * Grants permission to start the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTask.html
     */
    toStartReplicationTask() {
        return this.to('StartReplicationTask');
    }
    /**
     * Grants permission to start the replication task assessment for unsupported data types in the source database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessment.html
     */
    toStartReplicationTaskAssessment() {
        return this.to('StartReplicationTaskAssessment');
    }
    /**
     * Grants permission to start a new premigration assessment run for one or more individual assessments of a migration task
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StartReplicationTaskAssessmentRun.html
     */
    toStartReplicationTaskAssessmentRun() {
        return this.to('StartReplicationTaskAssessmentRun');
    }
    /**
     * Grants permission to stop the database migration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStopDataMigration() {
        return this.to('StopDataMigration');
    }
    /**
     * Grants permission to stop a replication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toStopReplication() {
        return this.to('StopReplication');
    }
    /**
     * Grants permission to stop the replication task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_StopReplicationTask.html
     */
    toStopReplicationTask() {
        return this.to('StopReplicationTask');
    }
    /**
     * Grants permission to test the connection between the replication instance and the endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_TestConnection.html
     */
    toTestConnection() {
        return this.to('TestConnection');
    }
    /**
     * Grants permission to update a conversion configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyConversionConfiguration.html
     */
    toUpdateConversionConfiguration() {
        return this.to('UpdateConversionConfiguration');
    }
    /**
     * Grants permission to update the specified data provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyDataProvider.html
     */
    toUpdateDataProvider() {
        return this.to('UpdateDataProvider');
    }
    /**
     * Grants permission to update the specified instance profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyInstanceProfile.html
     */
    toUpdateInstanceProfile() {
        return this.to('UpdateInstanceProfile');
    }
    /**
     * Grants permission to update the specified migration project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ModifyMigrationProject.html
     */
    toUpdateMigrationProject() {
        return this.to('UpdateMigrationProject');
    }
    /**
     * Grants permission to migrate DMS subcriptions to Eventbridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_UpdateSubscriptionsToEventBridge.html
     */
    toUpdateSubscriptionsToEventBridge() {
        return this.to('UpdateSubscriptionsToEventBridge');
    }
    /**
     * Grants permission to upload files to your Amazon S3 bucket
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     */
    toUploadFileMetadataList() {
        return this.to('UploadFileMetadataList');
    }
    /**
     * Adds a resource of type Certificate to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Certificate.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCertTag()
     */
    onCertificate(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cert:${resourceName}`);
    }
    /**
     * Adds a resource of type DataProvider to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDataProviderTag()
     */
    onDataProvider(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-provider:${resourceName}`);
    }
    /**
     * Adds a resource of type DataMigration to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifDataMigrationTag()
     */
    onDataMigration(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-migration:${resourceName}`);
    }
    /**
     * Adds a resource of type Endpoint to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_Endpoint.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEndpointTag()
     */
    onEndpoint(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:endpoint:${resourceName}`);
    }
    /**
     * Adds a resource of type EventSubscription to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_EventSubscription.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifEsTag()
     */
    onEventSubscription(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:es:${resourceName}`);
    }
    /**
     * Adds a resource of type InstanceProfile to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceProfileTag()
     */
    onInstanceProfile(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance-profile:${resourceName}`);
    }
    /**
     * Adds a resource of type MigrationProject to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifMigrationProjectTag()
     */
    onMigrationProject(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:migration-project:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationConfig to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/Welcome.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicationConfigTag()
     */
    onReplicationConfig(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:replication-config:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationInstance to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationInstance.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifRepTag()
     */
    onReplicationInstance(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rep:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationSubnetGroup to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationSubnetGroup.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifSubgrpTag()
     */
    onReplicationSubnetGroup(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subgrp:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationTask to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTask.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTaskTag()
     */
    onReplicationTask(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:task:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationTaskAssessmentRun to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTaskAssessmentRun.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAssessmentRunTag()
     */
    onReplicationTaskAssessmentRun(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assessment-run:${resourceName}`);
    }
    /**
     * Adds a resource of type ReplicationTaskIndividualAssessment to the statement
     *
     * https://docs.aws.amazon.com/dms/latest/APIReference/API_ReplicationTaskIndividualAssessment.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifIndividualAssessmentTag()
     */
    onReplicationTaskIndividualAssessment(resourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:individual-assessment:${resourceName}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toDescribeReplicationInstanceTaskLogs()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * Applies to resource types:
     * - Certificate
     * - DataProvider
     * - DataMigration
     * - Endpoint
     * - EventSubscription
     * - InstanceProfile
     * - MigrationProject
     * - ReplicationConfig
     * - ReplicationInstance
     * - ReplicationSubnetGroup
     * - ReplicationTask
     * - ReplicationTaskAssessmentRun
     * - ReplicationTaskIndividualAssessment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     * - .toDescribeReplicationInstanceTaskLogs()
     * - .toImportCertificate()
     * - .toRemoveTagsFromResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for AssessmentRun
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_assessment-run-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTaskAssessmentRun
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssessmentRunTag(tagKey, value, operator) {
        return this.if(`assessment-run-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for Certificate
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_cert-tag___TagKey_
     *
     * Applies to resource types:
     * - Certificate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCertTag(tagKey, value, operator) {
        return this.if(`cert-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for DataMigration
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_data-migration-tag___TagKey_
     *
     * Applies to resource types:
     * - DataMigration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataMigrationTag(tagKey, value, operator) {
        return this.if(`data-migration-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for DataProvider
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_dp-tag___TagKey_
     *
     * Applies to resource types:
     * - DataProvider
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataProviderTag(tagKey, value, operator) {
        return this.if(`data-provider-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for Endpoint
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_endpoint-tag___TagKey_
     *
     * Applies to resource types:
     * - Endpoint
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpointTag(tagKey, value, operator) {
        return this.if(`endpoint-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for EventSubscription
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_es-tag___TagKey_
     *
     * Applies to resource types:
     * - EventSubscription
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEsTag(tagKey, value, operator) {
        return this.if(`es-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for IndividualAssessment
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_individual-assessment-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTaskIndividualAssessment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIndividualAssessmentTag(tagKey, value, operator) {
        return this.if(`individual-assessment-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for InstanceProfile
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_ip-tag___TagKey_
     *
     * Applies to resource types:
     * - InstanceProfile
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceProfileTag(tagKey, value, operator) {
        return this.if(`instance-profile-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for MigrationProject
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice--dms_mp-tag___TagKey_
     *
     * Applies to resource types:
     * - MigrationProject
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMigrationProjectTag(tagKey, value, operator) {
        return this.if(`migration-project-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationInstance
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_rep-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationInstance
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepTag(tagKey, value, operator) {
        return this.if(`rep-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationConfig
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_replication-config-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationConfig
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicationConfigTag(tagKey, value, operator) {
        return this.if(`replication-config-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the given request
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_req-tag___TagKey_
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCreateDataMigration()
     * - .toCreateDataProvider()
     * - .toCreateEndpoint()
     * - .toCreateEventSubscription()
     * - .toCreateInstanceProfile()
     * - .toCreateMigrationProject()
     * - .toCreateReplicationConfig()
     * - .toCreateReplicationInstance()
     * - .toCreateReplicationSubnetGroup()
     * - .toCreateReplicationTask()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReqTag(tagKey, value, operator) {
        return this.if(`req-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationSubnetGroup
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_subgrp-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationSubnetGroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubgrpTag(tagKey, value, operator) {
        return this.if(`subgrp-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request for ReplicationTask
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html#awsdatabasemigrationservice-dms_task-tag___TagKey_
     *
     * Applies to resource types:
     * - ReplicationTask
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTaskTag(tagKey, value, operator) {
        return this.if(`task-tag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatabasemigrationservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dms';
        this.accessLevelList = {
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            Write: [
                'ApplyPendingMaintenanceAction',
                'AssociateExtensionPack',
                'BatchStartRecommendations',
                'CancelReplicationTaskAssessmentRun',
                'CreateDataMigration',
                'CreateDataProvider',
                'CreateEndpoint',
                'CreateEventSubscription',
                'CreateFleetAdvisorCollector',
                'CreateInstanceProfile',
                'CreateMigrationProject',
                'CreateReplicationConfig',
                'CreateReplicationInstance',
                'CreateReplicationSubnetGroup',
                'CreateReplicationTask',
                'DeleteCertificate',
                'DeleteConnection',
                'DeleteDataMigration',
                'DeleteDataProvider',
                'DeleteEndpoint',
                'DeleteEventSubscription',
                'DeleteFleetAdvisorCollector',
                'DeleteFleetAdvisorDatabases',
                'DeleteInstanceProfile',
                'DeleteMigrationProject',
                'DeleteReplicationConfig',
                'DeleteReplicationInstance',
                'DeleteReplicationSubnetGroup',
                'DeleteReplicationTask',
                'DeleteReplicationTaskAssessmentRun',
                'ExportMetadataModelAssessment',
                'ImportCertificate',
                'ModifyDataMigration',
                'ModifyEndpoint',
                'ModifyEventSubscription',
                'ModifyFleetAdvisorCollector',
                'ModifyFleetAdvisorCollectorStatuses',
                'ModifyReplicationConfig',
                'ModifyReplicationInstance',
                'ModifyReplicationSubnetGroup',
                'ModifyReplicationTask',
                'MoveReplicationTask',
                'RebootReplicationInstance',
                'RefreshSchemas',
                'ReloadReplicationTables',
                'ReloadTables',
                'RunFleetAdvisorLsaAnalysis',
                'StartDataMigration',
                'StartMetadataModelAssessment',
                'StartMetadataModelConversion',
                'StartMetadataModelExportAsScripts',
                'StartMetadataModelExportToTarget',
                'StartMetadataModelImport',
                'StartRecommendations',
                'StartReplication',
                'StartReplicationTask',
                'StartReplicationTaskAssessment',
                'StartReplicationTaskAssessmentRun',
                'StopDataMigration',
                'StopReplication',
                'StopReplicationTask',
                'UpdateConversionConfiguration',
                'UpdateDataProvider',
                'UpdateInstanceProfile',
                'UpdateMigrationProject',
                'UpdateSubscriptionsToEventBridge',
                'UploadFileMetadataList'
            ],
            Read: [
                'DescribeAccountAttributes',
                'DescribeApplicableIndividualAssessments',
                'DescribeCertificates',
                'DescribeConnections',
                'DescribeConversionConfiguration',
                'DescribeDataMigrations',
                'DescribeEndpointSettings',
                'DescribeEndpointTypes',
                'DescribeEndpoints',
                'DescribeEngineVersions',
                'DescribeEventCategories',
                'DescribeEventSubscriptions',
                'DescribeEvents',
                'DescribeFleetAdvisorCollectors',
                'DescribeFleetAdvisorDatabases',
                'DescribeFleetAdvisorLsaAnalysis',
                'DescribeFleetAdvisorSchemaObjectSummary',
                'DescribeFleetAdvisorSchemas',
                'DescribeMetadataModelImports',
                'DescribeOrderableReplicationInstances',
                'DescribePendingMaintenanceActions',
                'DescribeRecommendationLimitations',
                'DescribeRecommendations',
                'DescribeRefreshSchemasStatus',
                'DescribeReplicationConfigs',
                'DescribeReplicationInstanceTaskLogs',
                'DescribeReplicationInstances',
                'DescribeReplicationSubnetGroups',
                'DescribeReplicationTableStatistics',
                'DescribeReplicationTaskAssessmentResults',
                'DescribeReplicationTaskAssessmentRuns',
                'DescribeReplicationTaskIndividualAssessments',
                'DescribeReplicationTasks',
                'DescribeReplications',
                'DescribeSchemas',
                'DescribeTableStatistics',
                'ListDataProviders',
                'ListExtensionPacks',
                'ListInstanceProfiles',
                'ListMetadataModelAssessmentActionItems',
                'ListMetadataModelAssessments',
                'ListMetadataModelConversions',
                'ListMetadataModelExports',
                'ListMigrationProjects',
                'ListTagsForResource',
                'TestConnection'
            ]
        };
    }
}
exports.Dms = Dms;
//# sourceMappingURL=data:application/json;base64,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