import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [deadline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeadlinecloud.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Deadline extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to associate a member to a farm
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToFarm.html
     */
    toAssociateMemberToFarm(): this;
    /**
     * Grants permission to associate a member to a fleet
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToFleet.html
     */
    toAssociateMemberToFleet(): this;
    /**
     * Grants permission to associate a member to a job
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToJob.html
     */
    toAssociateMemberToJob(): this;
    /**
     * Grants permission to associate a member to a queue
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToQueue.html
     */
    toAssociateMemberToQueue(): this;
    /**
     * Grants permission to assume a fleet role for read-only access
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeFleetRoleForRead.html
     */
    toAssumeFleetRoleForRead(): this;
    /**
     * Grants permission to assume a fleet role for a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeFleetRoleForWorker.html
     */
    toAssumeFleetRoleForWorker(): this;
    /**
     * Grants permission to assume a queue role for read-only access
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForRead.html
     */
    toAssumeQueueRoleForRead(): this;
    /**
     * Grants permission to assume a queue role for a user
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForUser.html
     */
    toAssumeQueueRoleForUser(): this;
    /**
     * Grants permission to assume a queue role for a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForWorker.html
     */
    toAssumeQueueRoleForWorker(): this;
    /**
     * Grants permission to get a job entity for a worker
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_BatchGetJobEntity.html
     */
    toBatchGetJobEntity(): this;
    /**
     * Grants permission to copy a job template to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CopyJobTemplate.html
     */
    toCopyJobTemplate(): this;
    /**
     * Grants permission to create a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateBudget.html
     */
    toCreateBudget(): this;
    /**
     * Grants permission to create a farm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateFarm.html
     */
    toCreateFarm(): this;
    /**
     * Grants permission to create a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     * - logs:CreateLogGroup
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet(): this;
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - deadline:GetJobTemplate
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateJob.html
     */
    toCreateJob(): this;
    /**
     * Grants permission to create a license endpoint for licensed software or products
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateLicenseEndpoint.html
     */
    toCreateLicenseEndpoint(): this;
    /**
     * Grants permission to create a limit for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateLimit.html
     */
    toCreateLimit(): this;
    /**
     * Grants permission to create a monitor
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - sso:CreateApplication
     * - sso:DeleteApplication
     * - sso:PutApplicationAssignmentConfiguration
     * - sso:PutApplicationAuthenticationMethod
     * - sso:PutApplicationGrant
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateMonitor.html
     */
    toCreateMonitor(): this;
    /**
     * Grants permission to create a queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     * - logs:CreateLogGroup
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue(): this;
    /**
     * Grants permission to create a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueEnvironment.html
     */
    toCreateQueueEnvironment(): this;
    /**
     * Grants permission to create a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueFleetAssociation.html
     */
    toCreateQueueFleetAssociation(): this;
    /**
     * Grants permission to create a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueLimitAssociation.html
     */
    toCreateQueueLimitAssociation(): this;
    /**
     * Grants permission to create a storage profile for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateStorageProfile.html
     */
    toCreateStorageProfile(): this;
    /**
     * Grants permission to create a worker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:ListTagsForResource
     * - deadline:TagResource
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateWorker.html
     */
    toCreateWorker(): this;
    /**
     * Grants permission to delete a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteBudget.html
     */
    toDeleteBudget(): this;
    /**
     * Grants permission to delete a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteFarm.html
     */
    toDeleteFarm(): this;
    /**
     * Grants permission to delete a fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet(): this;
    /**
     * Grants permission to delete a license endpoint
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteLicenseEndpoint.html
     */
    toDeleteLicenseEndpoint(): this;
    /**
     * Grants permission to delete a limit
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteLimit.html
     */
    toDeleteLimit(): this;
    /**
     * Grants permission to delete a metered product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteMeteredProduct.html
     */
    toDeleteMeteredProduct(): this;
    /**
     * Grants permission to delete a monitor
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteApplication
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteMonitor.html
     */
    toDeleteMonitor(): this;
    /**
     * Grants permission to delete a queue
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue(): this;
    /**
     * Grants permission to delete a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueEnvironment.html
     */
    toDeleteQueueEnvironment(): this;
    /**
     * Grants permission to delete a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueFleetAssociation.html
     */
    toDeleteQueueFleetAssociation(): this;
    /**
     * Grants permission to delete a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueLimitAssociation.html
     */
    toDeleteQueueLimitAssociation(): this;
    /**
     * Grants permission to delete a storage profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteStorageProfile.html
     */
    toDeleteStorageProfile(): this;
    /**
     * Grants permission to delete a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteWorker.html
     */
    toDeleteWorker(): this;
    /**
     * Grants permission to disassociate a member from a farm
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromFarm.html
     */
    toDisassociateMemberFromFarm(): this;
    /**
     * Grants permission to disassociate a member from a fleet
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromFleet.html
     */
    toDisassociateMemberFromFleet(): this;
    /**
     * Grants permission to disassociate a member from a job
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromJob.html
     */
    toDisassociateMemberFromJob(): this;
    /**
     * Grants permission to disassociate a member from a queue
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromQueue.html
     */
    toDisassociateMemberFromQueue(): this;
    /**
     * Grants permission to get the latest version of an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetApplicationVersion.html
     */
    toGetApplicationVersion(): this;
    /**
     * Grants permission to get a budget
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetBudget.html
     */
    toGetBudget(): this;
    /**
     * Grants permission to get a farm
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetFarm.html
     */
    toGetFarm(): this;
    /**
     * Grants permission to get a fleet
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetFleet.html
     */
    toGetFleet(): this;
    /**
     * Grants permission to get a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetJob.html
     */
    toGetJob(): this;
    /**
     * Grants permission to read job template
     *
     * Access Level: Read
     */
    toGetJobTemplate(): this;
    /**
     * Grants permission to get a license endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetLicenseEndpoint.html
     */
    toGetLicenseEndpoint(): this;
    /**
     * Grants permission to get a limit
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetLimit.html
     */
    toGetLimit(): this;
    /**
     * Grants permission to get a monitor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetMonitor.html
     */
    toGetMonitor(): this;
    /**
     * Grants permission to get a queue
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueue.html
     */
    toGetQueue(): this;
    /**
     * Grants permission to get a queue environment
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueEnvironment.html
     */
    toGetQueueEnvironment(): this;
    /**
     * Grants permission to get a queue-fleet association
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueFleetAssociation.html
     */
    toGetQueueFleetAssociation(): this;
    /**
     * Grants permission to get a queue-limit association
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueLimitAssociation.html
     */
    toGetQueueLimitAssociation(): this;
    /**
     * Grants permission to get a session for a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSession.html
     */
    toGetSession(): this;
    /**
     * Grants permission to get a session action for a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSessionAction.html
     */
    toGetSessionAction(): this;
    /**
     * Grants permission to get all collected statistics for sessions
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSessionsStatisticsAggregation.html
     */
    toGetSessionsStatisticsAggregation(): this;
    /**
     * Grants permission to get a step in a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStep.html
     */
    toGetStep(): this;
    /**
     * Grants permission to get a storage profile
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStorageProfile.html
     */
    toGetStorageProfile(): this;
    /**
     * Grants permission to get a storage profile for a queue
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStorageProfileForQueue.html
     */
    toGetStorageProfileForQueue(): this;
    /**
     * Grants permission to get a job task
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetTask.html
     */
    toGetTask(): this;
    /**
     * Grants permission to get a worker
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetWorker.html
     */
    toGetWorker(): this;
    /**
     * Grants permission to list all available metered products within a license endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListAvailableMeteredProducts.html
     */
    toListAvailableMeteredProducts(): this;
    /**
     * Grants permission to list all budgets for a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListBudgets.html
     */
    toListBudgets(): this;
    /**
     * Grants permission to list all members of a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFarmMembers.html
     */
    toListFarmMembers(): this;
    /**
     * Grants permission to list all farms
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFarms.html
     */
    toListFarms(): this;
    /**
     * Grants permission to list all members of a fleet
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFleetMembers.html
     */
    toListFleetMembers(): this;
    /**
     * Grants permission to list all fleets
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFleets.html
     */
    toListFleets(): this;
    /**
     * Grants permission to list all members of a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobMembers.html
     */
    toListJobMembers(): this;
    /**
     * Grants permission to get a job's parameter definitions in the job template
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobParameterDefinitions.html
     */
    toListJobParameterDefinitions(): this;
    /**
     * Grants permission to list all jobs in a queue
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobs.html
     */
    toListJobs(): this;
    /**
     * Grants permission to list all license endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListLicenseEndpoints.html
     */
    toListLicenseEndpoints(): this;
    /**
     * Grants permission to list all limits in a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListLimits.html
     */
    toListLimits(): this;
    /**
     * Grants permission to list all metered products in a license endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListMeteredProducts.html
     */
    toListMeteredProducts(): this;
    /**
     * Grants permission to list all monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListMonitors.html
     */
    toListMonitors(): this;
    /**
     * Grants permission to list all queue environments to which a queue is associated
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueEnvironments.html
     */
    toListQueueEnvironments(): this;
    /**
     * Grants permission to list all queue-fleet associations
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueFleetAssociations.html
     */
    toListQueueFleetAssociations(): this;
    /**
     * Grants permission to list all queue-limit associations
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueLimitAssociations.html
     */
    toListQueueLimitAssociations(): this;
    /**
     * Grants permission to list all members in a queue
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueMembers.html
     */
    toListQueueMembers(): this;
    /**
     * Grants permission to list all queues on a farm
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueues.html
     */
    toListQueues(): this;
    /**
     * Grants permission to list all session actions for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessionActions.html
     */
    toListSessionActions(): this;
    /**
     * Grants permission to list all sessions for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessions.html
     */
    toListSessions(): this;
    /**
     * Grants permission to list all sessions for a worker
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessionsForWorker.html
     */
    toListSessionsForWorker(): this;
    /**
     * Grants permission to list the step consumers for a job step
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStepConsumers.html
     */
    toListStepConsumers(): this;
    /**
     * Grants permission to list dependencies for a job step
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStepDependencies.html
     */
    toListStepDependencies(): this;
    /**
     * Grants permission to list all steps for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSteps.html
     */
    toListSteps(): this;
    /**
     * Grants permission to list all storage profiles in a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStorageProfiles.html
     */
    toListStorageProfiles(): this;
    /**
     * Grants permission to list all storage profiles in a queue
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStorageProfilesForQueue.html
     */
    toListStorageProfilesForQueue(): this;
    /**
     * Grants permission to list all tags on specified Deadline Cloud resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCalledAction()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list all tasks for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListTasks.html
     */
    toListTasks(): this;
    /**
     * Grants permission to list all workers in a fleet
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListWorkers.html
     */
    toListWorkers(): this;
    /**
     * Grants permission to add a metered product to a license endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_PutMeteredProduct.html
     */
    toPutMeteredProduct(): this;
    /**
     * Grants permission to search for jobs in multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchJobs.html
     */
    toSearchJobs(): this;
    /**
     * Grants permission to search the steps within a single job or to search the steps for multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchSteps.html
     */
    toSearchSteps(): this;
    /**
     * Grants permission to search the tasks within a single job or to search the tasks for multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchTasks.html
     */
    toSearchTasks(): this;
    /**
     * Grants permission to search for workers in multiple fleets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchWorkers.html
     */
    toSearchWorkers(): this;
    /**
     * Grants permission to get all collected statistics for sessions
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_StartSessionsStatisticsAggregation.html
     */
    toStartSessionsStatisticsAggregation(): this;
    /**
     * Grants permission to add or overwrite one or more tags for the specified Deadline Cloud resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCalledAction()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to disassociate one or more tags from the specified Deadline Cloud resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateBudget.html
     */
    toUpdateBudget(): this;
    /**
     * Grants permission to update a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateFarm.html
     */
    toUpdateFarm(): this;
    /**
     * Grants permission to update a fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet(): this;
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateJob.html
     */
    toUpdateJob(): this;
    /**
     * Grants permission to update a limit for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateLimit.html
     */
    toUpdateLimit(): this;
    /**
     * Grants permission to update a monitor
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - sso:PutApplicationGrant
     * - sso:UpdateApplication
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateMonitor.html
     */
    toUpdateMonitor(): this;
    /**
     * Grants permission to update a queue
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueue.html
     */
    toUpdateQueue(): this;
    /**
     * Grants permission to update a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueEnvironment.html
     */
    toUpdateQueueEnvironment(): this;
    /**
     * Grants permission to update a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueFleetAssociation.html
     */
    toUpdateQueueFleetAssociation(): this;
    /**
     * Grants permission to update a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueLimitAssociation.html
     */
    toUpdateQueueLimitAssociation(): this;
    /**
     * Grants permission to update a session for a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateSession.html
     */
    toUpdateSession(): this;
    /**
     * Grants permission to update a step for a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateStep.html
     */
    toUpdateStep(): this;
    /**
     * Grants permission to update a storage profile for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateStorageProfile.html
     */
    toUpdateStorageProfile(): this;
    /**
     * Grants permission to update a task
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateTask.html
     */
    toUpdateTask(): this;
    /**
     * Grants permission to update a worker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogStream
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateWorker.html
     */
    toUpdateWorker(): this;
    /**
     * Grants permission to update the schedule for a worker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogStream
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateWorkerSchedule.html
     */
    toUpdateWorkerSchedule(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type budget to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/manage-costs.html
     *
     * @param farmId - Identifier for the farmId.
     * @param budgetId - Identifier for the budgetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifFarmMembershipLevels()
     */
    onBudget(farmId: string, budgetId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type farm to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/farms.html
     *
     * @param farmId - Identifier for the farmId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     */
    onFarm(farmId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/manage-fleets.html
     *
     * @param farmId - Identifier for the farmId.
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifFleetMembershipLevels()
     */
    onFleet(farmId: string, fleetId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/deadline-cloud-jobs.html
     *
     * @param farmId - Identifier for the farmId.
     * @param queueId - Identifier for the queueId.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifFarmMembershipLevels()
     * - .ifJobMembershipLevels()
     * - .ifQueueMembershipLevels()
     */
    onJob(farmId: string, queueId: string, jobId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type license-endpoint to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/cmf-ubl.html
     *
     * @param licenseEndpointId - Identifier for the licenseEndpointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLicenseEndpoint(licenseEndpointId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type metered-product to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/cmf-ubl.html
     *
     * @param licenseEndpointId - Identifier for the licenseEndpointId.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMeteredProduct(licenseEndpointId: string, productId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type monitor to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/working-with-deadline-monitor.html
     *
     * @param monitorId - Identifier for the monitorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMonitor(monitorId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/queues.html
     *
     * @param farmId - Identifier for the farmId.
     * @param queueId - Identifier for the queueId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifQueueMembershipLevels()
     */
    onQueue(farmId: string, queueId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type worker to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam.html
     *
     * @param farmId - Identifier for the farmId.
     * @param fleetId - Identifier for the fleetId.
     * @param workerId - Identifier for the workerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifFleetMembershipLevels()
     */
    onWorker(farmId: string, fleetId: string, workerId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFarm()
     * - .toCreateFleet()
     * - .toCreateLicenseEndpoint()
     * - .toCreateQueue()
     * - .toCreateWorker()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - farm
     * - fleet
     * - license-endpoint
     * - queue
     * - worker
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFarm()
     * - .toCreateFleet()
     * - .toCreateLicenseEndpoint()
     * - .toCreateQueue()
     * - .toCreateWorker()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the associated membership level of the principal provided in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateMemberToFarm()
     * - .toAssociateMemberToFleet()
     * - .toAssociateMemberToJob()
     * - .toAssociateMemberToQueue()
     * - .toDisassociateMemberFromFarm()
     * - .toDisassociateMemberFromFleet()
     * - .toDisassociateMemberFromJob()
     * - .toDisassociateMemberFromQueue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedMembershipLevel(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the allowed action in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCalledAction(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by membership levels on the farm
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - budget
     * - farm
     * - fleet
     * - job
     * - queue
     * - worker
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFarmMembershipLevels(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by membership levels on the fleet
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - fleet
     * - worker
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetMembershipLevels(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by membership levels on the job
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - job
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobMembershipLevels(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the membership level passed in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateMemberToFarm()
     * - .toAssociateMemberToFleet()
     * - .toAssociateMemberToJob()
     * - .toAssociateMemberToQueue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMembershipLevel(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the principle ID provided in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListFarms()
     * - .toListFleets()
     * - .toListJobs()
     * - .toListQueues()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipalId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by membership levels on the queue
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - job
     * - queue
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifQueueMembershipLevels(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the user calling the Deadline Cloud API
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListFarms()
     * - .toListFleets()
     * - .toListJobs()
     * - .toListQueues()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequesterPrincipalId(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [deadline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeadlinecloud.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
