"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ds = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a directory sharing request that was sent from the directory owner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    toAcceptSharedDirectory() {
        return this.to('AcceptSharedDirectory');
    }
    /**
     * Grants permission to access directory data using the Directory Service Data API
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toAccessDSData() {
        return this.to('AccessDSData');
    }
    /**
     * Grants permission to add a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:DescribeSecurityGroups
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    toAddIpRoutes() {
        return this.to('AddIpRoutes');
    }
    /**
     * Grants permission to add two domain controllers in the specified Region for the specified directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddRegion.html
     */
    toAddRegion() {
        return this.to('AddRegion');
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified Amazon Directory Services directory
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to authorize an application for your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toAuthorizeApplication() {
        return this.to('AuthorizeApplication');
    }
    /**
     * Grants permission to cancel an in-progress schema extension to a Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    toCancelSchemaExtension() {
        return this.to('CancelSchemaExtension');
    }
    /**
     * Grants permission to verify that the alias is available for use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toCheckAlias() {
        return this.to('CheckAlias');
    }
    /**
     * Grants permission to create an AD Connector to connect to an on-premises directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    toConnectDirectory() {
        return this.to('ConnectDirectory');
    }
    /**
     * Grants permission to create an alias for a directory and assigns the alias to the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create a computer account in the specified directory, and joins the computer to the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    toCreateComputer() {
        return this.to('CreateComputer');
    }
    /**
     * Grants permission to create a conditional forwarder associated with your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    toCreateConditionalForwarder() {
        return this.to('CreateConditionalForwarder');
    }
    /**
     * Grants permission to create a Simple AD directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    toCreateDirectory() {
        return this.to('CreateDirectory');
    }
    /**
     * Grants permission to create an IdentityPool Directory in the AWS cloud
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toCreateIdentityPoolDirectory() {
        return this.to('CreateIdentityPoolDirectory');
    }
    /**
     * Grants permission to create a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    toCreateLogSubscription() {
        return this.to('CreateLogSubscription');
    }
    /**
     * Grants permission to create a Microsoft AD in the AWS cloud
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    toCreateMicrosoftAD() {
        return this.to('CreateMicrosoftAD');
    }
    /**
     * Grants permission to create a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to initiate the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    toCreateTrust() {
        return this.to('CreateTrust');
    }
    /**
     * Grants permission to delete a conditional forwarder that has been set up for your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    toDeleteConditionalForwarder() {
        return this.to('DeleteConditionalForwarder');
    }
    /**
     * Grants permission to delete an AWS Directory Service directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteSecurityGroup
     * - ec2:DescribeNetworkInterfaces
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RevokeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        return this.to('DeleteDirectory');
    }
    /**
     * Grants permission to delete the specified log subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    toDeleteLogSubscription() {
        return this.to('DeleteLogSubscription');
    }
    /**
     * Grants permission to delete a directory snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteTrust.html
     */
    toDeleteTrust() {
        return this.to('DeleteTrust');
    }
    /**
     * Grants permission to delete from the system the certificate that was registered for a secured LDAP connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    toDeregisterCertificate() {
        return this.to('DeregisterCertificate');
    }
    /**
     * Grants permission to remove the specified directory as a publisher to the specified SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    toDeregisterEventTopic() {
        return this.to('DeregisterEventTopic');
    }
    /**
     * Grants permission to display information about the certificate registered for a secured LDAP connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        return this.to('DescribeCertificate');
    }
    /**
     * Grants permission to retrieve information about the type of client authentication for the specified directory, if the type is specified. If no type is specified, information about all client authentication types that are supported for the specified directory is retrieved. Currently, only SmartCard is supported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeClientAuthenticationSettings.html
     */
    toDescribeClientAuthenticationSettings() {
        return this.to('DescribeClientAuthenticationSettings');
    }
    /**
     * Grants permission to obtain information about the conditional forwarders for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    toDescribeConditionalForwarders() {
        return this.to('DescribeConditionalForwarders');
    }
    /**
     * Grants permission to obtain information about the directories that belong to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    toDescribeDirectories() {
        return this.to('DescribeDirectories');
    }
    /**
     * Grants permission to describe the Directory Service Data API status for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectoryDataAccess.html
     */
    toDescribeDirectoryDataAccess() {
        return this.to('DescribeDirectoryDataAccess');
    }
    /**
     * Grants permission to provide information about any domain controllers in your directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    toDescribeDomainControllers() {
        return this.to('DescribeDomainControllers');
    }
    /**
     * Grants permission to obtain information about which SNS topics receive status messages from the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    toDescribeEventTopics() {
        return this.to('DescribeEventTopics');
    }
    /**
     * Grants permission to describe the status of LDAP security for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    toDescribeLDAPSSettings() {
        return this.to('DescribeLDAPSSettings');
    }
    /**
     * Grants permission to provide information about the Regions that are configured for multi-Region replication
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeRegions.html
     */
    toDescribeRegions() {
        return this.to('DescribeRegions');
    }
    /**
     * Grants permission to retrieve information about the configurable settings for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSettings.html
     */
    toDescribeSettings() {
        return this.to('DescribeSettings');
    }
    /**
     * Grants permission to return the shared directories in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    toDescribeSharedDirectories() {
        return this.to('DescribeSharedDirectories');
    }
    /**
     * Grants permission to obtain information about the directory snapshots that belong to this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permission to obtain information about the trust relationships for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    toDescribeTrusts() {
        return this.to('DescribeTrusts');
    }
    /**
     * Grants permission to describe the updates of a directory for a particular update type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeUpdateDirectory.html
     */
    toDescribeUpdateDirectory() {
        return this.to('DescribeUpdateDirectory');
    }
    /**
     * Grants permission to disable alternative client authentication methods for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableClientAuthentication.html
     */
    toDisableClientAuthentication() {
        return this.to('DisableClientAuthentication');
    }
    /**
     * Grants permission to disable the Directory Service Data API for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableDirectoryDataAccess.html
     */
    toDisableDirectoryDataAccess() {
        return this.to('DisableDirectoryDataAccess');
    }
    /**
     * Grants permission to deactivate LDAP secure calls for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    toDisableLDAPS() {
        return this.to('DisableLDAPS');
    }
    /**
     * Grants permission to disable multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    toDisableRadius() {
        return this.to('DisableRadius');
    }
    /**
     * Grants permission to disable AWS Management Console access for identity in your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toDisableRoleAccess() {
        return this.to('DisableRoleAccess');
    }
    /**
     * Grants permission to disable single-sign on for a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    toDisableSso() {
        return this.to('DisableSso');
    }
    /**
     * Grants permission to enable alternative client authentication methods for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableClientAuthentication.html
     */
    toEnableClientAuthentication() {
        return this.to('EnableClientAuthentication');
    }
    /**
     * Grants permission to enable the Directory Service Data API for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableDirectoryDataAccess.html
     */
    toEnableDirectoryDataAccess() {
        return this.to('EnableDirectoryDataAccess');
    }
    /**
     * Grants permission to activate the switch for the specific directory to always use LDAP secure calls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    toEnableLDAPS() {
        return this.to('EnableLDAPS');
    }
    /**
     * Grants permission to enable multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    toEnableRadius() {
        return this.to('EnableRadius');
    }
    /**
     * Grants permission to enable AWS Management Console access for identity in your AWS Directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toEnableRoleAccess() {
        return this.to('EnableRoleAccess');
    }
    /**
     * Grants permission to enable single-sign on for a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    toEnableSso() {
        return this.to('EnableSso');
    }
    /**
     * Grants permission to retrieve the details of the authorized applications on a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toGetAuthorizedApplicationDetails() {
        return this.to('GetAuthorizedApplicationDetails');
    }
    /**
     * Grants permission to obtain directory limit information for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    toGetDirectoryLimits() {
        return this.to('GetDirectoryLimits');
    }
    /**
     * Grants permission to obtain the manual snapshot limits for a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    toGetSnapshotLimits() {
        return this.to('GetSnapshotLimits');
    }
    /**
     * Grants permission to obtain the AWS applications authorized for a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toListAuthorizedApplications() {
        return this.to('ListAuthorizedApplications');
    }
    /**
     * Grants permission to list all the certificates registered for a secured LDAP connection, for the specified directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    toListCertificates() {
        return this.to('ListCertificates');
    }
    /**
     * Grants permission to list the address blocks that you have added to a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    toListIpRoutes() {
        return this.to('ListIpRoutes');
    }
    /**
     * Grants permission to list the active log subscriptions for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    toListLogSubscriptions() {
        return this.to('ListLogSubscriptions');
    }
    /**
     * Grants permission to list all schema extensions applied to a Microsoft AD Directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    toListSchemaExtensions() {
        return this.to('ListSchemaExtensions');
    }
    /**
     * Grants permission to list all tags on an Amazon Directory Services directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register a certificate for secured LDAP connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    toRegisterCertificate() {
        return this.to('RegisterCertificate');
    }
    /**
     * Grants permission to associate a directory with an SNS topic
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    toRegisterEventTopic() {
        return this.to('RegisterEventTopic');
    }
    /**
     * Grants permission to reject a directory sharing request that was sent from the directory owner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    toRejectSharedDirectory() {
        return this.to('RejectSharedDirectory');
    }
    /**
     * Grants permission to remove IP address blocks from a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    toRemoveIpRoutes() {
        return this.to('RemoveIpRoutes');
    }
    /**
     * Grants permission to stop all replication and removes the domain controllers from the specified Region. You cannot remove the primary Region with this operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveRegion.html
     */
    toRemoveRegion() {
        return this.to('RemoveRegion');
    }
    /**
     * Grants permission to remove tags from an Amazon Directory Services directory
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DeleteTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to reset the password for any user in your AWS Managed Microsoft AD or Simple AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    toResetUserPassword() {
        return this.to('ResetUserPassword');
    }
    /**
     * Grants permission to restore a directory using an existing directory snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    toRestoreFromSnapshot() {
        return this.to('RestoreFromSnapshot');
    }
    /**
     * Grants permission to share a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    toShareDirectory() {
        return this.to('ShareDirectory');
    }
    /**
     * Grants permission to apply a schema extension to a Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    toStartSchemaExtension() {
        return this.to('StartSchemaExtension');
    }
    /**
     * Grants permission to unauthorize an application from your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUnauthorizeApplication() {
        return this.to('UnauthorizeApplication');
    }
    /**
     * Grants permission to stop the directory sharing between the directory owner and consumer accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    toUnshareDirectory() {
        return this.to('UnshareDirectory');
    }
    /**
     * Grants permission to update an authorized application for your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUpdateAuthorizedApplication() {
        return this.to('UpdateAuthorizedApplication');
    }
    /**
     * Grants permission to update a conditional forwarder that has been set up for your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    toUpdateConditionalForwarder() {
        return this.to('UpdateConditionalForwarder');
    }
    /**
     * Grants permission to update the configurations like service account credentials or DNS server IP addresses for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUpdateDirectory() {
        return this.to('UpdateDirectory');
    }
    /**
     * Grants permission to update the directory for a particular update type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateDirectorySetup.html
     */
    toUpdateDirectorySetup() {
        return this.to('UpdateDirectorySetup');
    }
    /**
     * Grants permission to add or remove domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    toUpdateNumberOfDomainControllers() {
        return this.to('UpdateNumberOfDomainControllers');
    }
    /**
     * Grants permission to update the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    toUpdateRadius() {
        return this.to('UpdateRadius');
    }
    /**
     * Grants permission to update the configurable settings for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateSettings.html
     */
    toUpdateSettings() {
        return this.to('UpdateSettings');
    }
    /**
     * Grants permission to update the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    toUpdateTrust() {
        return this.to('UpdateTrust');
    }
    /**
     * Grants permission to verify a trust relationship between your Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    toVerifyTrust() {
        return this.to('VerifyTrust');
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/welcome.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectory(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory/${directoryId}`);
    }
    /**
     * Filters access by the value of the request to AWS DS
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS DS Resource being acted upon
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to resource types:
     * - directory
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ds';
        this.accessLevelList = {
            Write: [
                'AcceptSharedDirectory',
                'AddIpRoutes',
                'AddRegion',
                'AuthorizeApplication',
                'CancelSchemaExtension',
                'ConnectDirectory',
                'CreateAlias',
                'CreateComputer',
                'CreateConditionalForwarder',
                'CreateDirectory',
                'CreateIdentityPoolDirectory',
                'CreateLogSubscription',
                'CreateMicrosoftAD',
                'CreateSnapshot',
                'CreateTrust',
                'DeleteConditionalForwarder',
                'DeleteDirectory',
                'DeleteLogSubscription',
                'DeleteSnapshot',
                'DeleteTrust',
                'DeregisterCertificate',
                'DeregisterEventTopic',
                'DisableClientAuthentication',
                'DisableDirectoryDataAccess',
                'DisableLDAPS',
                'DisableRadius',
                'DisableRoleAccess',
                'DisableSso',
                'EnableClientAuthentication',
                'EnableDirectoryDataAccess',
                'EnableLDAPS',
                'EnableRadius',
                'EnableRoleAccess',
                'EnableSso',
                'RegisterCertificate',
                'RegisterEventTopic',
                'RejectSharedDirectory',
                'RemoveIpRoutes',
                'RemoveRegion',
                'ResetUserPassword',
                'RestoreFromSnapshot',
                'ShareDirectory',
                'StartSchemaExtension',
                'UnauthorizeApplication',
                'UnshareDirectory',
                'UpdateAuthorizedApplication',
                'UpdateConditionalForwarder',
                'UpdateDirectory',
                'UpdateDirectorySetup',
                'UpdateNumberOfDomainControllers',
                'UpdateRadius',
                'UpdateSettings',
                'UpdateTrust'
            ],
            'Permissions management': [
                'AccessDSData'
            ],
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            Read: [
                'CheckAlias',
                'DescribeCertificate',
                'DescribeClientAuthenticationSettings',
                'DescribeConditionalForwarders',
                'DescribeDirectoryDataAccess',
                'DescribeDomainControllers',
                'DescribeEventTopics',
                'DescribeLDAPSSettings',
                'DescribeRegions',
                'DescribeSettings',
                'DescribeSharedDirectories',
                'DescribeSnapshots',
                'DescribeTrusts',
                'DescribeUpdateDirectory',
                'GetAuthorizedApplicationDetails',
                'GetDirectoryLimits',
                'GetSnapshotLimits',
                'ListAuthorizedApplications',
                'ListIpRoutes',
                'ListLogSubscriptions',
                'ListTagsForResource',
                'VerifyTrust'
            ],
            List: [
                'DescribeDirectories',
                'ListCertificates',
                'ListSchemaExtensions'
            ]
        };
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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