"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DsData = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ds-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservicedata.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class DsData extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a member to a group on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifMemberName()
     * - .ifRealm()
     * - .ifMemberRealm()
     * - .ifIdentifier()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_AddGroupMember.html
     */
    toAddGroupMember() {
        return this.to('AddGroupMember');
    }
    /**
     * Grants permission to create a group on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create a user on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a group on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete a user on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to describe a group on a directory
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        return this.to('DescribeGroup');
    }
    /**
     * Grants permission to describe a user on a directory
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to disable a user on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_DisableUser.html
     */
    toDisableUser() {
        return this.to('DisableUser');
    }
    /**
     * Grants permission to list members in a group on a directory
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifRealm()
     * - .ifMemberRealm()
     * - .ifIdentifier()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_ListGroupMembers.html
     */
    toListGroupMembers() {
        return this.to('ListGroupMembers');
    }
    /**
     * Grants permission to list groups on a directory
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list the groups that a member is in on a directory
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifRealm()
     * - .ifMemberRealm()
     * - .ifIdentifier()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_ListGroupsForMember.html
     */
    toListGroupsForMember() {
        return this.to('ListGroupsForMember');
    }
    /**
     * Grants permission to list users on a directory
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to remove a member from a group on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifMemberName()
     * - .ifRealm()
     * - .ifMemberRealm()
     * - .ifIdentifier()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_RemoveGroupMember.html
     */
    toRemoveGroupMember() {
        return this.to('RemoveGroupMember');
    }
    /**
     * Grants permission to search for groups on a directory
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds-data:DescribeGroup
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_SearchGroups.html
     */
    toSearchGroups() {
        return this.to('SearchGroups');
    }
    /**
     * Grants permission to search for users on a directory
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds-data:DescribeUser
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_SearchUsers.html
     */
    toSearchUsers() {
        return this.to('SearchUsers');
    }
    /**
     * Grants permission to update a group on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update a user on a directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSAMAccountName()
     * - .ifIdentifier()
     * - .ifRealm()
     *
     * Dependent actions:
     * - ds:AccessDSData
     *
     * https://docs.aws.amazon.com/directoryservicedata/latest/DirectoryServiceDataAPIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/IAM_Auth_Access_Overview.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectory(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory/${directoryId}`);
    }
    /**
     * Filters access by the AWS DS Resource being acted upon
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - directory
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of identifier provided in the request (i.e. SAM Account Name)
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/iam_dsdata-condition-keys.html#dsdata_condition-Identifier
     *
     * Applies to actions:
     * - .toAddGroupMember()
     * - .toCreateGroup()
     * - .toCreateUser()
     * - .toDeleteGroup()
     * - .toDeleteUser()
     * - .toDescribeGroup()
     * - .toDescribeUser()
     * - .toDisableUser()
     * - .toListGroupMembers()
     * - .toListGroupsForMember()
     * - .toRemoveGroupMember()
     * - .toUpdateGroup()
     * - .toUpdateUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIdentifier(value, operator) {
        return this.if(`Identifier`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the directory SAM Account Name included in the MemberName input of the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/iam_dsdata-condition-keys.html#dsdata_condition-MemberName
     *
     * Applies to actions:
     * - .toAddGroupMember()
     * - .toRemoveGroupMember()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMemberName(value, operator) {
        return this.if(`MemberName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the directory realm name included in the MemberRealm input of the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/iam_dsdata-condition-keys.html#dsdata_condition-MemberRealm
     *
     * Applies to actions:
     * - .toAddGroupMember()
     * - .toListGroupMembers()
     * - .toListGroupsForMember()
     * - .toRemoveGroupMember()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMemberRealm(value, operator) {
        return this.if(`MemberRealm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the directory realm name for the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/iam_dsdata-condition-keys.html#dsdata_condition-Realm
     *
     * Applies to actions:
     * - .toAddGroupMember()
     * - .toCreateGroup()
     * - .toCreateUser()
     * - .toDeleteGroup()
     * - .toDeleteUser()
     * - .toDescribeGroup()
     * - .toDescribeUser()
     * - .toDisableUser()
     * - .toListGroupMembers()
     * - .toListGroups()
     * - .toListGroupsForMember()
     * - .toListUsers()
     * - .toRemoveGroupMember()
     * - .toSearchGroups()
     * - .toSearchUsers()
     * - .toUpdateGroup()
     * - .toUpdateUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRealm(value, operator) {
        return this.if(`Realm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the directory SAM Account Name included in the SAMAccountName input of the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/iam_dsdata-condition-keys.html#dsdata_condition-SAMAccountName
     *
     * Applies to actions:
     * - .toAddGroupMember()
     * - .toCreateGroup()
     * - .toCreateUser()
     * - .toDeleteGroup()
     * - .toDeleteUser()
     * - .toDescribeGroup()
     * - .toDescribeUser()
     * - .toDisableUser()
     * - .toListGroupMembers()
     * - .toListGroupsForMember()
     * - .toRemoveGroupMember()
     * - .toUpdateGroup()
     * - .toUpdateUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSAMAccountName(value, operator) {
        return this.if(`SAMAccountName`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ds-data](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservicedata.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ds-data';
        this.accessLevelList = {
            Write: [
                'AddGroupMember',
                'CreateGroup',
                'CreateUser',
                'DeleteGroup',
                'DeleteUser',
                'DisableUser',
                'RemoveGroupMember',
                'UpdateGroup',
                'UpdateUser'
            ],
            Read: [
                'DescribeGroup',
                'DescribeUser',
                'SearchGroups',
                'SearchUsers'
            ],
            List: [
                'ListGroupMembers',
                'ListGroups',
                'ListGroupsForMember',
                'ListUsers'
            ]
        };
    }
}
exports.DsData = DsData;
//# sourceMappingURL=data:application/json;base64,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