import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [eks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastickubernetesservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Eks extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to view Kubernetes objects via AWS EKS console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/view-workloads.html
     */
    toAccessKubernetesApi(): this;
    /**
     * Grants permission to associate an Amazon EKS access policy to an Amazon EKS access entry
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyArn()
     * - .ifNamespaces()
     * - .ifAccessScope()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_AssociateAccessPolicy.html
     */
    toAssociateAccessPolicy(): this;
    /**
     * Grants permission to associate encryption configuration to a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_AssociateEncryptionConfig.html
     */
    toAssociateEncryptionConfig(): this;
    /**
     * Grants permission to associate an identity provider configuration to a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifClientId()
     * - .ifIssuerUrl()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_AssociateIdentityProviderConfig.html
     */
    toAssociateIdentityProviderConfig(): this;
    /**
     * Grants permission to create an Amazon EKS access entry
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifPrincipalArn()
     * - .ifKubernetesGroups()
     * - .ifUsername()
     * - .ifAccessEntryType()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAccessEntry.html
     */
    toCreateAccessEntry(): this;
    /**
     * Grants permission to create an Amazon EKS add-on
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateAddon.html
     */
    toCreateAddon(): this;
    /**
     * Grants permission to create an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBootstrapClusterCreatorAdminPermissions()
     * - .ifBootstrapSelfManagedAddons()
     * - .ifAuthenticationMode()
     * - .ifSupportType()
     * - .ifComputeConfigEnabled()
     * - .ifElasticLoadBalancingEnabled()
     * - .ifBlockStorageEnabled()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster(): this;
    /**
     * Grants permission to create an EKS Anywhere subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateEksAnywhereSubscription.html
     */
    toCreateEksAnywhereSubscription(): this;
    /**
     * Grants permission to create an AWS Fargate profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateFargateProfile.html
     */
    toCreateFargateProfile(): this;
    /**
     * Grants permission to create an Amazon EKS Nodegroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreateNodegroup.html
     */
    toCreateNodegroup(): this;
    /**
     * Grants permission to create an EKS Pod Identity association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_CreatePodIdentityAssociation.html
     */
    toCreatePodIdentityAssociation(): this;
    /**
     * Grants permission to delete an Amazon EKS access entry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteAccessEntry.html
     */
    toDeleteAccessEntry(): this;
    /**
     * Grants permission to delete an Amazon EKS add-on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteAddon.html
     */
    toDeleteAddon(): this;
    /**
     * Grants permission to delete an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster(): this;
    /**
     * Grants permission to describe an EKS Anywhere subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteEksAnywhereSubscription.html
     */
    toDeleteEksAnywhereSubscription(): this;
    /**
     * Grants permission to delete an AWS Fargate profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteFargateProfile.html
     */
    toDeleteFargateProfile(): this;
    /**
     * Grants permission to delete an Amazon EKS Nodegroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeleteNodegroup.html
     */
    toDeleteNodegroup(): this;
    /**
     * Grants permission to delete an EKS Pod Identity association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeletePodIdentityAssociation.html
     */
    toDeletePodIdentityAssociation(): this;
    /**
     * Grants permission to deregister an External cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DeregisterCluster.html
     */
    toDeregisterCluster(): this;
    /**
     * Grants permission to describe an Amazon EKS access entry
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAccessEntry.html
     */
    toDescribeAccessEntry(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS add-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAddon.html
     */
    toDescribeAddon(): this;
    /**
     * Grants permission to list configuration options about an Amazon EKS add-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAddonConfiguration.html
     */
    toDescribeAddonConfiguration(): this;
    /**
     * Grants permission to retrieve descriptive version information about the add-ons that Amazon EKS Add-ons supports
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeAddonVersions.html
     */
    toDescribeAddonVersions(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster(): this;
    /**
     * Grants permission to retrieve descriptive information about Kubernetes versions that Amazon EKS clusters support
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeClusterVersions.html
     */
    toDescribeClusterVersions(): this;
    /**
     * Grants permission to describe an EKS Anywhere subscription
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeEksAnywhereSubscription.html
     */
    toDescribeEksAnywhereSubscription(): this;
    /**
     * Grants permission to retrieve descriptive information about an AWS Fargate profile associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeFargateProfile.html
     */
    toDescribeFargateProfile(): this;
    /**
     * Grants permission to retrieve descriptive information about an Idp config associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeIdentityProviderConfig.html
     */
    toDescribeIdentityProviderConfig(): this;
    /**
     * Grants permission to retrieve descriptive information of a detected insight for a specified cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeInsight.html
     */
    toDescribeInsight(): this;
    /**
     * Grants permission to retrieve descriptive information about an Amazon EKS nodegroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeNodegroup.html
     */
    toDescribeNodegroup(): this;
    /**
     * Grants permission to describe an EKS Pod Identity association
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribePodIdentityAssociation.html
     */
    toDescribePodIdentityAssociation(): this;
    /**
     * Grants permission to retrieve a given update for a given Amazon EKS cluster/nodegroup/add-on (in the specified or default region)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DescribeUpdate.html
     */
    toDescribeUpdate(): this;
    /**
     * Grants permission to disassociate an Amazon EKS access policy from an Amazon EKS acces entry
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifPolicyArn()
     * - .ifNamespaces()
     * - .ifAccessScope()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DisassociateAccessPolicy.html
     */
    toDisassociateAccessPolicy(): this;
    /**
     * Grants permission to delete an asssociated Idp config
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_DisassociateIdentityProviderConfig.html
     */
    toDisassociateIdentityProviderConfig(): this;
    /**
     * Grants permission to list all Amazon EKS access entries
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAccessEntries.html
     */
    toListAccessEntries(): this;
    /**
     * Grants permission to list Amazon EKS access policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAccessPolicies.html
     */
    toListAccessPolicies(): this;
    /**
     * Grants permission to list the Amazon EKS add-ons in your AWS account (in the specified or default region) for a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAddons.html
     */
    toListAddons(): this;
    /**
     * Grants permission to list associated access policy on and Amazon EKS access entry
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListAssociatedAccessPolicies.html
     */
    toListAssociatedAccessPolicies(): this;
    /**
     * Grants permission to list the Amazon EKS clusters in your AWS account (in the specified or default region)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListClusters.html
     */
    toListClusters(): this;
    /**
     * Grants permission to list EKS Anywhere subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListEksAnywhereSubscriptions.html
     */
    toListEksAnywhereSubscriptions(): this;
    /**
     * Grants permission to list the AWS Fargate profiles in your AWS account (in the specified or default region) associated with a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListFargateProfiles.html
     */
    toListFargateProfiles(): this;
    /**
     * Grants permission to list the Idp configs in your AWS account (in the specified or default region) associated with a given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListIdentityProviderConfigs.html
     */
    toListIdentityProviderConfigs(): this;
    /**
     * Grants permission to list all detected insights for a specified cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListInsights.html
     */
    toListInsights(): this;
    /**
     * Grants permission to list the Amazon EKS nodegroups in your AWS account (in the specified or default region) attached to given cluster
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListNodegroups.html
     */
    toListNodegroups(): this;
    /**
     * Grants permission to list EKS Pod Identity associations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListPodIdentityAssociations.html
     */
    toListPodIdentityAssociations(): this;
    /**
     * Grants permission to list tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list the updates for a given Amazon EKS cluster/nodegroup/add-on (in the specified or default region)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_ListUpdates.html
     */
    toListUpdates(): this;
    /**
     * Grants permission to register an External cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_RegisterCluster.html
     */
    toRegisterCluster(): this;
    /**
     * Grants permission to tag the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update an Amazon EKS access entry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAccessEntry.html
     */
    toUpdateAccessEntry(): this;
    /**
     * Grants permission to update Amazon EKS add-on configurations, such as the VPC-CNI version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateAddon.html
     */
    toUpdateAddon(): this;
    /**
     * Grants permission to update Amazon EKS cluster configurations (eg: API server endpoint access)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthenticationMode()
     * - .ifSupportType()
     * - .ifComputeConfigEnabled()
     * - .ifElasticLoadBalancingEnabled()
     * - .ifBlockStorageEnabled()
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterConfig.html
     */
    toUpdateClusterConfig(): this;
    /**
     * Grants permission to update the Kubernetes version of an Amazon EKS cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateClusterVersion.html
     */
    toUpdateClusterVersion(): this;
    /**
     * Grants permission to update an EKS Anywhere subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateEksAnywhereSubscription.html
     */
    toUpdateEksAnywhereSubscription(): this;
    /**
     * Grants permission to update Amazon EKS nodegroup configurations (eg: min/max/desired capacity or labels)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupConfig.html
     */
    toUpdateNodegroupConfig(): this;
    /**
     * Grants permission to update the Kubernetes version of an Amazon EKS nodegroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdateNodegroupVersion.html
     */
    toUpdateNodegroupVersion(): this;
    /**
     * Grants permission to update an EKS Pod Identity association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eks/latest/APIReference/API_UpdatePodIdentityAssociation.html
     */
    toUpdatePodIdentityAssociation(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type nodegroup to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param nodegroupName - Identifier for the nodegroupName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNodegroup(clusterName: string, nodegroupName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type addon to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/eks-add-ons.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param addonName - Identifier for the addonName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAddon(clusterName: string, addonName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type fargateprofile to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param fargateProfileName - Identifier for the fargateProfileName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFargateprofile(clusterName: string, fargateProfileName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type identityproviderconfig to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/authenticate-oidc-identity-provider.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param identityProviderType - Identifier for the identityProviderType.
     * @param identityProviderConfigName - Identifier for the identityProviderConfigName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentityproviderconfig(clusterName: string, identityProviderType: string, identityProviderConfigName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type eks-anywhere-subscription to the statement
     *
     * https://anywhere.eks.amazonaws.com/docs/clustermgmt/support/cluster-license/
     *
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEksAnywhereSubscription(uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type podidentityassociation to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/pod-identities.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPodidentityassociation(clusterName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type access-entry to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/access-entries.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param iamIdentityType - Identifier for the iamIdentityType.
     * @param iamIdentityAccountID - Identifier for the iamIdentityAccountID.
     * @param iamIdentityName - Identifier for the iamIdentityName.
     * @param uUID - Identifier for the uUID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAccessEntryType()
     * - .ifClusterName()
     * - .ifKubernetesGroups()
     * - .ifPrincipalArn()
     * - .ifUsername()
     */
    onAccessEntry(clusterName: string, iamIdentityType: string, iamIdentityAccountID: string, iamIdentityName: string, uUID: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type access-policy to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/access-policies.html
     *
     * @param accessPolicyName - Identifier for the accessPolicyName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccessPolicy(accessPolicyName: string, partition?: string): this;
    /**
     * Filters access by a key that is present in the request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     * - .toCreateAccessEntry()
     * - .toCreateAddon()
     * - .toCreateCluster()
     * - .toCreateEksAnywhereSubscription()
     * - .toCreateFargateProfile()
     * - .toCreateNodegroup()
     * - .toCreatePodIdentityAssociation()
     * - .toRegisterCluster()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to resource types:
     * - cluster
     * - nodegroup
     * - addon
     * - fargateprofile
     * - identityproviderconfig
     * - eks-anywhere-subscription
     * - podidentityassociation
     * - access-entry
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     * - .toCreateAccessEntry()
     * - .toCreateAddon()
     * - .toCreateCluster()
     * - .toCreateEksAnywhereSubscription()
     * - .toCreateFargateProfile()
     * - .toCreateNodegroup()
     * - .toCreatePodIdentityAssociation()
     * - .toRegisterCluster()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the access entry type present in the access entry requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateAccessEntry()
     *
     * Applies to resource types:
     * - access-entry
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessEntryType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the accessScope present in the associate / disassociate access policy requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateAccessPolicy()
     * - .toDisassociateAccessPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessScope(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the authenticationMode present in the create / update cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toUpdateClusterConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthenticationMode(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the block storage enabled parameter in the create / update cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toUpdateClusterConfig()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBlockStorageEnabled(value?: boolean): this;
    /**
     * Filters access by the bootstrapClusterCreatorAdminPermissions present in the create cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBootstrapClusterCreatorAdminPermissions(value?: boolean): this;
    /**
     * Filters access by the bootstrapSelfManagedAddons present in the create cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBootstrapSelfManagedAddons(value?: boolean): this;
    /**
     * Filters access by the clientId present in the associateIdentityProviderConfig request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClientId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the clusterName present in the access entry requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to resource types:
     * - access-entry
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifClusterName(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the compute config enabled parameter in the create / update cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toUpdateClusterConfig()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifComputeConfigEnabled(value?: boolean): this;
    /**
     * Filters access by the elastic load balancing enabled parameter in the create / update cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toUpdateClusterConfig()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifElasticLoadBalancingEnabled(value?: boolean): this;
    /**
     * Filters access by the issuerUrl present in the associateIdentityProviderConfig request the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateIdentityProviderConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIssuerUrl(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the kubernetesGroups present in the access entry requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateAccessEntry()
     *
     * Applies to resource types:
     * - access-entry
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKubernetesGroups(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the namespaces present in the associate / disassociate access policy requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateAccessPolicy()
     * - .toDisassociateAccessPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifNamespaces(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the policyArn present in the access entry requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toAssociateAccessPolicy()
     * - .toDisassociateAccessPolicy()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPolicyArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the principalArn present in the access entry requests requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateAccessEntry()
     *
     * Applies to resource types:
     * - access-entry
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifPrincipalArn(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the supportType present in the create / update cluster request
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toUpdateClusterConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSupportType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the Kubernetes username present in the access entry requests the user makes to the EKS service
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateAccessEntry()
     *
     * Applies to resource types:
     * - access-entry
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUsername(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [eks](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastickubernetesservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
