"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconnect = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediaconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediaconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconnect extends shared_1.PolicyStatement {
    /**
     * Grants permission to add outputs to an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-outputs.html
     */
    toAddBridgeOutputs() {
        return this.to('AddBridgeOutputs');
    }
    /**
     * Grants permission to add sources to an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-sources.html
     */
    toAddBridgeSources() {
        return this.to('AddBridgeSources');
    }
    /**
     * Grants permission to add media streams to any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-mediastreams.html
     */
    toAddFlowMediaStreams() {
        return this.to('AddFlowMediaStreams');
    }
    /**
     * Grants permission to add outputs to any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs.html
     */
    toAddFlowOutputs() {
        return this.to('AddFlowOutputs');
    }
    /**
     * Grants permission to add sources to any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source.html
     */
    toAddFlowSources() {
        return this.to('AddFlowSources');
    }
    /**
     * Grants permission to add VPC interfaces to any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-vpcinterfaces.html
     */
    toAddFlowVpcInterfaces() {
        return this.to('AddFlowVpcInterfaces');
    }
    /**
     * Grants permission to create bridges
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges.html
     */
    toCreateBridge() {
        return this.to('CreateBridge');
    }
    /**
     * Grants permission to create flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    toCreateFlow() {
        return this.to('CreateFlow');
    }
    /**
     * Grants permission to create gateways
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateways.html
     */
    toCreateGateway() {
        return this.to('CreateGateway');
    }
    /**
     * Grants permission to delete bridges
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn.html
     */
    toDeleteBridge() {
        return this.to('DeleteBridge');
    }
    /**
     * Grants permission to delete flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    toDeleteFlow() {
        return this.to('DeleteFlow');
    }
    /**
     * Grants permission to delete gateways
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateways-gatewayarn.html
     */
    toDeleteGateway() {
        return this.to('DeleteGateway');
    }
    /**
     * Grants permission to deregister gateway instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateway-instances-gatewayinstancearn.html
     */
    toDeregisterGatewayInstance() {
        return this.to('DeregisterGatewayInstance');
    }
    /**
     * Grants permission to display the details of a bridge
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn.html
     */
    toDescribeBridge() {
        return this.to('DescribeBridge');
    }
    /**
     * Grants permission to display the details of a flow including the flow ARN, name, and Availability Zone, as well as details about the source, outputs, and entitlements
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    toDescribeFlow() {
        return this.to('DescribeFlow');
    }
    /**
     * Grants permission to view information about the flow's source transport stream and programs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-metadata.html
     */
    toDescribeFlowSourceMetadata() {
        return this.to('DescribeFlowSourceMetadata');
    }
    /**
     * Grants permission to view flow's source thumbnail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-thumbnail.html
     */
    toDescribeFlowSourceThumbnail() {
        return this.to('DescribeFlowSourceThumbnail');
    }
    /**
     * Grants permission to display the details of a gateway including the gateway ARN, name, and CIDR blocks, as well as details about the networks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateways-gatewayarn.html
     */
    toDescribeGateway() {
        return this.to('DescribeGateway');
    }
    /**
     * Grants permission to display the details of a gateway instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateway-instances-gatewayinstancearn.html
     */
    toDescribeGatewayInstance() {
        return this.to('DescribeGatewayInstance');
    }
    /**
     * Grants permission to display the details of an offering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-offerings-offeringarn.html
     */
    toDescribeOffering() {
        return this.to('DescribeOffering');
    }
    /**
     * Grants permission to display the details of a reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-reservations-reservationarn.html
     */
    toDescribeReservation() {
        return this.to('DescribeReservation');
    }
    /**
     * Grants permission to discover gateway poll endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    toDiscoverGatewayPollEndpoint() {
        return this.to('DiscoverGatewayPollEndpoint');
    }
    /**
     * Grants permission to grant entitlements on any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements.html
     */
    toGrantFlowEntitlements() {
        return this.to('GrantFlowEntitlements');
    }
    /**
     * Grants permission to display a list of bridges that are associated with this account and an optionally specified Arn
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges.html
     */
    toListBridges() {
        return this.to('ListBridges');
    }
    /**
     * Grants permission to display a list of all entitlements that have been granted to the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-entitlements.html
     */
    toListEntitlements() {
        return this.to('ListEntitlements');
    }
    /**
     * Grants permission to display a list of flows that are associated with this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    toListFlows() {
        return this.to('ListFlows');
    }
    /**
     * Grants permission to display a list of instances that are associated with this gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateway-instances.html
     */
    toListGatewayInstances() {
        return this.to('ListGatewayInstances');
    }
    /**
     * Grants permission to display a list of gateways that are associated with this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateways.html
     */
    toListGateways() {
        return this.to('ListGateways');
    }
    /**
     * Grants permission to display a list of all offerings that are available to the account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-offerings.html
     */
    toListOfferings() {
        return this.to('ListOfferings');
    }
    /**
     * Grants permission to display a list of all reservations that have been purchased by the account in the current AWS Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-reservations.html
     */
    toListReservations() {
        return this.to('ListReservations');
    }
    /**
     * Grants permission to display a list of all tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/tags-resourcearn.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to poll gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    toPollGateway() {
        return this.to('PollGateway');
    }
    /**
     * Grants permission to purchase an offering
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-offerings-offeringarn.html
     */
    toPurchaseOffering() {
        return this.to('PurchaseOffering');
    }
    /**
     * Grants permission to remove an output of an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-outputs-outputname.html
     */
    toRemoveBridgeOutput() {
        return this.to('RemoveBridgeOutput');
    }
    /**
     * Grants permission to remove a source of an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-sources-sourcename.html
     */
    toRemoveBridgeSource() {
        return this.to('RemoveBridgeSource');
    }
    /**
     * Grants permission to remove media streams from any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-mediastreams-mediastreamname.html
     */
    toRemoveFlowMediaStream() {
        return this.to('RemoveFlowMediaStream');
    }
    /**
     * Grants permission to remove outputs from any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    toRemoveFlowOutput() {
        return this.to('RemoveFlowOutput');
    }
    /**
     * Grants permission to remove sources from any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html
     */
    toRemoveFlowSource() {
        return this.to('RemoveFlowSource');
    }
    /**
     * Grants permission to remove VPC interfaces from any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-vpcinterfaces-vpcinterfacename.html
     */
    toRemoveFlowVpcInterface() {
        return this.to('RemoveFlowVpcInterface');
    }
    /**
     * Grants permission to revoke entitlements on any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    toRevokeFlowEntitlement() {
        return this.to('RevokeFlowEntitlement');
    }
    /**
     * Grants permission to start flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-start-flowarn.html
     */
    toStartFlow() {
        return this.to('StartFlow');
    }
    /**
     * Grants permission to stop flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-stop-flowarn.html
     */
    toStopFlow() {
        return this.to('StopFlow');
    }
    /**
     * Grants permission to submit gateway state change
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows.html
     */
    toSubmitGatewayStateChange() {
        return this.to('SubmitGatewayStateChange');
    }
    /**
     * Grants permission to associate tags with resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/tags-resourcearn.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from resources
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/tags-resourcearn.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update bridges
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn.html
     */
    toUpdateBridge() {
        return this.to('UpdateBridge');
    }
    /**
     * Grants permission to update an output of an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-outputs-outputname.html
     */
    toUpdateBridgeOutput() {
        return this.to('UpdateBridgeOutput');
    }
    /**
     * Grants permission to update a source of an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-sources-sourcename.html
     */
    toUpdateBridgeSource() {
        return this.to('UpdateBridgeSource');
    }
    /**
     * Grants permission to update the state of an existing bridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-bridges-bridgearn-state.html
     */
    toUpdateBridgeState() {
        return this.to('UpdateBridgeState');
    }
    /**
     * Grants permission to update flows
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn.html
     */
    toUpdateFlow() {
        return this.to('UpdateFlow');
    }
    /**
     * Grants permission to update entitlements on any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-entitlements-entitlementarn.html
     */
    toUpdateFlowEntitlement() {
        return this.to('UpdateFlowEntitlement');
    }
    /**
     * Grants permission to update media streams on any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-mediastreams-mediastreamname.html
     */
    toUpdateFlowMediaStream() {
        return this.to('UpdateFlowMediaStream');
    }
    /**
     * Grants permission to update outputs on any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-outputs-outputarn.html
     */
    toUpdateFlowOutput() {
        return this.to('UpdateFlowOutput');
    }
    /**
     * Grants permission to update the source of any flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-flows-flowarn-source-sourcearn.html
     */
    toUpdateFlowSource() {
        return this.to('UpdateFlowSource');
    }
    /**
     * Grants permission to update the configuration of an existing Gateway Instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/api/v1-gateway-instances-gatewayinstancearn.html
     */
    toUpdateGatewayInstance() {
        return this.to('UpdateGatewayInstance');
    }
    /**
     * Adds a resource of type Entitlement to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/entitlements.html
     *
     * @param flowId - Identifier for the flowId.
     * @param entitlementName - Identifier for the entitlementName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEntitlement(flowId, entitlementName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:entitlement:${flowId}:${entitlementName}`);
    }
    /**
     * Adds a resource of type Flow to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/flows.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFlow(flowId, flowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:flow:${flowId}:${flowName}`);
    }
    /**
     * Adds a resource of type Output to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/outputs.html
     *
     * @param outputId - Identifier for the outputId.
     * @param outputName - Identifier for the outputName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOutput(outputId, outputName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:output:${outputId}:${outputName}`);
    }
    /**
     * Adds a resource of type Source to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/sources.html
     *
     * @param sourceId - Identifier for the sourceId.
     * @param sourceName - Identifier for the sourceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSource(sourceId, sourceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:source:${sourceId}:${sourceName}`);
    }
    /**
     * Adds a resource of type Gateway to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/gateway.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param gatewayName - Identifier for the gatewayName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGateway(gatewayId, gatewayName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway:${gatewayId}:${gatewayName}`);
    }
    /**
     * Adds a resource of type Bridge to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/gateway-components-bridges.html
     *
     * @param flowId - Identifier for the flowId.
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBridge(flowId, flowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bridge:${flowId}:${flowName}`);
    }
    /**
     * Adds a resource of type GatewayInstance to the statement
     *
     * https://docs.aws.amazon.com/mediaconnect/latest/ug/gateway-components-instances.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param gatewayName - Identifier for the gatewayName.
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGatewayInstance(gatewayId, gatewayName, instanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconnect:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway:${gatewayId}:${gatewayName}:instance:${instanceId}`);
    }
    /**
     * Statement provider for service [mediaconnect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediaconnect.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediaconnect';
        this.accessLevelList = {
            Write: [
                'AddBridgeOutputs',
                'AddBridgeSources',
                'AddFlowMediaStreams',
                'AddFlowOutputs',
                'AddFlowSources',
                'AddFlowVpcInterfaces',
                'CreateBridge',
                'CreateFlow',
                'CreateGateway',
                'DeleteBridge',
                'DeleteFlow',
                'DeleteGateway',
                'DeregisterGatewayInstance',
                'DiscoverGatewayPollEndpoint',
                'GrantFlowEntitlements',
                'PollGateway',
                'PurchaseOffering',
                'RemoveBridgeOutput',
                'RemoveBridgeSource',
                'RemoveFlowMediaStream',
                'RemoveFlowOutput',
                'RemoveFlowSource',
                'RemoveFlowVpcInterface',
                'RevokeFlowEntitlement',
                'StartFlow',
                'StopFlow',
                'SubmitGatewayStateChange',
                'UpdateBridge',
                'UpdateBridgeOutput',
                'UpdateBridgeSource',
                'UpdateBridgeState',
                'UpdateFlow',
                'UpdateFlowEntitlement',
                'UpdateFlowMediaStream',
                'UpdateFlowOutput',
                'UpdateFlowSource',
                'UpdateGatewayInstance'
            ],
            Read: [
                'DescribeBridge',
                'DescribeFlow',
                'DescribeFlowSourceMetadata',
                'DescribeFlowSourceThumbnail',
                'DescribeGateway',
                'DescribeGatewayInstance',
                'DescribeOffering',
                'DescribeReservation',
                'ListTagsForResource'
            ],
            List: [
                'ListBridges',
                'ListEntitlements',
                'ListFlows',
                'ListGatewayInstances',
                'ListGateways',
                'ListOfferings',
                'ListReservations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mediaconnect = Mediaconnect;
//# sourceMappingURL=data:application/json;base64,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