import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [medialive](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmedialive.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Medialive extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to accept an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toAcceptInputDeviceTransfer(): this;
    /**
     * Grants permission to delete channels, inputs, input security groups, and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toBatchDelete(): this;
    /**
     * Grants permission to start channels and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toBatchStart(): this;
    /**
     * Grants permission to stop channels and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toBatchStop(): this;
    /**
     * Grants permission to add and remove actions from a channel's schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/submitting-batch-command.html
     */
    toBatchUpdateSchedule(): this;
    /**
     * Grants permission to cancel an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toCancelInputDeviceTransfer(): this;
    /**
     * Grants permission to claim an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toClaimDevice(): this;
    /**
     * Grants permission to create a channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-channel-scratch.html
     */
    toCreateChannel(): this;
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateChannelPlacementGroup(): this;
    /**
     * Grants permission to create a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toCreateCloudWatchAlarmTemplate(): this;
    /**
     * Grants permission to create a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toCreateCloudWatchAlarmTemplateGroup(): this;
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateCluster(): this;
    /**
     * Grants permission to create a eventbridge rule template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toCreateEventBridgeRuleTemplate(): this;
    /**
     * Grants permission to create a eventbridge rule template group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toCreateEventBridgeRuleTemplateGroup(): this;
    /**
     * Grants permission to create an input
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html
     */
    toCreateInput(): this;
    /**
     * Grants permission to create an input security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html
     */
    toCreateInputSecurityGroup(): this;
    /**
     * Grants permission to create a multiplex
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/multiplex-create.html
     */
    toCreateMultiplex(): this;
    /**
     * Grants permission to create a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/multiplex-create.html
     */
    toCreateMultiplexProgram(): this;
    /**
     * Grants permission to create a network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNetwork(): this;
    /**
     * Grants permission to create a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNode(): this;
    /**
     * Grants permission to create a node registration script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNodeRegistrationScript(): this;
    /**
     * Grants permission to create a partner input
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/input-create-cdi-partners.html
     */
    toCreatePartnerInput(): this;
    /**
     * Grants permission to create a SDI source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateSdiSource(): this;
    /**
     * Grants permission to create a signal map
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-create.html
     */
    toCreateSignalMap(): this;
    /**
     * Grants permission to create tags for channels, inputs, input security groups, multiplexes, reservations, nodes, networks, clusters, channel placement groups, signal maps, SDI sources, template groups, and templates
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toCreateTags(): this;
    /**
     * Grants permission to delete a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toDeleteChannel(): this;
    /**
     * Grants permission to delete a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteChannelPlacementGroup(): this;
    /**
     * Grants permission to delete a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toDeleteCloudWatchAlarmTemplate(): this;
    /**
     * Grants permission to delete a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toDeleteCloudWatchAlarmTemplateGroup(): this;
    /**
     * Grants permission to delete a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteCluster(): this;
    /**
     * Grants permission to delete a eventbridge rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toDeleteEventBridgeRuleTemplate(): this;
    /**
     * Grants permission to delete a eventbridge rule template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toDeleteEventBridgeRuleTemplateGroup(): this;
    /**
     * Grants permission to delete an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input.html
     */
    toDeleteInput(): this;
    /**
     * Grants permission to delete an input security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input-security-group.html
     */
    toDeleteInputSecurityGroup(): this;
    /**
     * Grants permission to delete a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex-program.html
     */
    toDeleteMultiplex(): this;
    /**
     * Grants permission to delete a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex-program.html
     */
    toDeleteMultiplexProgram(): this;
    /**
     * Grants permission to delete a network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteNetwork(): this;
    /**
     * Grants permission to delete a node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteNode(): this;
    /**
     * Grants permission to delete an expired reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/deleting-reservations.html
     */
    toDeleteReservation(): this;
    /**
     * Grants permission to delete all schedule actions for a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/schedule-using-console-delete.html
     */
    toDeleteSchedule(): this;
    /**
     * Grants permission to delete a SDI source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteSdiSource(): this;
    /**
     * Grants permission to delete a signal map
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-delete.html
     */
    toDeleteSignalMap(): this;
    /**
     * Grants permission to delete tags from channels, inputs, input security groups, multiplexes, reservations, nodes, clusters, networks, channel placement groups, SDI source, signal maps, template groups, and templates
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toDeleteTags(): this;
    /**
     * Grants permission to view the account configuration of the customer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toDescribeAccountConfiguration(): this;
    /**
     * Grants permission to get details about a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    toDescribeChannel(): this;
    /**
     * Grants permission to describe a channel placement group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeChannelPlacementGroup(): this;
    /**
     * Grants permission to describe a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeCluster(): this;
    /**
     * Grants permission to describe an input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toDescribeInput(): this;
    /**
     * Grants permission to describe an input device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toDescribeInputDevice(): this;
    /**
     * Grants permission to describe an input device thumbnail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toDescribeInputDeviceThumbnail(): this;
    /**
     * Grants permission to describe an input security group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toDescribeInputSecurityGroup(): this;
    /**
     * Grants permission to describe a multiplex
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toDescribeMultiplex(): this;
    /**
     * Grants permission to describe a multiplex program
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitoring-multiplex-console.html
     */
    toDescribeMultiplexProgram(): this;
    /**
     * Grants permission to describe a network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-networks.html
     */
    toDescribeNetwork(): this;
    /**
     * Grants permission to describe a node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-nodes-create.html
     */
    toDescribeNode(): this;
    /**
     * Grants permission to get details about a reservation offering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toDescribeOffering(): this;
    /**
     * Grants permission to get details about a reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    toDescribeReservation(): this;
    /**
     * Grants permission to view a list of actions scheduled on a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/schedule-using-console-view.html
     */
    toDescribeSchedule(): this;
    /**
     * Grants permission to describe a SDI source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeSdiSource(): this;
    /**
     * Grants permission to view the thumbnails for a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toDescribeThumbnails(): this;
    /**
     * Grants permission to get a cloudwatch alarm template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toGetCloudWatchAlarmTemplate(): this;
    /**
     * Grants permission to get a cloudwatch alarm template group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toGetCloudWatchAlarmTemplateGroup(): this;
    /**
     * Grants permission to get a eventbridge rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toGetEventBridgeRuleTemplate(): this;
    /**
     * Grants permission to get a eventbridge rule template group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toGetEventBridgeRuleTemplateGroup(): this;
    /**
     * Grants permission to get a signal map
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-view.html
     */
    toGetSignalMap(): this;
    /**
     * Grants permission to list channel placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListChannelPlacementGroups(): this;
    /**
     * Grants permission to list channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    toListChannels(): this;
    /**
     * Grants permission to list cloudwatch alarm template groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toListCloudWatchAlarmTemplateGroups(): this;
    /**
     * Grants permission to list cloudwatch alarm templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toListCloudWatchAlarmTemplates(): this;
    /**
     * Grants permission to list clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListClusters(): this;
    /**
     * Grants permission to list eventbridge rule template groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toListEventBridgeRuleTemplateGroups(): this;
    /**
     * Grants permission to list eventbridge rule templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toListEventBridgeRuleTemplates(): this;
    /**
     * Grants permission to list input device transfers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toListInputDeviceTransfers(): this;
    /**
     * Grants permission to list input devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toListInputDevices(): this;
    /**
     * Grants permission to list input security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toListInputSecurityGroups(): this;
    /**
     * Grants permission to list inputs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toListInputs(): this;
    /**
     * Grants permission to list multiplex programs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitoring-multiplex-console.html
     */
    toListMultiplexPrograms(): this;
    /**
     * Grants permission to list multiplexes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toListMultiplexes(): this;
    /**
     * Grants permission to list networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListNetworks(): this;
    /**
     * Grants permission to list nodes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListNodes(): this;
    /**
     * Grants permission to list reservation offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toListOfferings(): this;
    /**
     * Grants permission to list reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    toListReservations(): this;
    /**
     * Grants permission to list SDI sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListSdiSources(): this;
    /**
     * Grants permission to list signal maps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-view.html
     */
    toListSignalMaps(): this;
    /**
     * Grants permission to list tags for channels, inputs, input security groups, multiplexes, reservations, nodes, clusters, networks, channel placement groups, SDI sources, signal maps, template groups, and templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list available versions of MediaLive
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/medialive-versions.html
     */
    toListVersions(): this;
    /**
     * Grants permission to the node to poll the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/about-emla.html
     */
    toPollAnywhere(): this;
    /**
     * Grants permission to purchase a reservation offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toPurchaseOffering(): this;
    /**
     * Grants permission to reboot an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toRebootInputDevice(): this;
    /**
     * Grants permission to reject an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toRejectInputDeviceTransfer(): this;
    /**
     * Grants permission to restart pipelines on a running channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/maintenance-user-initiated.html
     */
    toRestartChannelPipelines(): this;
    /**
     * Grants permission to start a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toStartChannel(): this;
    /**
     * Grants permission to start deletion of a signal map's monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-delete.html
     */
    toStartDeleteMonitorDeployment(): this;
    /**
     * Grants permission to start an input device attached to a MediaConnect flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStartInputDevice(): this;
    /**
     * Grants permission to start a maintenance window for an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStartInputDeviceMaintenanceWindow(): this;
    /**
     * Grants permission to start a signal map monitor deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-deploy.html
     */
    toStartMonitorDeployment(): this;
    /**
     * Grants permission to start a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/start-multiplex.html
     */
    toStartMultiplex(): this;
    /**
     * Grants permission to start a signal map update
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-update.html
     */
    toStartUpdateSignalMap(): this;
    /**
     * Grants permission to stop a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toStopChannel(): this;
    /**
     * Grants permission to stop an input device attached to a MediaConnect flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStopInputDevice(): this;
    /**
     * Grants permission to stop a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/stop-multiplex.title.html
     */
    toStopMultiplex(): this;
    /**
     * Grants permission to the node to submit state changes to the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/about-emla.html
     */
    toSubmitAnywhereStateChange(): this;
    /**
     * Grants permission to transfer an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toTransferInputDevice(): this;
    /**
     * Grants permission to update a customer's account configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toUpdateAccountConfiguration(): this;
    /**
     * Grants permission to update a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toUpdateChannel(): this;
    /**
     * Grants permission to update the class of a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toUpdateChannelClass(): this;
    /**
     * Grants permission to update a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateChannelPlacementGroup(): this;
    /**
     * Grants permission to update a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toUpdateCloudWatchAlarmTemplate(): this;
    /**
     * Grants permission to update a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toUpdateCloudWatchAlarmTemplateGroup(): this;
    /**
     * Grants permission to update a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toUpdateCluster(): this;
    /**
     * Grants permission to update a eventbridge rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toUpdateEventBridgeRuleTemplate(): this;
    /**
     * Grants permission to update a eventbridge rule template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toUpdateEventBridgeRuleTemplateGroup(): this;
    /**
     * Grants permission to update an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toUpdateInput(): this;
    /**
     * Grants permission to update an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toUpdateInputDevice(): this;
    /**
     * Grants permission to update an input security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toUpdateInputSecurityGroup(): this;
    /**
     * Grants permission to update a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toUpdateMultiplex(): this;
    /**
     * Grants permission to update a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toUpdateMultiplexProgram(): this;
    /**
     * Grants permission to update the state of a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-deploy-identify-network-requirements.html
     */
    toUpdateNetwork(): this;
    /**
     * Grants permission to update a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-nodes-create.html
     */
    toUpdateNode(): this;
    /**
     * Grants permission to update the state of a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateNodeState(): this;
    /**
     * Grants permission to update a reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     */
    toUpdateReservation(): this;
    /**
     * Grants permission to update the state of a sdi source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateSdiSource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/container-channel.html
     *
     * @param channelId - Identifier for the channelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html
     *
     * @param inputId - Identifier for the inputId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInput(inputId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type input-device to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     *
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onInputDevice(deviceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type input-security-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html
     *
     * @param inputSecurityGroupId - Identifier for the inputSecurityGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInputSecurityGroup(inputSecurityGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type multiplex to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-multiplex.html
     *
     * @param multiplexId - Identifier for the multiplexId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMultiplex(multiplexId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type reservation to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReservation(reservationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type offering to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/input-output-reservations.html
     *
     * @param offeringId - Identifier for the offeringId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOffering(offeringId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type signal-map to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-create.html
     *
     * @param signalMapId - Identifier for the signalMapId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSignalMap(signalMapId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cloudwatch-alarm-template-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     *
     * @param cloudWatchAlarmTemplateGroupId - Identifier for the cloudWatchAlarmTemplateGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudwatchAlarmTemplateGroup(cloudWatchAlarmTemplateGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cloudwatch-alarm-template to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     *
     * @param cloudWatchAlarmTemplateId - Identifier for the cloudWatchAlarmTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudwatchAlarmTemplate(cloudWatchAlarmTemplateId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type eventbridge-rule-template-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     *
     * @param eventBridgeRuleTemplateGroupId - Identifier for the eventBridgeRuleTemplateGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventbridgeRuleTemplateGroup(eventBridgeRuleTemplateGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type eventbridge-rule-template to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     *
     * @param eventBridgeRuleTemplateId - Identifier for the eventBridgeRuleTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventbridgeRuleTemplate(eventBridgeRuleTemplateId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNode(clusterId: string, nodeId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param networkId - Identifier for the networkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetwork(networkId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type channel-placement-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param channelPlacementGroupId - Identifier for the channelPlacementGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannelPlacementGroup(clusterId: string, channelPlacementGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type sdi-source to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param sdiSourceId - Identifier for the sdiSourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSdiSource(sdiSourceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelPlacementGroup()
     * - .toCreateCloudWatchAlarmTemplate()
     * - .toCreateCloudWatchAlarmTemplateGroup()
     * - .toCreateCluster()
     * - .toCreateEventBridgeRuleTemplate()
     * - .toCreateEventBridgeRuleTemplateGroup()
     * - .toCreateInput()
     * - .toCreateInputSecurityGroup()
     * - .toCreateMultiplex()
     * - .toCreateNetwork()
     * - .toCreateNode()
     * - .toCreatePartnerInput()
     * - .toCreateSdiSource()
     * - .toCreateSignalMap()
     * - .toCreateTags()
     * - .toPurchaseOffering()
     * - .toUpdateChannelPlacementGroup()
     * - .toUpdateCluster()
     * - .toUpdateInputSecurityGroup()
     * - .toUpdateNetwork()
     * - .toUpdateNode()
     * - .toUpdateNodeState()
     * - .toUpdateSdiSource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to resource types:
     * - channel
     * - input
     * - input-security-group
     * - multiplex
     * - reservation
     * - signal-map
     * - cloudwatch-alarm-template-group
     * - cloudwatch-alarm-template
     * - eventbridge-rule-template-group
     * - eventbridge-rule-template
     * - cluster
     * - node
     * - network
     * - channel-placement-group
     * - sdi-source
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelPlacementGroup()
     * - .toCreateCloudWatchAlarmTemplate()
     * - .toCreateCloudWatchAlarmTemplateGroup()
     * - .toCreateCluster()
     * - .toCreateEventBridgeRuleTemplate()
     * - .toCreateEventBridgeRuleTemplateGroup()
     * - .toCreateInput()
     * - .toCreateInputSecurityGroup()
     * - .toCreateMultiplex()
     * - .toCreateNetwork()
     * - .toCreateNode()
     * - .toCreatePartnerInput()
     * - .toCreateSdiSource()
     * - .toCreateSignalMap()
     * - .toCreateTags()
     * - .toDeleteTags()
     * - .toPurchaseOffering()
     * - .toUpdateChannelPlacementGroup()
     * - .toUpdateCluster()
     * - .toUpdateInputSecurityGroup()
     * - .toUpdateNetwork()
     * - .toUpdateNode()
     * - .toUpdateNodeState()
     * - .toUpdateSdiSource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [medialive](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmedialive.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
