"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediapackage = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediapackage](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackage.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediapackage extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure access logs for a Channel
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-configure_logs.html#channels-id-configure_logsput
     */
    toConfigureLogs() {
        return this.to('ConfigureLogs');
    }
    /**
     * Grants permission to create a channel in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelspost
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a harvest job in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs.html#harvest_jobspost
     */
    toCreateHarvestJob() {
        return this.to('CreateHarvestJob');
    }
    /**
     * Grants permission to create an endpoint in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointspost
     */
    toCreateOriginEndpoint() {
        return this.to('CreateOriginEndpoint');
    }
    /**
     * Grants permission to delete a channel in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-iddelete
     */
    toDeleteChannel() {
        return this.to('DeleteChannel');
    }
    /**
     * Grants permission to delete an endpoint in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-iddelete
     */
    toDeleteOriginEndpoint() {
        return this.to('DeleteOriginEndpoint');
    }
    /**
     * Grants permission to view the details of a channel in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idget
     */
    toDescribeChannel() {
        return this.to('DescribeChannel');
    }
    /**
     * Grants permission to view the details of a harvest job in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs-id.html#harvest_jobs-idget
     */
    toDescribeHarvestJob() {
        return this.to('DescribeHarvestJob');
    }
    /**
     * Grants permission to view the details of an endpoint in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idget
     */
    toDescribeOriginEndpoint() {
        return this.to('DescribeOriginEndpoint');
    }
    /**
     * Grants permission to view a list of channels in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels.html#channelsget
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to view a list of harvest jobs in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/harvest_jobs.html#harvest_jobsget
     */
    toListHarvestJobs() {
        return this.to('ListHarvestJobs');
    }
    /**
     * Grants permission to view a list of endpoints in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints.html#origin_endpointsget
     */
    toListOriginEndpoints() {
        return this.to('ListOriginEndpoints');
    }
    /**
     * Grants permission to list the tags assigned to a Channel or OriginEndpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to rotate credentials for the first IngestEndpoint of a Channel in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-credentials.html#channels-id-credentialsput
     */
    toRotateChannelCredentials() {
        return this.to('RotateChannelCredentials');
    }
    /**
     * Grants permission to rotate IngestEndpoint credentials for a Channel in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id-ingest_endpoints-ingest_endpoint_id-credentials.html#channels-id-ingest_endpoints-ingest_endpoint_id-credentialsput
     */
    toRotateIngestEndpointCredentials() {
        return this.to('RotateIngestEndpointCredentials');
    }
    /**
     * Grants permission to tag a MediaPackage resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/hj-create.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to delete tags to a Channel or OriginEndpoint
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to make changes to a channel in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/channels-id.html#channels-idput
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to make changes to an endpoint in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage/latest/apireference/origin_endpoints-id.html#origin_endpoints-idput
     */
    toUpdateOriginEndpoint() {
        return this.to('UpdateOriginEndpoint');
    }
    /**
     * Adds a resource of type channels to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/channels.html
     *
     * @param channelIdentifier - Identifier for the channelIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannels(channelIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channels/${channelIdentifier}`);
    }
    /**
     * Adds a resource of type origin_endpoints to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/endpoints.html
     *
     * @param originEndpointIdentifier - Identifier for the originEndpointIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOriginEndpoints(originEndpointIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:origin_endpoints/${originEndpointIdentifier}`);
    }
    /**
     * Adds a resource of type harvest_jobs to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/harvest-jobs.html
     *
     * @param harvestJobIdentifier - Identifier for the harvestJobIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHarvestJobs(harvestJobIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:harvest_jobs/${harvestJobIdentifier}`);
    }
    /**
     * Filters access by the tag for a MediaPackage request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateHarvestJob()
     * - .toCreateOriginEndpoint()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag for a MediaPackage resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - channels
     * - origin_endpoints
     * - harvest_jobs
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys for a MediaPackage resource or request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateHarvestJob()
     * - .toCreateOriginEndpoint()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mediapackage](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackage.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediapackage';
        this.accessLevelList = {
            Write: [
                'ConfigureLogs',
                'CreateChannel',
                'CreateHarvestJob',
                'CreateOriginEndpoint',
                'DeleteChannel',
                'DeleteOriginEndpoint',
                'RotateChannelCredentials',
                'RotateIngestEndpointCredentials',
                'UpdateChannel',
                'UpdateOriginEndpoint'
            ],
            Read: [
                'DescribeChannel',
                'DescribeHarvestJob',
                'DescribeOriginEndpoint',
                'ListChannels',
                'ListHarvestJobs',
                'ListOriginEndpoints',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mediapackage = Mediapackage;
//# sourceMappingURL=data:application/json;base64,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