"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediastore = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediastore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediastore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediastore extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a container
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_CreateContainer.html
     */
    toCreateContainer() {
        return this.to('CreateContainer');
    }
    /**
     * Grants permission to delete a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainer.html
     */
    toDeleteContainer() {
        return this.to('DeleteContainer');
    }
    /**
     * Grants permission to delete the access policy of a container
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteContainerPolicy.html
     */
    toDeleteContainerPolicy() {
        return this.to('DeleteContainerPolicy');
    }
    /**
     * Grants permission to delete the CORS policy from a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteCorsPolicy.html
     */
    toDeleteCorsPolicy() {
        return this.to('DeleteCorsPolicy');
    }
    /**
     * Grants permission to delete the lifecycle policy from a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteLifecyclePolicy.html
     */
    toDeleteLifecyclePolicy() {
        return this.to('DeleteLifecyclePolicy');
    }
    /**
     * Grants permission to delete the metric policy from a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DeleteMetricPolicy.html
     */
    toDeleteMetricPolicy() {
        return this.to('DeleteMetricPolicy');
    }
    /**
     * Grants permission to delete an object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DeleteObject.html
     */
    toDeleteObject() {
        return this.to('DeleteObject');
    }
    /**
     * Grants permission to retrieve details on a container
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_DescribeContainer.html
     */
    toDescribeContainer() {
        return this.to('DescribeContainer');
    }
    /**
     * Grants permission to retrieve metadata for an object
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_DescribeObject.html
     */
    toDescribeObject() {
        return this.to('DescribeObject');
    }
    /**
     * Grants permission to retrieve the access policy of a container
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetContainerPolicy.html
     */
    toGetContainerPolicy() {
        return this.to('GetContainerPolicy');
    }
    /**
     * Grants permission to retrieve the CORS policy of a container
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetCorsPolicy.html
     */
    toGetCorsPolicy() {
        return this.to('GetCorsPolicy');
    }
    /**
     * Grants permission to retrieve the lifecycle policy that is assigned to a container
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetLifecyclePolicy.html
     */
    toGetLifecyclePolicy() {
        return this.to('GetLifecyclePolicy');
    }
    /**
     * Grants permission to retrieve the metric policy that is assigned to a container
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_GetMetricPolicy.html
     */
    toGetMetricPolicy() {
        return this.to('GetMetricPolicy');
    }
    /**
     * Grants permission to retrieve an object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_GetObject.html
     */
    toGetObject() {
        return this.to('GetObject');
    }
    /**
     * Grants permission to retrieve a list of containers in the current account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListContainers.html
     */
    toListContainers() {
        return this.to('ListContainers');
    }
    /**
     * Grants permission to retrieve a list of objects and subfolders that are stored in a folder
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_ListItems.html
     */
    toListItems() {
        return this.to('ListItems');
    }
    /**
     * Grants permission to list tags on a container
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or replace the access policy of a container
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutContainerPolicy.html
     */
    toPutContainerPolicy() {
        return this.to('PutContainerPolicy');
    }
    /**
     * Grants permission to add or modify the CORS policy of a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutCorsPolicy.html
     */
    toPutCorsPolicy() {
        return this.to('PutCorsPolicy');
    }
    /**
     * Grants permission to add or modify the lifecycle policy that is assigned to a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutLifecyclePolicy.html
     */
    toPutLifecyclePolicy() {
        return this.to('PutLifecyclePolicy');
    }
    /**
     * Grants permission to add or modify the metric policy that is assigned to a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_PutMetricPolicy.html
     */
    toPutMetricPolicy() {
        return this.to('PutMetricPolicy');
    }
    /**
     * Grants permission to upload an object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_objstore_PutObject.html
     */
    toPutObject() {
        return this.to('PutObject');
    }
    /**
     * Grants permission to start access logging on a container
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StartAccessLogging.html
     */
    toStartAccessLogging() {
        return this.to('StartAccessLogging');
    }
    /**
     * Grants permission to stop access logging on a container
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_StopAccessLogging.html
     */
    toStopAccessLogging() {
        return this.to('StopAccessLogging');
    }
    /**
     * Grants permission to add tags to a container
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a container
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediastore/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Adds a resource of type container to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/containers.html
     *
     * @param containerName - Identifier for the containerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContainer(containerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediastore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:container/${containerName}`);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/objects.html
     *
     * @param containerName - Identifier for the containerName.
     * @param objectPath - Identifier for the objectPath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onObject(containerName, objectPath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediastore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:container/${containerName}/${objectPath}`);
    }
    /**
     * Adds a resource of type folder to the statement
     *
     * https://docs.aws.amazon.com/mediastore/latest/ug/folders.html
     *
     * @param containerName - Identifier for the containerName.
     * @param folderPath - Identifier for the folderPath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFolder(containerName, folderPath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediastore:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:container/${containerName}/${folderPath}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateContainer()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - container
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateContainer()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mediastore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediastore.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediastore';
        this.accessLevelList = {
            Write: [
                'CreateContainer',
                'DeleteContainer',
                'DeleteCorsPolicy',
                'DeleteLifecyclePolicy',
                'DeleteMetricPolicy',
                'DeleteObject',
                'PutCorsPolicy',
                'PutLifecyclePolicy',
                'PutMetricPolicy',
                'PutObject',
                'StartAccessLogging',
                'StopAccessLogging'
            ],
            'Permissions management': [
                'DeleteContainerPolicy',
                'PutContainerPolicy'
            ],
            List: [
                'DescribeContainer',
                'DescribeObject',
                'ListContainers',
                'ListItems'
            ],
            Read: [
                'GetContainerPolicy',
                'GetCorsPolicy',
                'GetLifecyclePolicy',
                'GetMetricPolicy',
                'GetObject',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mediastore = Mediastore;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxlbWVudGFsbWVkaWFzdG9yZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImVsZW1lbnRhbG1lZGlhc3RvcmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSx3QkFBZTtJQUc3Qzs7Ozs7Ozs7OztPQVVHO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBeUNEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFdBQVcsQ0FBQyxhQUFxQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxjQUFlLGFBQWMsRUFBRSxDQUFDLENBQUM7SUFDOUssQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxRQUFRLENBQUMsYUFBcUIsRUFBRSxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzlHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGVBQWdCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxjQUFlLGFBQWMsSUFBSyxVQUFXLEVBQUUsQ0FBQyxDQUFDO0lBQzlMLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksUUFBUSxDQUFDLGFBQXFCLEVBQUUsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsY0FBZSxhQUFjLElBQUssVUFBVyxFQUFFLENBQUMsQ0FBQztJQUM5TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7O09BV0c7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUEzYlIsa0JBQWEsR0FBRyxZQUFZLENBQUM7UUE4UzFCLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxpQkFBaUI7Z0JBQ2pCLGlCQUFpQjtnQkFDakIsa0JBQWtCO2dCQUNsQix1QkFBdUI7Z0JBQ3ZCLG9CQUFvQjtnQkFDcEIsY0FBYztnQkFDZCxlQUFlO2dCQUNmLG9CQUFvQjtnQkFDcEIsaUJBQWlCO2dCQUNqQixXQUFXO2dCQUNYLG9CQUFvQjtnQkFDcEIsbUJBQW1CO2FBQ3BCO1lBQ0Qsd0JBQXdCLEVBQUU7Z0JBQ3hCLHVCQUF1QjtnQkFDdkIsb0JBQW9CO2FBQ3JCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLG1CQUFtQjtnQkFDbkIsZ0JBQWdCO2dCQUNoQixnQkFBZ0I7Z0JBQ2hCLFdBQVc7YUFDWjtZQUNELElBQUksRUFBRTtnQkFDSixvQkFBb0I7Z0JBQ3BCLGVBQWU7Z0JBQ2Ysb0JBQW9CO2dCQUNwQixpQkFBaUI7Z0JBQ2pCLFdBQVc7Z0JBQ1gscUJBQXFCO2FBQ3RCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUF5R0YsQ0FBQztDQUNGO0FBOWJELGdDQThiQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVkaWFzdG9yZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzZWxlbWVudGFsbWVkaWFzdG9yZS5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIE1lZGlhc3RvcmUgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdtZWRpYXN0b3JlJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlQ29udGFpbmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNvbnRhaW5lcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ29udGFpbmVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlQ29udGFpbmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNvbnRhaW5lcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ29udGFpbmVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBhY2Nlc3MgcG9saWN5IG9mIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb250YWluZXJQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29udGFpbmVyUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDb250YWluZXJQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIENPUlMgcG9saWN5IGZyb20gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVDb3JzUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNvcnNQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNvcnNQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIGxpZmVjeWNsZSBwb2xpY3kgZnJvbSBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZUxpZmVjeWNsZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUxpZmVjeWNsZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgbWV0cmljIHBvbGljeSBmcm9tIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlTWV0cmljUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU1ldHJpY1BvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTWV0cmljUG9saWN5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX29ianN0b3JlX0RlbGV0ZU9iamVjdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVPYmplY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZU9iamVjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGRldGFpbHMgb24gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29udGFpbmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ29udGFpbmVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUNvbnRhaW5lcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIG1ldGFkYXRhIGZvciBhbiBvYmplY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX29ianN0b3JlX0Rlc2NyaWJlT2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlT2JqZWN0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU9iamVjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBhY2Nlc3MgcG9saWN5IG9mIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9HZXRDb250YWluZXJQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29udGFpbmVyUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDb250YWluZXJQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgQ09SUyBwb2xpY3kgb2YgYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldENvcnNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q29yc1BvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0Q29yc1BvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBsaWZlY3ljbGUgcG9saWN5IHRoYXQgaXMgYXNzaWduZWQgdG8gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldExpZmVjeWNsZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldExpZmVjeWNsZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIHRoZSBtZXRyaWMgcG9saWN5IHRoYXQgaXMgYXNzaWduZWQgdG8gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldE1ldHJpY1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNZXRyaWNQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldE1ldHJpY1BvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIG9iamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfb2Jqc3RvcmVfR2V0T2JqZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE9iamVjdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0T2JqZWN0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGNvbnRhaW5lcnMgaW4gdGhlIGN1cnJlbnQgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdENvbnRhaW5lcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENvbnRhaW5lcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RDb250YWluZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIG9iamVjdHMgYW5kIHN1YmZvbGRlcnMgdGhhdCBhcmUgc3RvcmVkIGluIGEgZm9sZGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9MaXN0SXRlbXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEl0ZW1zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SXRlbXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3Mgb24gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFRhZ3NGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBvciByZXBsYWNlIHRoZSBhY2Nlc3MgcG9saWN5IG9mIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRDb250YWluZXJQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0Q29udGFpbmVyUG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRDb250YWluZXJQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgb3IgbW9kaWZ5IHRoZSBDT1JTIHBvbGljeSBvZiBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1B1dENvcnNQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0Q29yc1BvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0Q29yc1BvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciBtb2RpZnkgdGhlIGxpZmVjeWNsZSBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1B1dExpZmVjeWNsZVBvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRMaWZlY3ljbGVQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dExpZmVjeWNsZVBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBvciBtb2RpZnkgdGhlIG1ldHJpYyBwb2xpY3kgdGhhdCBpcyBhc3NpZ25lZCB0byBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1B1dE1ldHJpY1BvbGljeS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRNZXRyaWNQb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dE1ldHJpY1BvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwbG9hZCBhbiBvYmplY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9vYmpzdG9yZV9QdXRPYmplY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0T2JqZWN0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRPYmplY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhY2Nlc3MgbG9nZ2luZyBvbiBhIGNvbnRhaW5lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpQYXNzUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9TdGFydEFjY2Vzc0xvZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRBY2Nlc3NMb2dnaW5nKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydEFjY2Vzc0xvZ2dpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdG9wIGFjY2VzcyBsb2dnaW5nIG9uIGEgY29udGFpbmVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfU3RvcEFjY2Vzc0xvZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RvcEFjY2Vzc0xvZ2dpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BBY2Nlc3NMb2dnaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHRhZ3MgdG8gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYSBjb250YWluZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYXN0b3JlL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQ3JlYXRlQ29udGFpbmVyJyxcbiAgICAgICdEZWxldGVDb250YWluZXInLFxuICAgICAgJ0RlbGV0ZUNvcnNQb2xpY3knLFxuICAgICAgJ0RlbGV0ZUxpZmVjeWNsZVBvbGljeScsXG4gICAgICAnRGVsZXRlTWV0cmljUG9saWN5JyxcbiAgICAgICdEZWxldGVPYmplY3QnLFxuICAgICAgJ1B1dENvcnNQb2xpY3knLFxuICAgICAgJ1B1dExpZmVjeWNsZVBvbGljeScsXG4gICAgICAnUHV0TWV0cmljUG9saWN5JyxcbiAgICAgICdQdXRPYmplY3QnLFxuICAgICAgJ1N0YXJ0QWNjZXNzTG9nZ2luZycsXG4gICAgICAnU3RvcEFjY2Vzc0xvZ2dpbmcnXG4gICAgXSxcbiAgICAnUGVybWlzc2lvbnMgbWFuYWdlbWVudCc6IFtcbiAgICAgICdEZWxldGVDb250YWluZXJQb2xpY3knLFxuICAgICAgJ1B1dENvbnRhaW5lclBvbGljeSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdEZXNjcmliZUNvbnRhaW5lcicsXG4gICAgICAnRGVzY3JpYmVPYmplY3QnLFxuICAgICAgJ0xpc3RDb250YWluZXJzJyxcbiAgICAgICdMaXN0SXRlbXMnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0Q29udGFpbmVyUG9saWN5JyxcbiAgICAgICdHZXRDb3JzUG9saWN5JyxcbiAgICAgICdHZXRMaWZlY3ljbGVQb2xpY3knLFxuICAgICAgJ0dldE1ldHJpY1BvbGljeScsXG4gICAgICAnR2V0T2JqZWN0JyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY29udGFpbmVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L3VnL2NvbnRhaW5lcnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29udGFpbmVyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjb250YWluZXJOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25Db250YWluZXIoY29udGFpbmVyTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhc3RvcmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNvbnRhaW5lci8keyBjb250YWluZXJOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBvYmplY3QgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFzdG9yZS9sYXRlc3QvdWcvb2JqZWN0cy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjb250YWluZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGNvbnRhaW5lck5hbWUuXG4gICAqIEBwYXJhbSBvYmplY3RQYXRoIC0gSWRlbnRpZmllciBmb3IgdGhlIG9iamVjdFBhdGguXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uT2JqZWN0KGNvbnRhaW5lck5hbWU6IHN0cmluZywgb2JqZWN0UGF0aDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhc3RvcmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNvbnRhaW5lci8keyBjb250YWluZXJOYW1lIH0vJHsgb2JqZWN0UGF0aCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZm9sZGVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhc3RvcmUvbGF0ZXN0L3VnL2ZvbGRlcnMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY29udGFpbmVyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjb250YWluZXJOYW1lLlxuICAgKiBAcGFyYW0gZm9sZGVyUGF0aCAtIElkZW50aWZpZXIgZm9yIHRoZSBmb2xkZXJQYXRoLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkZvbGRlcihjb250YWluZXJOYW1lOiBzdHJpbmcsIGZvbGRlclBhdGg6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTptZWRpYXN0b3JlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjb250YWluZXIvJHsgY29udGFpbmVyTmFtZSB9LyR7IGZvbGRlclBhdGggfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQ29udGFpbmVyKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFncyBhc3NvY2lhdGVkIHdpdGggdGhlIHJlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy1yZXNvdXJjZXRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIHJlc291cmNlIHR5cGVzOlxuICAgKiAtIGNvbnRhaW5lclxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgdGFnIGtleXMgdGhhdCBhcmUgcGFzc2VkIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDb250YWluZXIoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVkaWFzdG9yZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzZWxlbWVudGFsbWVkaWFzdG9yZS5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=