"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElementalSupportCases = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elemental-support-cases](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalsupportcases.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ElementalSupportCases extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a comment to a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toAddCaseComment() {
        return this.to('AddCaseComment');
    }
    /**
     * Grants permission to verify whether the caller has the permissions to perform support case operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toCheckCasePermission() {
        return this.to('CheckCasePermission');
    }
    /**
     * Grants permission to complete a multipart file upload to a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toCompleteMultipartUpload() {
        return this.to('CompleteMultipartUpload');
    }
    /**
     * Grants permission to create a support case
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toCreateCase() {
        return this.to('CreateCase');
    }
    /**
     * Grants permission to create a cli command to allow a file upload to a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toCreateS3CLIUploadCommand() {
        return this.to('CreateS3CLIUploadCommand');
    }
    /**
     * Grants permission to download a file from a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toCreateS3DownloadUrl() {
        return this.to('CreateS3DownloadUrl');
    }
    /**
     * Grants permission to describe a support case in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toGetCase() {
        return this.to('GetCase');
    }
    /**
     * Grants permission to verify whether the caller has the permissions to perform support case operations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toGetCasePermission() {
        return this.to('GetCasePermission');
    }
    /**
     * Grants permission to list the support cases in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toGetCases() {
        return this.to('GetCases');
    }
    /**
     * Grants permission to retrieve cached case user data for use in the Console
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toGetUICache() {
        return this.to('GetUICache');
    }
    /**
     * Grants permission to list tags on a support case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toListTagsForCase() {
        return this.to('ListTagsForCase');
    }
    /**
     * Grants permission to start a multipart file upload to a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toStartMultipartUpload() {
        return this.to('StartMultipartUpload');
    }
    /**
     * Grants permission to add a tag on a support case
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toTagCase() {
        return this.to('TagCase');
    }
    /**
     * Grants permission to remove a tag on a support case
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toUntagCase() {
        return this.to('UntagCase');
    }
    /**
     * Grants permission to update a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toUpdateCase() {
        return this.to('UpdateCase');
    }
    /**
     * Grants permission to update a support case status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toUpdateCaseStatus() {
        return this.to('UpdateCaseStatus');
    }
    /**
     * Grants permission to update a multipart file upload to a support case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elemental-appliances-software
     */
    toUpdateMultipartUpload() {
        return this.to('UpdateMultipartUpload');
    }
    /**
     * Adds a resource of type case to the statement
     *
     * https://docs.aws.amazon.com/elemental-appliances-software/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCase(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elemental-support-cases::${account ?? this.defaultAccount}:case/${resourceId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toTagCase()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - case
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toTagCase()
     * - .toUntagCase()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [elemental-support-cases](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalsupportcases.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elemental-support-cases';
        this.accessLevelList = {
            Write: [
                'AddCaseComment',
                'CheckCasePermission',
                'CompleteMultipartUpload',
                'CreateCase',
                'CreateS3CLIUploadCommand',
                'CreateS3DownloadUrl',
                'StartMultipartUpload',
                'UpdateCase',
                'UpdateCaseStatus',
                'UpdateMultipartUpload'
            ],
            Read: [
                'GetCase',
                'GetCasePermission',
                'GetCases',
                'GetUICache',
                'ListTagsForCase'
            ],
            Tagging: [
                'TagCase',
                'UntagCase'
            ]
        };
    }
}
exports.ElementalSupportCases = ElementalSupportCases;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxlbWVudGFsc3VwcG9ydGNhc2VzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZWxlbWVudGFsc3VwcG9ydGNhc2VzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxxQkFBc0IsU0FBUSx3QkFBZTtJQUd4RDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDN0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFNBQVM7UUFDZCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQTRCRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLE1BQU0sQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsU0FBa0I7UUFDcEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsNkJBQThCLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxTQUFVLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDbEosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7T0FHRztJQUNILFlBQVksS0FBZ0M7UUFDMUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBelNSLGtCQUFhLEdBQUcseUJBQXlCLENBQUM7UUF3TXZDLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCxnQkFBZ0I7Z0JBQ2hCLHFCQUFxQjtnQkFDckIseUJBQXlCO2dCQUN6QixZQUFZO2dCQUNaLDBCQUEwQjtnQkFDMUIscUJBQXFCO2dCQUNyQixzQkFBc0I7Z0JBQ3RCLFlBQVk7Z0JBQ1osa0JBQWtCO2dCQUNsQix1QkFBdUI7YUFDeEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osU0FBUztnQkFDVCxtQkFBbUI7Z0JBQ25CLFVBQVU7Z0JBQ1YsWUFBWTtnQkFDWixpQkFBaUI7YUFDbEI7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsU0FBUztnQkFDVCxXQUFXO2FBQ1o7U0FDRixDQUFDO0lBMEVGLENBQUM7Q0FDRjtBQTVTRCxzREE0U0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCwgT3BlcmF0b3IgfSBmcm9tICcuLi8uLi9zaGFyZWQnO1xuaW1wb3J0IHsgYXdzX2lhbSBhcyBpYW0gfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcblxuLyoqXG4gKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2VsZW1lbnRhbC1zdXBwb3J0LWNhc2VzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NlbGVtZW50YWxzdXBwb3J0Y2FzZXMuaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBFbGVtZW50YWxTdXBwb3J0Q2FzZXMgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdlbGVtZW50YWwtc3VwcG9ydC1jYXNlcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCBhIGNvbW1lbnQgdG8gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0FkZENhc2VDb21tZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBZGRDYXNlQ29tbWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZlcmlmeSB3aGV0aGVyIHRoZSBjYWxsZXIgaGFzIHRoZSBwZXJtaXNzaW9ucyB0byBwZXJmb3JtIHN1cHBvcnQgY2FzZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsZW1lbnRhbC1hcHBsaWFuY2VzLXNvZnR3YXJlXG4gICAqL1xuICBwdWJsaWMgdG9DaGVja0Nhc2VQZXJtaXNzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDaGVja0Nhc2VQZXJtaXNzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY29tcGxldGUgYSBtdWx0aXBhcnQgZmlsZSB1cGxvYWQgdG8gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0NvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb21wbGV0ZU11bHRpcGFydFVwbG9hZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHN1cHBvcnQgY2FzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ2FzZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ2FzZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNsaSBjb21tYW5kIHRvIGFsbG93IGEgZmlsZSB1cGxvYWQgdG8gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVMzQ0xJVXBsb2FkQ29tbWFuZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUzNDTElVcGxvYWRDb21tYW5kJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZG93bmxvYWQgYSBmaWxlIGZyb20gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVMzRG93bmxvYWRVcmwoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVMzRG93bmxvYWRVcmwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhIHN1cHBvcnQgY2FzZSBpbiB5b3VyIGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZVxuICAgKi9cbiAgcHVibGljIHRvR2V0Q2FzZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0Q2FzZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHZlcmlmeSB3aGV0aGVyIHRoZSBjYWxsZXIgaGFzIHRoZSBwZXJtaXNzaW9ucyB0byBwZXJmb3JtIHN1cHBvcnQgY2FzZSBvcGVyYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0dldENhc2VQZXJtaXNzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDYXNlUGVybWlzc2lvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHN1cHBvcnQgY2FzZXMgaW4geW91ciBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0dldENhc2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDYXNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGNhY2hlZCBjYXNlIHVzZXIgZGF0YSBmb3IgdXNlIGluIHRoZSBDb25zb2xlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b0dldFVJQ2FjaGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFVJQ2FjaGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3Mgb24gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvckNhc2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIG11bHRpcGFydCBmaWxlIHVwbG9hZCB0byBhIHN1cHBvcnQgY2FzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZVxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRNdWx0aXBhcnRVcGxvYWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0YXJ0TXVsdGlwYXJ0VXBsb2FkJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIGEgdGFnIG9uIGEgc3VwcG9ydCBjYXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b1RhZ0Nhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ0Nhc2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZW1vdmUgYSB0YWcgb24gYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZVxuICAgKi9cbiAgcHVibGljIHRvVW50YWdDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ0Nhc2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBzdXBwb3J0IGNhc2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxlbWVudGFsLWFwcGxpYW5jZXMtc29mdHdhcmVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNhc2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNhc2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBzdXBwb3J0IGNhc2Ugc3RhdHVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsZW1lbnRhbC1hcHBsaWFuY2VzLXNvZnR3YXJlXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDYXNlU3RhdHVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVDYXNlU3RhdHVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgbXVsdGlwYXJ0IGZpbGUgdXBsb2FkIHRvIGEgc3VwcG9ydCBjYXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsZW1lbnRhbC1hcHBsaWFuY2VzLXNvZnR3YXJlXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNdWx0aXBhcnRVcGxvYWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZU11bHRpcGFydFVwbG9hZCcpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFdyaXRlOiBbXG4gICAgICAnQWRkQ2FzZUNvbW1lbnQnLFxuICAgICAgJ0NoZWNrQ2FzZVBlcm1pc3Npb24nLFxuICAgICAgJ0NvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAgICdDcmVhdGVDYXNlJyxcbiAgICAgICdDcmVhdGVTM0NMSVVwbG9hZENvbW1hbmQnLFxuICAgICAgJ0NyZWF0ZVMzRG93bmxvYWRVcmwnLFxuICAgICAgJ1N0YXJ0TXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAgICdVcGRhdGVDYXNlJyxcbiAgICAgICdVcGRhdGVDYXNlU3RhdHVzJyxcbiAgICAgICdVcGRhdGVNdWx0aXBhcnRVcGxvYWQnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0Q2FzZScsXG4gICAgICAnR2V0Q2FzZVBlcm1pc3Npb24nLFxuICAgICAgJ0dldENhc2VzJyxcbiAgICAgICdHZXRVSUNhY2hlJyxcbiAgICAgICdMaXN0VGFnc0ZvckNhc2UnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnQ2FzZScsXG4gICAgICAnVW50YWdDYXNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2FzZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGVtZW50YWwtYXBwbGlhbmNlcy1zb2Z0d2FyZS9cbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNhc2UocmVzb3VyY2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbGVtZW50YWwtc3VwcG9ydC1jYXNlczo6JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y2FzZS8keyByZXNvdXJjZUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0YWdzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVxdWVzdHRhZ1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQ2FzZSgpXG4gICAqIC0gLnRvVGFnQ2FzZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlc291cmNldGFnXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gY2FzZVxuICAgKlxuICAgKiBAcGFyYW0gdGFnS2V5IFRoZSB0YWcga2V5IHRvIGNoZWNrXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzUmVzb3VyY2VUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVzb3VyY2VUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNhc2UoKVxuICAgKiAtIC50b1RhZ0Nhc2UoKVxuICAgKiAtIC50b1VudGFnQ2FzZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2VsZW1lbnRhbC1zdXBwb3J0LWNhc2VzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NlbGVtZW50YWxzdXBwb3J0Y2FzZXMuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19