"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SmsVoiceV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sms-voice-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsmsandvoicev2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SmsVoiceV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an origination phone number or sender ID to a pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_AssociateOriginationIdentity.html
     */
    toAssociateOriginationIdentity() {
        return this.to('AssociateOriginationIdentity');
    }
    /**
     * Grants permission to associate a protect configuration to a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_AssociateProtectConfiguration.html
     */
    toAssociateProtectConfiguration() {
        return this.to('AssociateProtectConfiguration');
    }
    /**
     * Grants permission to create a configuration set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateConfigurationSet.html
     */
    toCreateConfigurationSet() {
        return this.to('CreateConfigurationSet');
    }
    /**
     * Grants permission to create an event destination within a configuration set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateEventDestination.html
     */
    toCreateEventDestination() {
        return this.to('CreateEventDestination');
    }
    /**
     * Grants permission to create an opt-out list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateOptOutList.html
     */
    toCreateOptOutList() {
        return this.to('CreateOptOutList');
    }
    /**
     * Grants permission to create a pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreatePool.html
     */
    toCreatePool() {
        return this.to('CreatePool');
    }
    /**
     * Grants permission to create a protect configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateProtectConfiguration.html
     */
    toCreateProtectConfiguration() {
        return this.to('CreateProtectConfiguration');
    }
    /**
     * Grants permission to create a registration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateRegistration.html
     */
    toCreateRegistration() {
        return this.to('CreateRegistration');
    }
    /**
     * Grants permission to associate a registration with a phone number or another registration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateRegistrationAssociation.html
     */
    toCreateRegistrationAssociation() {
        return this.to('CreateRegistrationAssociation');
    }
    /**
     * Grants permission to create a registration attachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateRegistrationAttachment.html
     */
    toCreateRegistrationAttachment() {
        return this.to('CreateRegistrationAttachment');
    }
    /**
     * Grants permission to create a registration version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateRegistrationVersion.html
     */
    toCreateRegistrationVersion() {
        return this.to('CreateRegistrationVersion');
    }
    /**
     * Grants permission to create a verified destination number
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateVerifiedDestinationNumber.html
     */
    toCreateVerifiedDestinationNumber() {
        return this.to('CreateVerifiedDestinationNumber');
    }
    /**
     * Grants permission to delete the account default protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteAccountDefaultProtectConfiguration.html
     */
    toDeleteAccountDefaultProtectConfiguration() {
        return this.to('DeleteAccountDefaultProtectConfiguration');
    }
    /**
     * Grants permission to delete a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteConfigurationSet.html
     */
    toDeleteConfigurationSet() {
        return this.to('DeleteConfigurationSet');
    }
    /**
     * Grants permission to delete the default message type for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteDefaultMessageType.html
     */
    toDeleteDefaultMessageType() {
        return this.to('DeleteDefaultMessageType');
    }
    /**
     * Grants permission to delete the default sender ID for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteDefaultSenderId.html
     */
    toDeleteDefaultSenderId() {
        return this.to('DeleteDefaultSenderId');
    }
    /**
     * Grants permission to delete an event destination within a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteEventDestination.html
     */
    toDeleteEventDestination() {
        return this.to('DeleteEventDestination');
    }
    /**
     * Grants permission to delete a keyword for a pool or origination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteKeyword.html
     */
    toDeleteKeyword() {
        return this.to('DeleteKeyword');
    }
    /**
     * Grants permission to delete an override for your account's media messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteMediaMessageSpendLimitOverride.html
     */
    toDeleteMediaMessageSpendLimitOverride() {
        return this.to('DeleteMediaMessageSpendLimitOverride');
    }
    /**
     * Grants permission to delete an opt-out list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteOptOutList.html
     */
    toDeleteOptOutList() {
        return this.to('DeleteOptOutList');
    }
    /**
     * Grants permission to delete a destination phone number from an opt-out list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteOptedOutNumber.html
     */
    toDeleteOptedOutNumber() {
        return this.to('DeleteOptedOutNumber');
    }
    /**
     * Grants permission to delete a pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeletePool.html
     */
    toDeletePool() {
        return this.to('DeletePool');
    }
    /**
     * Grants permission to delete a protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteProtectConfiguration.html
     */
    toDeleteProtectConfiguration() {
        return this.to('DeleteProtectConfiguration');
    }
    /**
     * Grants permission to delete a phone number override for a protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteProtectConfigurationRuleSetNumberOverride.html
     */
    toDeleteProtectConfigurationRuleSetNumberOverride() {
        return this.to('DeleteProtectConfigurationRuleSetNumberOverride');
    }
    /**
     * Grants permission to delete a registration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteRegistration.html
     */
    toDeleteRegistration() {
        return this.to('DeleteRegistration');
    }
    /**
     * Grants permission to delete a registration attachment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteRegistrationAttachment.html
     */
    toDeleteRegistrationAttachment() {
        return this.to('DeleteRegistrationAttachment');
    }
    /**
     * Grants permission to delete an optional registration field value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteRegistrationFieldValue.html
     */
    toDeleteRegistrationFieldValue() {
        return this.to('DeleteRegistrationFieldValue');
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete an override for your account's text messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteTextMessageSpendLimitOverride.html
     */
    toDeleteTextMessageSpendLimitOverride() {
        return this.to('DeleteTextMessageSpendLimitOverride');
    }
    /**
     * Grants permission to delete a verified destination number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteVerifiedDestinationNumber.html
     */
    toDeleteVerifiedDestinationNumber() {
        return this.to('DeleteVerifiedDestinationNumber');
    }
    /**
     * Grants permission to delete an override for your account's voice messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DeleteVoiceMessageSpendLimitOverride.html
     */
    toDeleteVoiceMessageSpendLimitOverride() {
        return this.to('DeleteVoiceMessageSpendLimitOverride');
    }
    /**
     * Grants permission to describe the attributes of your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        return this.to('DescribeAccountAttributes');
    }
    /**
     * Grants permission to describe the service quotas for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        return this.to('DescribeAccountLimits');
    }
    /**
     * Grants permission to describe the configuration sets in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeConfigurationSets.html
     */
    toDescribeConfigurationSets() {
        return this.to('DescribeConfigurationSets');
    }
    /**
     * Grants permission to describe the keywords for a pool or origination phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeKeywords.html
     */
    toDescribeKeywords() {
        return this.to('DescribeKeywords');
    }
    /**
     * Grants permission to describe the opt-out lists in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeOptOutLists.html
     */
    toDescribeOptOutLists() {
        return this.to('DescribeOptOutLists');
    }
    /**
     * Grants permission to describe the destination phone numbers in an opt-out list
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeOptedOutNumbers.html
     */
    toDescribeOptedOutNumbers() {
        return this.to('DescribeOptedOutNumbers');
    }
    /**
     * Grants permission to describe the origination phone numbers in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribePhoneNumbers.html
     */
    toDescribePhoneNumbers() {
        return this.to('DescribePhoneNumbers');
    }
    /**
     * Grants permission to describe the pools in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribePools.html
     */
    toDescribePools() {
        return this.to('DescribePools');
    }
    /**
     * Grants permission to describe the protect configurations in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeProtectConfigurations.html
     */
    toDescribeProtectConfigurations() {
        return this.to('DescribeProtectConfigurations');
    }
    /**
     * Grants permission to describe the registration attachments in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribePools.html
     */
    toDescribeRegistrationAttachments() {
        return this.to('DescribeRegistrationAttachments');
    }
    /**
     * Grants permission to describe the field definitions for a given registration type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationFieldDefinitions.html
     */
    toDescribeRegistrationFieldDefinitions() {
        return this.to('DescribeRegistrationFieldDefinitions');
    }
    /**
     * Grants permission to describe the field values for a given registration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationFieldValues.html
     */
    toDescribeRegistrationFieldValues() {
        return this.to('DescribeRegistrationFieldValues');
    }
    /**
     * Grants permission to describe the section definitions for a given registration type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationSectionDefinitions.html
     */
    toDescribeRegistrationSectionDefinitions() {
        return this.to('DescribeRegistrationSectionDefinitions');
    }
    /**
     * Grants permission to describe the registration types supported by the service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationTypeDefinitions.html
     */
    toDescribeRegistrationTypeDefinitions() {
        return this.to('DescribeRegistrationTypeDefinitions');
    }
    /**
     * Grants permission to describe the versions for a given registration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationVersions.html
     */
    toDescribeRegistrationVersions() {
        return this.to('DescribeRegistrationVersions');
    }
    /**
     * Grants permission to describe the registrations in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrations.html
     */
    toDescribeRegistrations() {
        return this.to('DescribeRegistrations');
    }
    /**
     * Grants permission to describe the sender IDs in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeSenderIds.html
     */
    toDescribeSenderIds() {
        return this.to('DescribeSenderIds');
    }
    /**
     * Grants permission to describe the monthly spend limits for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeSpendLimits.html
     */
    toDescribeSpendLimits() {
        return this.to('DescribeSpendLimits');
    }
    /**
     * Grants permission to describe the verified destination numbers in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeVerifiedDestinationNumbers.html
     */
    toDescribeVerifiedDestinationNumbers() {
        return this.to('DescribeVerifiedDestinationNumbers');
    }
    /**
     * Grants permission to disassociate an origination phone number or sender ID from a pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DisassociateOriginationIdentity.html
     */
    toDisassociateOriginationIdentity() {
        return this.to('DisassociateOriginationIdentity');
    }
    /**
     * Grants permission to disassociate a protect configuration from a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DisassociateProtectConfiguration.html
     */
    toDisassociateProtectConfiguration() {
        return this.to('DisassociateProtectConfiguration');
    }
    /**
     * Grants permission to discard the latest version of a given registration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DiscardRegistrationVersion.html
     */
    toDiscardRegistrationVersion() {
        return this.to('DiscardRegistrationVersion');
    }
    /**
     * Grants permission to get the country rule set for a protect configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_GetProtectConfigurationCountryRuleSet.html
     */
    toGetProtectConfigurationCountryRuleSet() {
        return this.to('GetProtectConfigurationCountryRuleSet');
    }
    /**
     * Grants permission to get a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list all origination phone numbers and sender IDs associated to a pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ListPoolOriginationIdentities.html
     */
    toListPoolOriginationIdentities() {
        return this.to('ListPoolOriginationIdentities');
    }
    /**
     * Grants permission to list all phone number overrides for a protect configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ListProtectConfigurationRuleSetNumberOverrides.html
     */
    toListProtectConfigurationRuleSetNumberOverrides() {
        return this.to('ListProtectConfigurationRuleSetNumberOverrides');
    }
    /**
     * Grants permission to list all resources associated to a registration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ListRegistrationAssociations.html
     */
    toListRegistrationAssociations() {
        return this.to('ListRegistrationAssociations');
    }
    /**
     * Grants permission to list the tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update a keyword for a pool or origination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutKeyword.html
     */
    toPutKeyword() {
        return this.to('PutKeyword');
    }
    /**
     * Grants permission to put feedback for a text, voice, or media message
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutMessageFeedback.html
     */
    toPutMessageFeedback() {
        return this.to('PutMessageFeedback');
    }
    /**
     * Grants permission to put a destination phone number into an opt-out list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutOptedOutNumber.html
     */
    toPutOptedOutNumber() {
        return this.to('PutOptedOutNumber');
    }
    /**
     * Grants permission to put a phone number override for a protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutProtectConfigurationRuleSetNumberOverride.html
     */
    toPutProtectConfigurationRuleSetNumberOverride() {
        return this.to('PutProtectConfigurationRuleSetNumberOverride');
    }
    /**
     * Grants permission to put a registration field value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutRegistrationFieldValue.html
     */
    toPutRegistrationFieldValue() {
        return this.to('PutRegistrationFieldValue');
    }
    /**
     * Grants permission to put a resource policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to release an origination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ReleasePhoneNumber.html
     */
    toReleasePhoneNumber() {
        return this.to('ReleasePhoneNumber');
    }
    /**
     * Grants permission to release a sender ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_ReleaseSenderId.html
     */
    toReleaseSenderId() {
        return this.to('ReleaseSenderId');
    }
    /**
     * Grants permission to request an origination phone number
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:AssociateOriginationIdentity
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_RequestPhoneNumber.html
     */
    toRequestPhoneNumber() {
        return this.to('RequestPhoneNumber');
    }
    /**
     * Grants permission to request an unregistered sender ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sms-voice:TagResource
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_RequestSenderId.html
     */
    toRequestSenderId() {
        return this.to('RequestSenderId');
    }
    /**
     * Grants permission to send a text or voice message containing a verification code to a destination phone number
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sms-voice:SendTextMessage
     * - sms-voice:SendVoiceMessage
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SendDestinationNumberVerificationCode.html
     */
    toSendDestinationNumberVerificationCode() {
        return this.to('SendDestinationNumberVerificationCode');
    }
    /**
     * Grants permission to send a media message to a destination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SendMediaMessage.html
     */
    toSendMediaMessage() {
        return this.to('SendMediaMessage');
    }
    /**
     * Grants permission to send a text message to a destination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SendTextMessage.html
     */
    toSendTextMessage() {
        return this.to('SendTextMessage');
    }
    /**
     * Grants permission to send a voice message to a destination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SendVoiceMessage.html
     */
    toSendVoiceMessage() {
        return this.to('SendVoiceMessage');
    }
    /**
     * Grants permission to set a default protect configuration for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetAccountDefaultProtectConfiguration.html
     */
    toSetAccountDefaultProtectConfiguration() {
        return this.to('SetAccountDefaultProtectConfiguration');
    }
    /**
     * Grants permission to set the default message feedback for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetDefaultMessageFeedbackEnabled.html
     */
    toSetDefaultMessageFeedbackEnabled() {
        return this.to('SetDefaultMessageFeedbackEnabled');
    }
    /**
     * Grants permission to set the default message type for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetDefaultMessageType.html
     */
    toSetDefaultMessageType() {
        return this.to('SetDefaultMessageType');
    }
    /**
     * Grants permission to set the default sender ID for a configuration set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetDefaultSenderId.html
     */
    toSetDefaultSenderId() {
        return this.to('SetDefaultSenderId');
    }
    /**
     * Grants permission to set an override for your account's media messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetMediaMessageSpendLimitOverride.html
     */
    toSetMediaMessageSpendLimitOverride() {
        return this.to('SetMediaMessageSpendLimitOverride');
    }
    /**
     * Grants permission to set an override for your account's text messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetTextMessageSpendLimitOverride.html
     */
    toSetTextMessageSpendLimitOverride() {
        return this.to('SetTextMessageSpendLimitOverride');
    }
    /**
     * Grants permission to set an override for your account's voice messaging monthly spend limit
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SetVoiceMessageSpendLimitOverride.html
     */
    toSetVoiceMessageSpendLimitOverride() {
        return this.to('SetVoiceMessageSpendLimitOverride');
    }
    /**
     * Grants permission to submit the latest version of a given registration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_SubmitRegistrationVersion.html
     */
    toSubmitRegistrationVersion() {
        return this.to('SubmitRegistrationVersion');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an event destination within a configuration set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdateEventDestination.html
     */
    toUpdateEventDestination() {
        return this.to('UpdateEventDestination');
    }
    /**
     * Grants permission to update an origination phone number's configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdatePhoneNumber.html
     */
    toUpdatePhoneNumber() {
        return this.to('UpdatePhoneNumber');
    }
    /**
     * Grants permission to update a pool's configuration
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdatePool.html
     */
    toUpdatePool() {
        return this.to('UpdatePool');
    }
    /**
     * Grants permission to update a protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdateProtectConfiguration.html
     */
    toUpdateProtectConfiguration() {
        return this.to('UpdateProtectConfiguration');
    }
    /**
     * Grants permission to update a country rule set for a protect configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdateProtectConfigurationCountryRuleSet.html
     */
    toUpdateProtectConfigurationCountryRuleSet() {
        return this.to('UpdateProtectConfigurationCountryRuleSet');
    }
    /**
     * Grants permission to update a sender ID's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_UpdateSenderId.html
     */
    toUpdateSenderId() {
        return this.to('UpdateSenderId');
    }
    /**
     * Grants permission to verify a destination phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_VerifyDestinationNumber.html
     */
    toVerifyDestinationNumber() {
        return this.to('VerifyDestinationNumber');
    }
    /**
     * Adds a resource of type ConfigurationSet to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateConfigurationSet.html
     *
     * @param configurationSetName - Identifier for the configurationSetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurationSet(configurationSetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration-set/${configurationSetName}`);
    }
    /**
     * Adds a resource of type OptOutList to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateOptOutList.html
     *
     * @param optOutListName - Identifier for the optOutListName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOptOutList(optOutListName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:opt-out-list/${optOutListName}`);
    }
    /**
     * Adds a resource of type PhoneNumber to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_RequestPhoneNumber.html
     *
     * @param phoneNumberId - Identifier for the phoneNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPhoneNumber(phoneNumberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:phone-number/${phoneNumberId}`);
    }
    /**
     * Adds a resource of type Pool to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreatePool.html
     *
     * @param poolId - Identifier for the poolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPool(poolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pool/${poolId}`);
    }
    /**
     * Adds a resource of type ProtectConfiguration to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_CreateProtectConfiguration.html
     *
     * @param protectConfigurationId - Identifier for the protectConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProtectConfiguration(protectConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:protect-configuration/${protectConfigurationId}`);
    }
    /**
     * Adds a resource of type SenderId to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeSenderIds.html
     *
     * @param senderId - Identifier for the senderId.
     * @param isoCountryCode - Identifier for the isoCountryCode.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSenderId(senderId, isoCountryCode, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sender-id/${senderId}/${isoCountryCode}`);
    }
    /**
     * Adds a resource of type Registration to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrations.html
     *
     * @param registrationId - Identifier for the registrationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegistration(registrationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:registration/${registrationId}`);
    }
    /**
     * Adds a resource of type RegistrationAttachment to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeRegistrationAttachments.html
     *
     * @param registrationAttachmentId - Identifier for the registrationAttachmentId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegistrationAttachment(registrationAttachmentId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:registration-attachment/${registrationAttachmentId}`);
    }
    /**
     * Adds a resource of type VerifiedDestinationNumber to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_DescribeVerifiedDestinationNumbers.html
     *
     * @param verifiedDestinationNumberId - Identifier for the verifiedDestinationNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVerifiedDestinationNumber(verifiedDestinationNumberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:verified-destination-number/${verifiedDestinationNumberId}`);
    }
    /**
     * Adds a resource of type Message to the statement
     *
     * https://docs.aws.amazon.com/pinpoint/latest/apireference_smsvoicev2/API_PutMessageFeedback.html
     *
     * @param messageId - Identifier for the messageId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMessage(messageId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sms-voice:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:message/${messageId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateOptOutList()
     * - .toCreatePool()
     * - .toCreateProtectConfiguration()
     * - .toCreateRegistration()
     * - .toCreateRegistrationAttachment()
     * - .toCreateVerifiedDestinationNumber()
     * - .toRequestPhoneNumber()
     * - .toRequestSenderId()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - ConfigurationSet
     * - OptOutList
     * - PhoneNumber
     * - Pool
     * - ProtectConfiguration
     * - SenderId
     * - Registration
     * - RegistrationAttachment
     * - VerifiedDestinationNumber
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConfigurationSet()
     * - .toCreateOptOutList()
     * - .toCreatePool()
     * - .toCreateProtectConfiguration()
     * - .toCreateRegistration()
     * - .toCreateRegistrationAttachment()
     * - .toCreateVerifiedDestinationNumber()
     * - .toRequestPhoneNumber()
     * - .toRequestSenderId()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sms-voice-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsmsandvoicev2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sms-voice';
        this.accessLevelList = {
            Write: [
                'AssociateOriginationIdentity',
                'AssociateProtectConfiguration',
                'CreateConfigurationSet',
                'CreateEventDestination',
                'CreateOptOutList',
                'CreatePool',
                'CreateProtectConfiguration',
                'CreateRegistration',
                'CreateRegistrationAssociation',
                'CreateRegistrationAttachment',
                'CreateRegistrationVersion',
                'CreateVerifiedDestinationNumber',
                'DeleteAccountDefaultProtectConfiguration',
                'DeleteConfigurationSet',
                'DeleteDefaultMessageType',
                'DeleteDefaultSenderId',
                'DeleteEventDestination',
                'DeleteKeyword',
                'DeleteMediaMessageSpendLimitOverride',
                'DeleteOptOutList',
                'DeleteOptedOutNumber',
                'DeletePool',
                'DeleteProtectConfiguration',
                'DeleteProtectConfigurationRuleSetNumberOverride',
                'DeleteRegistration',
                'DeleteRegistrationAttachment',
                'DeleteRegistrationFieldValue',
                'DeleteTextMessageSpendLimitOverride',
                'DeleteVerifiedDestinationNumber',
                'DeleteVoiceMessageSpendLimitOverride',
                'DisassociateOriginationIdentity',
                'DisassociateProtectConfiguration',
                'DiscardRegistrationVersion',
                'PutKeyword',
                'PutMessageFeedback',
                'PutOptedOutNumber',
                'PutProtectConfigurationRuleSetNumberOverride',
                'PutRegistrationFieldValue',
                'ReleasePhoneNumber',
                'ReleaseSenderId',
                'RequestPhoneNumber',
                'RequestSenderId',
                'SendDestinationNumberVerificationCode',
                'SendMediaMessage',
                'SendTextMessage',
                'SendVoiceMessage',
                'SetAccountDefaultProtectConfiguration',
                'SetDefaultMessageFeedbackEnabled',
                'SetDefaultMessageType',
                'SetDefaultSenderId',
                'SetMediaMessageSpendLimitOverride',
                'SetTextMessageSpendLimitOverride',
                'SetVoiceMessageSpendLimitOverride',
                'SubmitRegistrationVersion',
                'UpdateEventDestination',
                'UpdatePhoneNumber',
                'UpdatePool',
                'UpdateProtectConfiguration',
                'UpdateProtectConfigurationCountryRuleSet',
                'UpdateSenderId',
                'VerifyDestinationNumber'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy'
            ],
            Read: [
                'DescribeAccountAttributes',
                'DescribeAccountLimits',
                'DescribeConfigurationSets',
                'DescribeKeywords',
                'DescribeOptOutLists',
                'DescribeOptedOutNumbers',
                'DescribePhoneNumbers',
                'DescribePools',
                'DescribeProtectConfigurations',
                'DescribeRegistrationAttachments',
                'DescribeRegistrationFieldDefinitions',
                'DescribeRegistrationFieldValues',
                'DescribeRegistrationSectionDefinitions',
                'DescribeRegistrationTypeDefinitions',
                'DescribeRegistrationVersions',
                'DescribeRegistrations',
                'DescribeSenderIds',
                'DescribeSpendLimits',
                'DescribeVerifiedDestinationNumbers',
                'GetProtectConfigurationCountryRuleSet',
                'GetResourcePolicy',
                'ListPoolOriginationIdentities',
                'ListProtectConfigurationRuleSetNumberOverrides',
                'ListRegistrationAssociations',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SmsVoiceV2 = SmsVoiceV2;
//# sourceMappingURL=data:application/json;base64,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