"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Entityresolution = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [entityresolution](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsentityresolution.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Entityresolution extends shared_1.PolicyStatement {
    /**
     * Grants permission to give an AWS service or another account permission to use an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_AddPolicyStatement.html
     */
    toAddPolicyStatement() {
        return this.to('AddPolicyStatement');
    }
    /**
     * Grants permission to batch delete unique Id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_BatchDeleteUniqueId.html
     */
    toBatchDeleteUniqueId() {
        return this.to('BatchDeleteUniqueId');
    }
    /**
     * Grants permission to create a idmapping workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateIdMappingWorkflow.html
     */
    toCreateIdMappingWorkflow() {
        return this.to('CreateIdMappingWorkflow');
    }
    /**
     * Grants permission to create a IdNamespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateIdNamespace.html
     */
    toCreateIdNamespace() {
        return this.to('CreateIdNamespace');
    }
    /**
     * Grants permission to create a matching workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateMatchingWorkflow.html
     */
    toCreateMatchingWorkflow() {
        return this.to('CreateMatchingWorkflow');
    }
    /**
     * Grants permission to create a schema mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateSchemaMapping.html
     */
    toCreateSchemaMapping() {
        return this.to('CreateSchemaMapping');
    }
    /**
     * Grants permission to delete a idmapping workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteIdMappingWorkflow.html
     */
    toDeleteIdMappingWorkflow() {
        return this.to('DeleteIdMappingWorkflow');
    }
    /**
     * Grants permission to delete a IdNamespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteIdNamespace.html
     */
    toDeleteIdNamespace() {
        return this.to('DeleteIdNamespace');
    }
    /**
     * Grants permission to delete a matching workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteMatchingWorkflow.html
     */
    toDeleteMatchingWorkflow() {
        return this.to('DeleteMatchingWorkflow');
    }
    /**
     * Grants permission to delete permission given to an AWS service or another account permission to use an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeletePolicyStatement.html
     */
    toDeletePolicyStatement() {
        return this.to('DeletePolicyStatement');
    }
    /**
     * Grants permission to delete a schema mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteSchemaMapping.html
     */
    toDeleteSchemaMapping() {
        return this.to('DeleteSchemaMapping');
    }
    /**
     * Grants permission to get a idmapping job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdMappingJob.html
     */
    toGetIdMappingJob() {
        return this.to('GetIdMappingJob');
    }
    /**
     * Grants permission to get a idmapping workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdMappingWorkflow.html
     */
    toGetIdMappingWorkflow() {
        return this.to('GetIdMappingWorkflow');
    }
    /**
     * Grants permission to get a IdNamespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdNamespace.html
     */
    toGetIdNamespace() {
        return this.to('GetIdNamespace');
    }
    /**
     * Grants permission to get match Id
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchId.html
     */
    toGetMatchId() {
        return this.to('GetMatchId');
    }
    /**
     * Grants permission to get a matching job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchingJob.html
     */
    toGetMatchingJob() {
        return this.to('GetMatchingJob');
    }
    /**
     * Grants permission to get a matching workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchingWorkflow.html
     */
    toGetMatchingWorkflow() {
        return this.to('GetMatchingWorkflow');
    }
    /**
     * Grants permission to get a resource policy for an AWS Entity Resolution resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get provider service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetProviderService.html
     */
    toGetProviderService() {
        return this.to('GetProviderService');
    }
    /**
     * Grants permission to get a schema mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetSchemaMapping.html
     */
    toGetSchemaMapping() {
        return this.to('GetSchemaMapping');
    }
    /**
     * Grants permission to list idmapping jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdMappingJobs.html
     */
    toListIdMappingJobs() {
        return this.to('ListIdMappingJobs');
    }
    /**
     * Grants permission to list idmapping workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdMappingWorkflows.html
     */
    toListIdMappingWorkflows() {
        return this.to('ListIdMappingWorkflows');
    }
    /**
     * Grants permission to list IdNamespaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdNamespaces.html
     */
    toListIdNamespaces() {
        return this.to('ListIdNamespaces');
    }
    /**
     * Grants permission to list matching jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListMatchingJobs.html
     */
    toListMatchingJobs() {
        return this.to('ListMatchingJobs');
    }
    /**
     * Grants permission to list matching workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListMatchingWorkflows.html
     */
    toListMatchingWorkflows() {
        return this.to('ListMatchingWorkflows');
    }
    /**
     * Grants permission to list provider service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListProviderServices.html
     */
    toListProviderServices() {
        return this.to('ListProviderServices');
    }
    /**
     * Grants permission to list schema mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListSchemaMappings.html
     */
    toListSchemaMappings() {
        return this.to('ListSchemaMappings');
    }
    /**
     * Grants permission to List tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to put a resource policy for an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_PutPolicy.html
     */
    toPutPolicy() {
        return this.to('PutPolicy');
    }
    /**
     * Grants permission to start a idmapping job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_StartIdMappingJob.html
     */
    toStartIdMappingJob() {
        return this.to('StartIdMappingJob');
    }
    /**
     * Grants permission to start a matching job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_StartMatchingJob.html
     */
    toStartMatchingJob() {
        return this.to('StartMatchingJob');
    }
    /**
     * Grants permission to adds tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a idmapping workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateIdMappingWorkflow.html
     */
    toUpdateIdMappingWorkflow() {
        return this.to('UpdateIdMappingWorkflow');
    }
    /**
     * Grants permission to update a IdNamespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateIdNamespace.html
     */
    toUpdateIdNamespace() {
        return this.to('UpdateIdNamespace');
    }
    /**
     * Grants permission to update a matching workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateMatchingWorkflow.html
     */
    toUpdateMatchingWorkflow() {
        return this.to('UpdateMatchingWorkflow');
    }
    /**
     * Grants permission to update a schema mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateSchemaMapping.html
     */
    toUpdateSchemaMapping() {
        return this.to('UpdateSchemaMapping');
    }
    /**
     * Grants permission to give an AWS service or another account permission to use IdNamespace within a workflow
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UseIdNamespace.html
     */
    toUseIdNamespace() {
        return this.to('UseIdNamespace');
    }
    /**
     * Grants permission to give an AWS service or another account permission to use workflow within a IdNamespace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UseWorkflow.html
     */
    toUseWorkflow() {
        return this.to('UseWorkflow');
    }
    /**
     * Adds a resource of type MatchingWorkflow to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchingWorkflow(workflowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:matchingworkflow/${workflowName}`);
    }
    /**
     * Adds a resource of type SchemaMapping to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchemaMapping(schemaName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:schemamapping/${schemaName}`);
    }
    /**
     * Adds a resource of type IdMappingWorkflow to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdMappingWorkflow(workflowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:idmappingworkflow/${workflowName}`);
    }
    /**
     * Adds a resource of type ProviderService to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param providerName - Identifier for the providerName.
     * @param providerServiceName - Identifier for the providerServiceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProviderService(providerName, providerServiceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:providerservice/${providerName}/${providerServiceName}`);
    }
    /**
     * Adds a resource of type IdNamespace to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param idNamespaceName - Identifier for the idNamespaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdNamespace(idNamespaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:idnamespace/${idNamespaceName}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the entity resolution service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateIdMappingWorkflow()
     * - .toCreateIdNamespace()
     * - .toCreateMatchingWorkflow()
     * - .toCreateSchemaMapping()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to resource types:
     * - MatchingWorkflow
     * - SchemaMapping
     * - IdMappingWorkflow
     * - ProviderService
     * - IdNamespace
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the entity resolution service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateIdMappingWorkflow()
     * - .toCreateIdNamespace()
     * - .toCreateMatchingWorkflow()
     * - .toCreateSchemaMapping()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [entityresolution](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsentityresolution.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'entityresolution';
        this.accessLevelList = {
            'Permissions management': [
                'AddPolicyStatement',
                'DeletePolicyStatement',
                'PutPolicy',
                'UseIdNamespace',
                'UseWorkflow'
            ],
            Write: [
                'BatchDeleteUniqueId',
                'CreateIdMappingWorkflow',
                'CreateIdNamespace',
                'CreateMatchingWorkflow',
                'CreateSchemaMapping',
                'DeleteIdMappingWorkflow',
                'DeleteIdNamespace',
                'DeleteMatchingWorkflow',
                'DeleteSchemaMapping',
                'StartIdMappingJob',
                'StartMatchingJob',
                'UpdateIdMappingWorkflow',
                'UpdateIdNamespace',
                'UpdateMatchingWorkflow',
                'UpdateSchemaMapping'
            ],
            Read: [
                'GetIdMappingJob',
                'GetIdMappingWorkflow',
                'GetIdNamespace',
                'GetMatchId',
                'GetMatchingJob',
                'GetMatchingWorkflow',
                'GetPolicy',
                'GetProviderService',
                'GetSchemaMapping',
                'ListTagsForResource'
            ],
            List: [
                'ListIdMappingJobs',
                'ListIdMappingWorkflows',
                'ListIdNamespaces',
                'ListMatchingJobs',
                'ListMatchingWorkflows',
                'ListProviderServices',
                'ListSchemaMappings'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Entityresolution = Entityresolution;
//# sourceMappingURL=data:application/json;base64,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