"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fis = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [fis](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfaultinjectionservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fis extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an AWS FIS experiment template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_CreateExperimentTemplate.html
     */
    toCreateExperimentTemplate() {
        return this.to('CreateExperimentTemplate');
    }
    /**
     * Grants permission to create an AWS FIS target account configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_CreateTargetAccountConfiguration.html
     */
    toCreateTargetAccountConfiguration() {
        return this.to('CreateTargetAccountConfiguration');
    }
    /**
     * Grants permission to delete the AWS FIS experiment template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_DeleteExperimentTemplate.html
     */
    toDeleteExperimentTemplate() {
        return this.to('DeleteExperimentTemplate');
    }
    /**
     * Grants permission to delete an AWS FIS target account configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_DeleteTargetAccountConfiguration.html
     */
    toDeleteTargetAccountConfiguration() {
        return this.to('DeleteTargetAccountConfiguration');
    }
    /**
     * Grants permission to retrieve an AWS FIS action
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetAction.html
     */
    toGetAction() {
        return this.to('GetAction');
    }
    /**
     * Grants permission to retrieve an AWS FIS experiment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetExperiment.html
     */
    toGetExperiment() {
        return this.to('GetExperiment');
    }
    /**
     * Grants permission to retrieve an AWS FIS target account configuration for an AWS FIS experiment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetExperimentTargetAccountConfiguration.html
     */
    toGetExperimentTargetAccountConfiguration() {
        return this.to('GetExperimentTargetAccountConfiguration');
    }
    /**
     * Grants permission to retrieve an AWS FIS Experiment Template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetExperimentTemplate.html
     */
    toGetExperimentTemplate() {
        return this.to('GetExperimentTemplate');
    }
    /**
     * Grants permission to get information about the safety lever
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetSafetyLever.html
     */
    toGetSafetyLever() {
        return this.to('GetSafetyLever');
    }
    /**
     * Grants permission to retrieve an AWS FIS target account configuration for an AWS FIS experiment template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetTargetAccountConfiguration.html
     */
    toGetTargetAccountConfiguration() {
        return this.to('GetTargetAccountConfiguration');
    }
    /**
     * Grants permission to get information about the specified resource type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_GetTargetResourceType.html
     */
    toGetTargetResourceType() {
        return this.to('GetTargetResourceType');
    }
    /**
     * Grants permission to inject an API internal error on the provided AWS service from an FIS Experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     * - .ifOperations()
     * - .ifPercentage()
     * - .ifTargets()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#fis-actions-reference-fis
     */
    toInjectApiInternalError() {
        return this.to('InjectApiInternalError');
    }
    /**
     * Grants permission to inject an API throttle error on the provided AWS service from an FIS Experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     * - .ifOperations()
     * - .ifPercentage()
     * - .ifTargets()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#fis-actions-reference-fis
     */
    toInjectApiThrottleError() {
        return this.to('InjectApiThrottleError');
    }
    /**
     * Grants permission to inject an API unavailable error on the provided AWS service from an FIS Experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifService()
     * - .ifOperations()
     * - .ifPercentage()
     * - .ifTargets()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#fis-actions-reference-fis
     */
    toInjectApiUnavailableError() {
        return this.to('InjectApiUnavailableError');
    }
    /**
     * Grants permission to list all available AWS FIS actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListActions.html
     */
    toListActions() {
        return this.to('ListActions');
    }
    /**
     * Grants permission to list resolved targets for AWS FIS experiments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListExperimentResolvedTargets.html
     */
    toListExperimentResolvedTargets() {
        return this.to('ListExperimentResolvedTargets');
    }
    /**
     * Grants permission to list target account configurations for AWS FIS experiments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListExperimentTargetAccountConfigurations.html
     */
    toListExperimentTargetAccountConfigurations() {
        return this.to('ListExperimentTargetAccountConfigurations');
    }
    /**
     * Grants permission to list all available AWS FIS experiment templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListExperimentTemplates.html
     */
    toListExperimentTemplates() {
        return this.to('ListExperimentTemplates');
    }
    /**
     * Grants permission to list all available AWS FIS experiments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListExperiments.html
     */
    toListExperiments() {
        return this.to('ListExperiments');
    }
    /**
     * Grants permission to list the tags for an AWS FIS resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list target account configurations for AWS FIS experiment templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListTargetAccountConfigurations.html
     */
    toListTargetAccountConfigurations() {
        return this.to('ListTargetAccountConfigurations');
    }
    /**
     * Grants permission to list the resource types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_ListTargetResourceTypes.html
     */
    toListTargetResourceTypes() {
        return this.to('ListTargetResourceTypes');
    }
    /**
     * Grants permission to run an AWS FIS experiment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_StartExperiment.html
     */
    toStartExperiment() {
        return this.to('StartExperiment');
    }
    /**
     * Grants permission to stop an AWS FIS experiment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_StopExperiment.html
     */
    toStopExperiment() {
        return this.to('StopExperiment');
    }
    /**
     * Grants permission to tag AWS FIS resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag AWS FIS resources
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the specified AWS FIS experiment template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_UpdateExperimentTemplate.html
     */
    toUpdateExperimentTemplate() {
        return this.to('UpdateExperimentTemplate');
    }
    /**
     * Grants permission to update the state of the safety lever
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_UpdateSafetyLeverState.html
     */
    toUpdateSafetyLeverState() {
        return this.to('UpdateSafetyLeverState');
    }
    /**
     * Grants permission to update an AWS FIS target account configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fis/latest/APIReference/API_UpdateTargetAccountConfiguration.html
     */
    toUpdateTargetAccountConfiguration() {
        return this.to('UpdateTargetAccountConfiguration');
    }
    /**
     * Adds a resource of type action to the statement
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/actions.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAction(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:action/${id}`);
    }
    /**
     * Adds a resource of type experiment to the statement
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/experiments.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExperiment(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:experiment/${id}`);
    }
    /**
     * Adds a resource of type experiment-template to the statement
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/working-with-templates.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExperimentTemplate(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:experiment-template/${id}`);
    }
    /**
     * Adds a resource of type safety-lever to the statement
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/safety-lever.html
     *
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSafetyLever(id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fis:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:safety-lever/${id}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateExperimentTemplate()
     * - .toStartExperiment()
     * - .toTagResource()
     * - .toUpdateExperimentTemplate()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetAction()
     * - .toGetExperiment()
     * - .toGetExperimentTemplate()
     *
     * Applies to resource types:
     * - action
     * - experiment
     * - experiment-template
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateExperimentTemplate()
     * - .toStartExperiment()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateExperimentTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of operations on the AWS service that is being affected by the AWS FIS action
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toInjectApiInternalError()
     * - .toInjectApiThrottleError()
     * - .toInjectApiUnavailableError()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOperations(value, operator) {
        return this.if(`Operations`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the percentage of calls being affected by the AWS FIS action
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toInjectApiInternalError()
     * - .toInjectApiThrottleError()
     * - .toInjectApiUnavailableError()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifPercentage(value, operator) {
        return this.if(`Percentage`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the AWS service that is being affected by the AWS FIS action
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toInjectApiInternalError()
     * - .toInjectApiThrottleError()
     * - .toInjectApiUnavailableError()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifService(value, operator) {
        return this.if(`Service`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of resource ARNs being targeted by the AWS FIS action
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toInjectApiInternalError()
     * - .toInjectApiThrottleError()
     * - .toInjectApiUnavailableError()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTargets(value, operator) {
        return this.if(`Targets`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [fis](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsfaultinjectionservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'fis';
        this.accessLevelList = {
            Write: [
                'CreateExperimentTemplate',
                'CreateTargetAccountConfiguration',
                'DeleteExperimentTemplate',
                'DeleteTargetAccountConfiguration',
                'InjectApiInternalError',
                'InjectApiThrottleError',
                'InjectApiUnavailableError',
                'StartExperiment',
                'StopExperiment',
                'UpdateExperimentTemplate',
                'UpdateSafetyLeverState',
                'UpdateTargetAccountConfiguration'
            ],
            Read: [
                'GetAction',
                'GetExperiment',
                'GetExperimentTargetAccountConfiguration',
                'GetExperimentTemplate',
                'GetSafetyLever',
                'GetTargetAccountConfiguration',
                'GetTargetResourceType',
                'ListTagsForResource'
            ],
            List: [
                'ListActions',
                'ListExperimentResolvedTargets',
                'ListExperimentTargetAccountConfigurations',
                'ListExperimentTemplates',
                'ListExperiments',
                'ListTargetAccountConfigurations',
                'ListTargetResourceTypes'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Fis = Fis;
//# sourceMappingURL=data:application/json;base64,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